package com.jamonapi;


import java.util.*;
import java.text.*;
import com.jamonapi.utils.*;

/** MonitorComposite can contain other MonitorComposite and TimingMonitors (at the leaf level) **/


public class MonitorComposite implements MinimalMonitor,CompositeNode,MonitorReportInterface {
    /** Returns the accrued value of all Monitors underneath this MonitorComposite.  Note this method will recursively call 
     *  getAccrued() for any nested MonitorComposites.  Note this method uses the Gang of 4's Command pattern.
     **/
    public long getAccrued() {
        // sums up the total of all TotalMonitor() objects within the TimingMonitor.  Note this means that
        // the TimingMonitor() must be constructed like new TimingMonitor(new TotalMonitor(...))
        class AccruedCommand implements Command {
            long accrued;
            public void execute( Object monitor) throws Exception {
                if (monitor instanceof TimingMonitor)
                    accrued += (((TimingMonitor)monitor).childMonitor).getAccrued();
                else
                    accrued += ((MonitorComposite)monitor).getAccrued();
            }
        }
        
        AccruedCommand command = new AccruedCommand();
        iterate(command);
        
        return command.accrued;
    }
    
    /** Display the accrued value as a string **/
    public String toString() {
        return getAccruedString();
    }
    
    /** Call reset() on all Monitors that this MonitorComposite instance contains **/
    public void reset() {
        class ResetCommand implements Command {
            public void execute( Object monitor) throws Exception {
                ((MinimalMonitor)monitor).reset();
            }
        }
        
        iterate(new ResetCommand());
    }
    
    
    /** Iterate through the MonitorComposite executing the passed in Command object.  Note This is an example of the Gang of 4's 
     *  Command and Iterator patterns
     **/
    void iterate(Command command) {
        try {
            CommandIterator.iterate(monitorList.values().iterator(), command);
        } catch (Exception e) {
            throw AppBaseException.getRuntimeException(e);
        }
    }
    
    
    
    void iterateMapEntries(Command command) {
        try {
            CommandIterator.iterate(monitorList, command);
        } catch (Exception e) {
            throw AppBaseException.getRuntimeException(e);
        }
    }
    
    /** Call increase(long) on all Monitors that this MonitorComposite instance contains **/
    public void increase(final long increaseValue) {
        
        class IncreaseCommand implements Command {
            public void execute( Object monitor) throws Exception {
                ((MinimalMonitor)monitor).increase(increaseValue);
            }
        }
        
        iterate(new IncreaseCommand());
    }
    
    
    // the next method really shouldn't need to be called.  i put it in to honor the interface.  it should probably
    // be refactored out
    public void getData(ArrayList rowData)    {
    }
    
    
    // shouldn't be called.  refactor this out.
    public void getHeader(ArrayList header)    {
    }
    
    /** Return the contents of this MonitorComposite as an HTML table.  This method is called to display the data in a MonitorComposite
     *  in a servlet or JSP.  See www.jamonapi.com for a sample of what the report shoud look like.
     *
     * Sample Call:
     *  String html=monitorComposite.getReport();
     **/
    public String getReport() throws Exception    {
        MonitorConverter mc=new MonitorConverter(this);
        return mc.getReport();
    }
    
    /** Return the contents of this MonitorComposite as an HTML table sorted by the specified column number in ascending or descending order.  
     *  This method is called to display the data in a MonitorComposite in a servlet or JSP.  See www.jamonapi.com for a sample of what 
     *  the report shoud look like.
     *
     * Sample Call:
     *  String html=monitorComposite.getReport(2, "asc");  // return the html report sorted by column 2 in ascending order
     **/
    public String getReport(int sortCol, String sortOrder) throws Exception {
        MonitorConverter mc=new MonitorConverter(this);
        return mc.getReport(sortCol, sortOrder);
        
    }

    
    /** Return the contents of this MonitorComposite as a 2 dimensional array of Strings.  This is the data minus the formatting of 
     *  what is returned by the getReport() method.  
     **/
    public String[][] getData()    {
        MonitorConverter mc=new MonitorConverter(this);
        return mc.getData();
        
    }
    

     /** Return the contents of this MonitorComposite as a 2 dimensional array of Strings.  The passed "label" will be prepended to the 
      *  first column in the array.  
      * Sample Call:
      *  String[][] arr=monitorComposite.getData("myLabel");
     **/
    public String[][] getData(String label)      {
        MonitorConverter mc=new MonitorConverter(this);
        return mc.getData(label);
        
    }
    
    
    
    private static String displayDelimiter=".";
    
    /** Delimiter to be used when displaying the monitor label returned by the getReport() and getData() methods.  The default is a period.
     *  A better approach would have been to simply display the delimeter that was used on the MonitorComposites creation
     **/
    public static void setDisplayDelimiter(String localDisplayDelimiter) {
        displayDelimiter=localDisplayDelimiter;
    }
    
    /** This is CompositeNode interface method that returns the child MonitorComposite identified by the given label or it creates a 
      * new CompositeMonitor.  This method is used by the factory classes to create MonitorComposites.
     *
     * Sample Call:
     *   monitorComposite.getCompositeNode("myApp.jsp")
     **/
    public CompositeNode getCompositeNode(String childNodeName)    {
        if (compositeNodeExists(childNodeName))
            return getExistingCompositeNode(childNodeName);
        else
            return new MonitorComposite();  // Use MonitorFactory or is it not needed anymore?
    }
    
    
    
    /** This is CompositeNode interface method that returns a leaf node identified by the given label or it creates a 
      * new leaf node by calling the Leaf.  This method is used by the factory classes to create TimingMonitors.
     *
     * Sample Call:
     *   monitorComposite.getCompositeNode("myApp.jsp");
     **/
    public LeafNode getLeafNode(String childNodeName, String childNodeType)    {
        if (leafNodeExists(childNodeName)) {
            TimingMonitor mon = (TimingMonitor)  getExistingLeafNode(childNodeName);
            return mon;
        }
        else  
            return  leafFactory.createInstance(childNodeType);
    }
    
    
    
    /** Add a CompositeNode to the data structure if it doesn't exist. **/
    public void addCompositeNode(String childNodeName, CompositeNode addNode) {
        
        boolean nodeExists=compositeNodeExists(childNodeName);
        
        if (!nodeExists)
            monitorList.put(getCompositeNodeKey(childNodeName), addNode);
    }
    

    
    /** Does the passed in leaf node exist.  Returns true if there has been a LeafNode created with this label.
     *
     * Sample Call:
     *  monitorComposite.leafNodeExists("myApp.jsp.homePage");
     **/
    public boolean leafNodeExists(String childNodeName) {
        return monitorList.containsKey(getLeafNodeKey(childNodeName));
        
    }
    
    /** Does the passed in CompositeNode exist.  Returns true if there has been a CompositeNode created with this label.
     *
     * Sample Call:
     *  monitorComposite.compositeNodeExists("myApp.jsp");
     **/
    public boolean compositeNodeExists(String childNodeName) {
        return monitorList.containsKey(getCompositeNodeKey(childNodeName));
        
    }
    
    
    /** Returns the CompositeNode if it exists. 
     *
     * Sample Call:
     *  monitorComposite.compositeNodeExists("myApp.jsp");
     **/
    public CompositeNode getExistingCompositeNode(String childNodeName) {
        return (CompositeNode) monitorList.get(getCompositeNodeKey(childNodeName));
    }
    
    
    /** Returns the LeafNode if it exists. 
     *
     * Sample Call:
     *  monitorComposite.compositeNodeExists("myApp.jsp");
     **/
    public LeafNode getExistingLeafNode(String childNodeName) {
        return (LeafNode) monitorList.get(getLeafNodeKey(childNodeName));
    }
    
    
    /** Add a LeafNode to the data structure if it doesn't exist. **/
    public void addLeafNode(String childNodeName, LeafNode addNode) {
        boolean nodeExists=leafNodeExists(childNodeName);
        
        if (!nodeExists)
            monitorList.put(getLeafNodeKey(childNodeName), addNode);
    }
    
    
    
    /** Return this MonitorComposite as a CompositeNode **/
    public CompositeNode getRootNode() {
        return this;
    }
    
    /** This reverses getCompositeNodeKey() and getLeafNodeKey() **/
    protected String getLabelFromKey(String key) { // if the location that we are after is pages.homepage then in the hashmap the values look like
        // pagesChomepageL  where pagesC means it is a composite and homepageL means it is a leaf node.
        // The logic below gets rid of the C and L and puts a period at the end of a composite.  i.e.
        // pagesChomepageL becomes pages.homepage
        
        // if key is "pagesC" return "pages.".  if key is homepageL return "homepage"
        int lastCharPos=key.length()-1;
        String lastChar="";
        
        if ('C'==key.charAt(lastCharPos))
            lastChar=displayDelimiter;
        
        return key.substring(0,lastCharPos)+lastChar;
    }
    
    
    /**
     * getCompositeNodeKey(...) and getLeafNodeKey(...) are used to ensure that composite nodes are not replaced by leaf nodes 
     * and vice versa.  For example the following 2 entries could both be simultaneously valid even though their labels are 
     * identical.
     *
     * Leaf Node:  pages.homePage
     * Composite Node:  pages.homePage
     *
     * This is because leafs and composites have different keys behind the scenes.  Using the example above the
     * keys may look something like:
     *
     * Leaf Node Key:  pagesC.homePageL
     * Composite Node:  pagesC.homePageC
     *
     */
    public String getCompositeNodeKey(String nodeName) {
        final String COMPOSITE_NODE_KEY_APPENDER="C";
        return nodeName+COMPOSITE_NODE_KEY_APPENDER;
    }
    
    
    /**
     * getCompositeNodeKey(...) and getLeafNodeKey(...) are used to ensure that composite nodes are not replaced by leaf nodes 
     * and vice versa.  For example the following 2 entries could both be simultaneously valid even though their labels are 
     * identical.
     *
     * Leaf Node:  pages.homePage
     * Composite Node:  pages.homePage
     *
     * This is because leafs and composites have different keys behind the scenes.  Using the example above the
     * keys may look something like:
     *
     * Leaf Node Key:  pagesC.homePageL
     * Composite Node:  pagesC.homePageC
     *
     */
    public String getLeafNodeKey(String nodeName) {
        final String LEAF_NODE_KEY_APPENDER="L";
        return nodeName+LEAF_NODE_KEY_APPENDER;
    }
    
    /** Return the accrued value in String format **/
    public  String getAccruedString() {
        DecimalFormat numberFormat = (DecimalFormat) NumberFormat.getNumberInstance();
        numberFormat.applyPattern("#,###");
        return numberFormat.format(getAccrued());
    }
    
    
    /** Return an array that contains the Header to be used in reports **/
    protected  static String[] getHeader() {
        return leafFactory.getHeader();
    }
    private Map monitorList = new TreeMap();
    private static MonitorLeafFactory leafFactory = new MonitorLeafFactory();
}

