package com.jamonapi;

import java.util.*;


/** Tracks statistics for Monitors that are currently running.  The statistics tracked are how often this Monitor
 * is called (hits), how many outstanding invocations of start() have been called without a stop() for this instance (active), what
 * is the average number of active for this instance, and what was the maximum number of active for this instance.
 **/

public class ActiveStatsMonitor extends AccumulateMonitor {
    // ANY VARIABLES SET HERE MUST ALSO BE SET IN resetThis()!!!!
    private int max=Integer.MIN_VALUE;
    private int active;
    private int hits;
    private int total;  // avg is calculated when needed by avg=total/hits
    
    
    public ActiveStatsMonitor()     {
        super();
    }
    
    /** Constructor that uses the Gang Of 4's decorator pattern. **/
    public ActiveStatsMonitor(AccumulateMonitorInterface childMonitor)     {
        super(childMonitor);
    }
    

    /** Adjust the statistics being tracked whenever start() is called.  For example whenever start() is called on this instance
      * the active variable is increased by 1.
     **/
    synchronized protected void startThis()     {
        active++;
        hits++;
        total+=active;
        
        if (active>max)
            max=active;
    }
    
    /** Adjust the statistics being tracked whenever stop() is called.  For example whenever stop() is called on this instance
      * the active variable is decreased by 1.
     **/
   
    synchronized protected void stopThis()     {
        if (active>0)
            active--;
    }
    
    
    
    protected void increaseThis(long increase)     {
    }
    

    /** Reset the variables for this instance to their default values **/
    synchronized protected void resetThis()     {
        max=Integer.MIN_VALUE;
        hits=active=total=0;
    }
    
    /** Convert this Monitor to its String value for display purposes **/
    synchronized protected String toStringThis()     {
        return  getDisplayString(ACTIVE, convertToString(active), NONE) +
        getDisplayString(AVGACTIVE, convertToString(avg()), NONE) +
        getDisplayString(MAXACTIVE, convertToString(max), NONE);
    }
    
    
    /** Add this instances data to the ArrayList.  This method is used to convert Monitor data to a tabular format **/
    synchronized protected void getDataThis(ArrayList rowData)     {
        rowData.add(convertToString(active));
        rowData.add(convertToString(avg()));
        rowData.add(convertToString(max));
    }
    

    /** Add this instances header to the ArrayList.  This method is used to convert Monitor header data to a tabular format. 
     *  The header will usually be used to display the Monitor in HTML format
     **/
    protected void getHeaderThis(ArrayList header)     {
        header.add(ACTIVE);
        header.add(AVGACTIVE);
        header.add(MAXACTIVE);
    }
    
    
    /** Return the average number of Active monitors of this instance **/
    final float avg()     {
        if (hits==0)
            return 0;
        else
            return (float) total/hits;
    }
    
    /** Return the number of Active Monitors of this instance **/
    synchronized public long getAccrued()     {
        return active;
    }
    
    
    /** Method that contains test data for this class **/
    public static void main(String[] args) throws Exception     {
        AccumulateMonitor mon=new ActiveStatsMonitor();
        System.out.println("should not have 0 divide error="+mon);
        mon.start();  //active=1
        mon.start();  //2
        mon.start();  //3
        mon.stop();   //2
        mon.stop();   //1
        mon.start();  //2
        
        System.out.println("should be active 2, avg 2, max active 3="+mon);
        
    }
}

