package com.jamonapi;

import java.util.*;
import java.text.*;
/**
 *  AccumulateMonitors represent Monitors that increase in value.  AccumulateMonitors use the Gang of 4's decorator pattern to pass
 *  method invocations to the next Monitor in the chain.  Note many of the public methods such as start(), stop() and reset() call the
 *  same method of next Monitor in the decorator chain.  Any classes that inherit from AccumulateMonitor implement the protected methods 
 *  startThis(), stopThis(), and resetThis() and leave the public methods as is.  In general the various ...This() (startThis(), stopThis(), 
 *  resetThis()) methods perform the action on that instance and the methods without ...This() (start(), stop(), reset()) take care of 
 *  passing the command down the decorator chain.
 **/

public class AccumulateMonitor implements AccumulateMonitorInterface {
    // Constants used in the various AccumulateMonitor classes
    static protected final String TOTAL="Total";
    static protected final String MIN="Min";
    static protected final String MAX="Max";
    static protected final String HITS="Hits";
    static protected final String AVG="Avg";
    static protected final String STANDARD_DEVIATION="Std Dev";
    static protected final String ACTIVE="Active";
    static protected final String AVGACTIVE="Avg Active";
    static protected final String MAXACTIVE="Max Active";
    
    protected long accrued;  // holds the value of what we are monitoring.
    private   AccumulateMonitorInterface childMonitor;  // pointer to next Monitor in the decorator chain.
    private   boolean primary=false;
    
    static protected final String MILLISECONDS="ms.";
    static protected final String NONE="";
    
    /** Get the value of the monitor. What this value means depends on what the monitor does.  In the case of a Timing monitor it would be
     * the elapsed time since the monitor was started.
     *
     * Sample call:
     *  long accrued=monitor.getAccrued();
     *
     **/
    
    synchronized public long getAccrued()     {
        return accrued;
    }
    
    /** Default constructor.
     *
     * Note Monitors use the Gang of 4 decorator pattern where each monitor points to and calls the next monitor in the chain.  With
     * the default constructor there is no need for a next Monitors in the chain, but to keep the logic identical to when there is 
     * a next Monitor in the chain a NullMonitor (that does nothing) is used as the next Monitor to call in the decorator pattern.
     * Martin Fowler's Refactoring book discusses NullMonitor's.
     */
    public AccumulateMonitor()     {
        this(NullAccumulateMonitor.createInstance());
    }

    /** Monitors use the Gang of 4 decorator pattern where each monitor points to and calls the next monitor in the chain.  
     * This constructor as its argument takes the next AccumulateMonitor in the chain.  
     *
     * Sample Call:
     *  AccumulateMonitor mon=new AccumulateMonitor(new AccumulateMonitor(new AccumulateMonitor()));
     */

    public AccumulateMonitor(AccumulateMonitorInterface childMonitor)     {
        this.childMonitor=childMonitor;
    }
    
    
    /** Erase/wipe out any accrued statistics for this monitor **/
    synchronized protected void resetThis()     {
        accrued=0;
    }
    
    /** Increase the monitors accrued value by the ammount specified in the parameter, and call increase on all monitors in the 
     *  decorator chain.
     *
     * Sample Call:
     *  monitor.increase(100);
     **/
    public void increase(long increaseValue)     {
        increaseThis(increaseValue);
        childMonitor.increase(increaseValue);
    }
    
    /** Increase the monitors accrued value by 1 unit.
     *
     *  Sample Call:
     *   monitor.increase();
     **/
    public void increase()     {
        increase(1);
    }

    /** Erase/wipe out any accrued statistics for this monitor, and call reset on all monitors in the decorator chain 
     *  
     * Sample Call:
     *  monitor.reset();
     *     
     **/
    public  void reset()     {
        resetThis();
        childMonitor.reset();
    }
    
    /** Display this Monitor as well as all Monitor's in the decorator chain as Strings **/
    public String toString()     {
        return toStringThis()+toStringChild();
    }

    /** Dispay the units appropriate for this monitor **/
    public String getUnits()     {
        return NONE;
    }
    
    /** Display this Monitors type **/
    public String getType()     {
        return NONE;
    }
    
    /** Populate the ArrayList with data from this Monitor as well as all other Monitors in the decorator chain **/
    public void getData(ArrayList rowData)    {
        getDataThis(rowData);
        childMonitor.getData(rowData);
    }
    
    /** Start gathering statistics for this Monitor.  The Monitor will run until the stop() method is called.  start() will also 
     *  call the start() method for all other Monitors in the decorator chain.
     **/
    public void start()     {
        startThis();
        childMonitor.start();
    }
    
    /** Contains any logic that this Monitor must perform when it is started **/
    protected void startThis()    {
    }
    
    /** Stop gathering statistics for the Monitor.   stop() is called after a Monitor has been started with the start() method.
     *  stop() will also call the stop() method for all Monitors in the decorator chain.
     **/
    public  void stop()     {
        childMonitor.stop();
        stopThis();
    }
    
    /** Contains any logic that this Monitor must perform when it is stopped **/
    protected void stopThis()     {
    }
    
    /** Return the accrued value in String format **/
    public  String getAccruedString() {
        return convertToString(getAccrued());
    }
    
    /** Add this Monitor's accrued value in string format to an ArrayList.  **/
    protected void getDataThis(ArrayList rowData)    {
        rowData.add(getAccruedString());
    }
    
    /** Display this Monitor in String format **/
    synchronized protected String toStringThis()     {
        return getDisplayString(getType(),getAccruedString(),getUnits());
        
    }
    
    /** Display Monitor's in the Decorator chain in String format **/
    protected String toStringChild()     {
        return childMonitor.toString();
    }
    
    /** Convert a float value to a String
     *
     *  Sample Call:
     *   String number=AccumulateMonitor.convertToString(1234.5);  // returns 1,234.5
     **/
    protected static String convertToString(double value) {
        DecimalFormat numberFormat = (DecimalFormat) NumberFormat.getNumberInstance();
        numberFormat.applyPattern("#,###.#");
        return numberFormat.format(value);
        
    }
    
    
    protected String getDisplayString(String type, String value, String units)     {
        return type+"="+value+" "+units+" ";
    }
    
    /** Add the value passed to this method to the Monitor's accrued value **/
    synchronized protected void increaseThis(long increase)     {
        accrued += increase;
        
    }
    
    /** Add the display header that is associated with this Monitor to the header (as an ArrayList entry).  Also give all Monitors in the 
     *  decorator chain the opportunity to add their header entries.
     **/
    public void getHeader(ArrayList header) {
        getHeaderThis(header);
        childMonitor.getHeader(header);
        
    }
    
    protected void getHeaderThis(ArrayList header)     {
        header.add(getType()+" "+getUnits());
    }
    
    /** Indicates whether or not this Monitor is primary.  See www.jamonapi.com for an explanation of primary Monitors. **/
    public boolean isPrimary() {
        return primary;
    }
    
    /** Specify whether or not this Monitor is primary.  See www.jamonapi.com for an explanation of primary Monitors. Call setPrimary()
      * for all Monitors in the decorator chain
     **/
    public void setPrimary(boolean primary) {
        this.primary=primary;
        childMonitor.setPrimary(primary);
    }
    
    /** Method that calls test code for this class. **/
    public static void main(String[] args) throws Exception     {
        // Note for any of the AccumulateMonitors don't need to be started first. i.e. when they
        // are created you may call increase() without first having called start()
        AccumulateMonitor m=new AccumulateMonitor();
        System.out.println("0="+m);
        
        m.increase(25);
        System.out.println("25="+m);
        m.increase(50);
        System.out.println("75="+m);
        
        m.reset();
        System.out.println("0="+m);
        m.increase(1000);
        System.out.println("1000="+m);
        
        
    }
}

