package com.intercom.twig;

import android.support.annotation.IntDef;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.util.Log;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

import static android.util.Log.ASSERT;
import static android.util.Log.DEBUG;
import static android.util.Log.ERROR;
import static android.util.Log.INFO;
import static android.util.Log.VERBOSE;
import static android.util.Log.WARN;

public class Twig {

    private static final int MAX_LOG_LENGTH = 4000;

    @NonNull private final String tag;
    @LogLevel private int logLevel;
    private final boolean internalLoggingEnabled;

    public Twig(@LogLevel int logLevel, @Nullable String tag, boolean internalLoggingEnabled) {
        this.logLevel = logLevel;
        this.tag = tag != null ? tag : "Twig";
        this.internalLoggingEnabled = internalLoggingEnabled;
    }

    //region Logging methods
    public void v(String message, Object... args) {
        prepareLog(VERBOSE, null, message, args);
    }

    /* Log a verbose exception and a message with optional format args. */
    public void v(Throwable t, String message, Object... args) {
        prepareLog(VERBOSE, t, message, args);
    }

    /* Log a verbose exception. */
    public void v(Throwable t) {
        prepareLog(VERBOSE, t, null);
    }

    /* Log a debug message with optional format args. */
    public void d(String message, Object... args) {
        prepareLog(DEBUG, null, message, args);
    }

    /* Log a debug exception and a message with optional format args. */
    public void d(Throwable t, String message, Object... args) {
        prepareLog(DEBUG, t, message, args);
    }

    /* Log a debug exception. */
    public void d(Throwable t) {
        prepareLog(DEBUG, t, null);
    }

    /* Log an info message with optional format args. */
    public void i(String message, Object... args) {
        prepareLog(INFO, null, message, args);
    }

    /* Log an info exception and a message with optional format args. */
    public void i(Throwable t, String message, Object... args) {
        prepareLog(INFO, t, message, args);
    }

    /* Log an info exception. */
    public void i(Throwable t) {
        prepareLog(INFO, t, null);
    }

    /* Log a warning message with optional format args. */
    public void w(String message, Object... args) {
        prepareLog(WARN, null, message, args);
    }

    /* Log a warning exception and a message with optional format args. */
    public void w(Throwable t, String message, Object... args) {
        prepareLog(WARN, t, message, args);
    }

    /* Log a warning exception. */
    public void w(Throwable t) {
        prepareLog(WARN, t, null);
    }

    /* Log an error message with optional format args. */
    public void e(String message, Object... args) {
        prepareLog(ERROR, null, message, args);
    }

    /* Log an error exception and a message with optional format args. */
    public void e(Throwable t, String message, Object... args) {
        prepareLog(ERROR, t, message, args);
    }

    /* Log an error exception. */
    public void e(Throwable t) {
        prepareLog(ERROR, t, null);
    }

    /* Log an assert message with optional format args. */
    public void wtf(String message, Object... args) {
        prepareLog(ASSERT, null, message, args);
    }

    /* Log an assert exception and a message with optional format args. */
    public void wtf(Throwable t, String message, Object... args) {
        prepareLog(ASSERT, t, message, args);
    }

    /* Log an assert exception. */
    public void wtf(Throwable t) {
        prepareLog(ASSERT, t, null);
    }
    //endregion

    //region Internal logging
    /* Log an internal message. Uses Twig's tag */
    public void internal(String text) {
        internal(tag, text);
    }

    /* Log an internal message with tag. */
    public void internal(String tag, String text) {
        if (internalLoggingEnabled) {
            Log.d(tag, "INTERNAL: " + text);
        }
    }
    //endregion

    //region LogLevel
    public static final int DISABLED = ASSERT + 1;

    @Retention(RetentionPolicy.SOURCE)
    @IntDef({VERBOSE, DEBUG, INFO, WARN, ERROR, ASSERT, DISABLED})
    public @interface LogLevel {}
    //endregion

    /* Update the log level. */
    public void setLogLevel(@LogLevel int logLevel) {
        this.logLevel = logLevel;
    }

    /* Get the current level. Used in testing.*/
    @VisibleForTesting int getLogLevel() {
        return logLevel;
    }

    /* Override this for custom tagging */
    public String getTag() {
        return tag;
    }

    //region Log printing
    private void prepareLog(@LogLevel int priority, @Nullable Throwable t,
                            String message, Object... args) {

        if (priority < logLevel) {
            return;
        }

        String tag = getTag();

        if (message != null && message.length() == 0) {
            message = null;
        }
        if (message == null) {
            if (t == null) {
                return; // Swallow message if it's null and there's no throwable.
            }
            message = getStackTraceString(t);
        } else {
            if (args.length > 0) {
                message = String.format(message, args);
            }
            if (t != null) {
                message += "\n" + getStackTraceString(t);
            }
        }

        log(priority, tag, message);
    }

    @VisibleForTesting void log(@LogLevel int priority, String message, Object... args) {
        prepareLog(priority, null, message, args);
    }

    private void log(@LogLevel int priority, String tag, String message) {

        if (message.length() < MAX_LOG_LENGTH) {
            printLog(priority, tag, message);
            return;
        }

        // Split by line, then ensure each line can fit into Log's maximum length.
        for (int i = 0, length = message.length(); i < length; i++) {
            int newline = message.indexOf('\n', i);
            newline = newline != -1 ? newline : length;
            do {
                int end = Math.min(newline, i + MAX_LOG_LENGTH);
                String part = message.substring(i, end);
                printLog(priority, tag, part);
                i = end;
            } while (i < newline);
        }
    }

    private void printLog(@LogLevel int priority, String tag, String message) {
        if (priority == ASSERT) {
            Log.wtf(tag, message);
        } else {
            Log.println(priority, tag, message);
        }
    }

    private String getStackTraceString(Throwable t) {
        // Don't replace this with Log.getStackTraceString() - it hides
        // UnknownHostException, which is not what we want.
        StringWriter sw = new StringWriter(256);
        PrintWriter pw = new PrintWriter(sw, false);
        t.printStackTrace(pw);
        pw.flush();
        return sw.toString();
    }
    //endregion
}
