package com.instabug.featuresrequest.ui.featuredetails;

import android.content.Context;

import androidx.annotation.Nullable;

import com.instabug.featuresrequest.cache.FeatureRequestsDBHelper;
import com.instabug.featuresrequest.eventbus.FeatureRequestsEventBus;
import com.instabug.featuresrequest.models.FeatureRequest;
import com.instabug.featuresrequest.models.TimelineResponse;
import com.instabug.featuresrequest.network.service.FeaturesRequestVoteJob;
import com.instabug.featuresrequest.network.timelinerepository.FeatureRequestsRepository;
import com.instabug.featuresrequest.network.timelinerepository.RepositoryObserver;
import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.util.threading.PoolProvider;

import org.json.JSONException;

/**
 * Created by mohamedzakaria on 9/20/17.
 */

public class FeaturesRequestsDetailsPresenter extends BasePresenter<FeaturesRequestsDetailsContract.View>
        implements FeaturesRequestsDetailsContract.Presenter, RepositoryObserver<TimelineResponse> {

    @Nullable
    private FeaturesRequestsDetailsContract.View viewInstance;
    @Nullable
    private FeatureRequestsRepository repository;

    public FeaturesRequestsDetailsPresenter(FeaturesRequestsDetailsContract.View view) {
        super(view);

        if (this.view != null) {
            viewInstance = this.view.get();

            if (view.getViewContext() != null && view.getViewContext().getContext() != null) {
                repository = FeatureRequestsRepository.getInstance();
            }
        }
    }

    @Override
    public void onBackBtnClicked() {
        if (viewInstance != null) {
            viewInstance.handleBackBtnClicked();
        }
    }

    @Override
    public void onVoteButtonClick(FeatureRequest featureRequest) {
        if (featureRequest.isLiked()) {
            featureRequest.setLiked(false);
            featureRequest.setLikesCount(featureRequest.getLikesCount() - 1);
            unVote(featureRequest);
        } else {
            featureRequest.setLiked(true);
            featureRequest.setLikesCount(featureRequest.getLikesCount() + 1);
            vote(featureRequest);
        }
        if (viewInstance != null) {
            viewInstance.updateVote(featureRequest);
        }
    }

    private void vote(FeatureRequest featureRequest) {
        featureRequest.setUserVoteStatus(FeatureRequest.UserVoteStatus.USER_VOTED_UP);
        try {
            FeatureRequestsDBHelper.insert(featureRequest);
        } catch (JSONException e) {
        }
        submitVote();
        FeatureRequestsEventBus.getInstance().post(featureRequest);
    }

    private void submitVote() {
        if (viewInstance == null) return;
        Context context = viewInstance.getViewContext().getContext();
        if (context != null) {
            FeaturesRequestVoteJob.getInstance().start();
        }
    }

    private void unVote(FeatureRequest featureRequest) {
        featureRequest.setUserVoteStatus(FeatureRequest.UserVoteStatus.USER_UN_VOTED);
        try {
            FeatureRequestsDBHelper.insert(featureRequest);
        } catch (JSONException e) {
        }
        submitVote();
        FeatureRequestsEventBus.getInstance().post(featureRequest);
    }

    public void getFeatureTimeline(long featureId) {
        PoolProvider.postIOTask(() -> {
            if (repository != null) {
                repository.getFeatureTimeline(featureId, this);
            }
        });
    }

    @Override
    public void onDataChanged(TimelineResponse timeline) {
        PoolProvider.postMainThreadTask(new Runnable() {
            @Override
            public void run() {
                if (viewInstance == null) return;
                if (timeline.getTimelineObjects() != null && timeline.getTimelineObjects().size() > 0) {
                    viewInstance.showComments(timeline);
                    viewInstance.handelCompletedFeaturesActions();
                } else {
                    viewInstance.showNoCommentsLayout();
                }
            }
        });
    }

    @Override
    public void onError(Throwable error) {
        error.printStackTrace();

    }
}
