package com.instabug.featuresrequest.cache;

import com.instabug.featuresrequest.Constants;
import com.instabug.featuresrequest.models.FeatureRequest;
import com.instabug.library.internal.storage.cache.Cache;
import com.instabug.library.internal.storage.cache.CacheManager;
import com.instabug.library.internal.storage.cache.InMemoryCache;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

/**
 * @author hossam.
 */

public class NewFeatureRequestsCacheManager {

    public static final String NEW_FEATURES_DISK_CACHE_FILE_NAME = "/new_feature_requests.cache";
    public static final String NEW_FEATURES_DISK_CACHE_KEY = "new_feature_requests_disk_cache";
    public static final String NEW_FEATURES_MEMORY_CACHE_KEY = "new_feature_requests._memory_cache";

    /**
     * Loads new_feature_requests cache from disk if it's not in memory
     *
     * @return in-memory cache for new_feature_requests
     * @throws IllegalArgumentException if the from cache is not found
     */
    @Nullable
    public static InMemoryCache<Long, FeatureRequest> getCache() throws IllegalArgumentException {
        if (!CacheManager.getInstance().cacheExists(NEW_FEATURES_MEMORY_CACHE_KEY)) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "In-memory new_feature_requests cache not found, " +
                    "loading it from disk " + CacheManager.getInstance().getCache
                    (NEW_FEATURES_MEMORY_CACHE_KEY));
            CacheManager.getInstance().migrateCache(NEW_FEATURES_DISK_CACHE_KEY,
                    NEW_FEATURES_MEMORY_CACHE_KEY, new CacheManager.KeyExtractor<Long, FeatureRequest>() {
                        @Override
                        public Long extractKey(FeatureRequest featureRequest) {
                            return featureRequest.getFeatureId();
                        }
                    });
            Cache newFeaturesMemoryCache = CacheManager.getInstance().getCache
                    (NEW_FEATURES_MEMORY_CACHE_KEY);
            if (newFeaturesMemoryCache != null) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "In-memory new_feature_requests cache " +
                        "restored from disk, "
                        + newFeaturesMemoryCache.getValues().size() + " elements restored");
            }
        }
        InstabugSDKLogger.v(Constants.LOG_TAG, "In-memory new_feature_requests cache found");
        return (InMemoryCache<Long, FeatureRequest>) CacheManager.getInstance().getCache
                (NEW_FEATURES_MEMORY_CACHE_KEY);
    }

    @VisibleForTesting
    static void tearDown() {
        CacheManager.getInstance().deleteCache(NEW_FEATURES_MEMORY_CACHE_KEY);
        CacheManager.getInstance().deleteCache(NEW_FEATURES_DISK_CACHE_KEY);
    }

    /**
     * Saves all cached new_feature_requests from {@code NewFeatureRequestsCacheManager.NEW_FEATURES_MEMORY_CACHE_KEY}
     * to disk cache
     */
    public static void saveCacheToDisk() {

        final Cache newFeatureRequestsMemoryCache = CacheManager.getInstance().getCache
                (NEW_FEATURES_MEMORY_CACHE_KEY);
        final Cache newFeatureRequestsDiskCache = CacheManager.getInstance().getCache
                (NEW_FEATURES_DISK_CACHE_KEY);
        if (newFeatureRequestsMemoryCache != null
                && newFeatureRequestsDiskCache != null) {
            Thread savenewFeatureRequestsCacheToDiskInBackgroundThread = new Thread(new Runnable() {
                @Override
                public void run() {
                    CacheManager.getInstance().migrateCache(newFeatureRequestsMemoryCache, newFeatureRequestsDiskCache,
                            new CacheManager.KeyExtractor<String, FeatureRequest>() {
                                @Override
                                public String extractKey(FeatureRequest featureRequest) {
                                    return String.valueOf(featureRequest.getFeatureId());
                                }
                            });
                }
            });
            savenewFeatureRequestsCacheToDiskInBackgroundThread.setName("IBG-saveNewFeatureRequestsCacheToDiskInBackgroundThread");
            //start save new_feature_requests cache in background
            savenewFeatureRequestsCacheToDiskInBackgroundThread.start();
        }
    }

    /**
     * @param featureRequest new_feature_request that will be added to cache.
     */
    public static void addNewFeatureRequest(FeatureRequest featureRequest) {
        InMemoryCache<Long, FeatureRequest> cache = getCache();
        if (cache != null) {
            cache.put(featureRequest.getFeatureId(), featureRequest);
        }
    }

    /**
     * @param featureRequestId new_feature_request id.
     * @return featureRequest object that has the provided {@code featureRequestId}
     */
    @Nullable
    public static FeatureRequest getNewFeatureRequest(long featureRequestId) {
        InMemoryCache<Long, FeatureRequest> cache = getCache();
        if (cache != null) {
            return cache.get(featureRequestId);
        }
        return null;
    }

    /**
     * @param featureRequest new_feature_request that will be deleted.
     * @return featureRequest object that has the provided {@code featureRequest}
     */
    @Nullable
    public static FeatureRequest deleteNewFeatureRequest(FeatureRequest featureRequest) {
        InMemoryCache<Long, FeatureRequest> cache = getCache();
        if (cache != null) {
            return cache.delete(featureRequest.getFeatureId());
        }
        return null;
    }

    /**
     * @return All new_feature_requests in cache.
     */
    public static List<FeatureRequest> getNewFeatureRequests() {
        InMemoryCache<Long, FeatureRequest> cache = getCache();
        if (cache != null) {
            return cache.getValues();
        }
        return new ArrayList<>();
    }
}
