package com.instabug.featuresrequest.ui.featuredetails;

import android.content.Context;
import android.graphics.BitmapFactory;
import android.graphics.Color;
import android.graphics.Typeface;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;
import androidx.core.graphics.ColorUtils;

import com.instabug.featuresrequest.Constants;
import com.instabug.featuresrequest.R;
import com.instabug.featuresrequest.models.Comment;
import com.instabug.featuresrequest.models.StatusChange;
import com.instabug.featuresrequest.models.TimelineObject;
import com.instabug.featuresrequest.utils.DateUtils;
import com.instabug.featuresrequest.utils.DrawableUtils;
import com.instabug.featuresrequest.utils.StatusBinder;
import com.instabug.featuresrequest.utils.TextViewUtils;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugColorTheme;
import com.instabug.library.internal.storage.cache.AssetsCacheManager;
import com.instabug.library.model.AssetEntity;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.threading.PoolProvider;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by mzelzoghbi on 3/6/18.
 */

public class TimelineAdapter extends BaseAdapter {

    private static final int TYPE_USER_COMMENT = 0;
    private static final int TYPE_ADMIN_COMMENT = 1;
    private static final int TYPE_STATUS_CHANGE = 2;
    private static final String placeholderStatus = " %1s";
    ArrayList<TimelineObject> comments;
    FeaturesRequestsDetailsContract.View viewCallback;

    public TimelineAdapter(ArrayList<TimelineObject> comments, FeaturesRequestsDetailsContract.View viewCallback) {
        this.comments = comments;
        this.viewCallback = viewCallback;
    }

    @Override
    public int getCount() {
        return comments.size();
    }

    @Override
    public Object getItem(int position) {
        return comments.get(position);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @Override
    public int getItemViewType(int position) {
        if (comments.get(position) instanceof Comment) {
            if (((Comment) comments.get(position)).isAdmin()) {
                return TYPE_ADMIN_COMMENT;
            } else {
                return TYPE_USER_COMMENT;
            }
        } else {
            return TYPE_STATUS_CHANGE;
        }
    }

    @Override
    public View getView(final int position, View convertView, ViewGroup parent) {
        TimelineViewHolder viewHolder;
        int type = getItemViewType(position);
        if (convertView == null) {
            switch (type) {
                default:
                case TYPE_USER_COMMENT:
                    convertView = LayoutInflater.from(parent.getContext())
                            .inflate(R.layout.ib_fr_item_comment,
                                    parent, false);
                    break;
                case TYPE_ADMIN_COMMENT:
                    convertView = LayoutInflater.from(parent.getContext())
                            .inflate(R.layout.ib_fr_item_admin_comment,
                                    parent, false);
                    break;
                case TYPE_STATUS_CHANGE:
                    convertView = LayoutInflater.from(parent.getContext())
                            .inflate(R.layout.ib_fr_item_status_change,
                                    parent, false);
                    break;

            }
            viewHolder = new TimelineViewHolder(convertView);
            convertView.setTag(viewHolder);

        } else {
            viewHolder = (TimelineViewHolder) convertView.getTag();
        }

        switch (type) {
            default:
            case TYPE_USER_COMMENT:
                bindComment(convertView.getContext(), viewHolder, (Comment) getItem(position));
                break;
            case TYPE_ADMIN_COMMENT:
                bindAdminComment(convertView.getContext(), viewHolder, (Comment) getItem(position));
                break;
            case TYPE_STATUS_CHANGE:
                bindStatus(convertView.getContext(), viewHolder, (StatusChange) getItem(position));
                break;

        }
        return convertView;
    }


    private void bindStatus(Context context, TimelineViewHolder viewHolder, StatusChange item) {
        if (viewHolder.newStatus != null && viewHolder.newStatusDate != null) {
            viewHolder.newStatusDate.setText(DateUtils.getTimeAgo(context, item.getCreatedAt()));
            StatusBinder.bind(item.getNewStatus(), item.getNewStatusColorCode(), viewHolder.newStatus, context);
            viewHolder.newStatus.setTextColor(Color.parseColor(item.getNewStatusColorCode()));
            viewHolder.newStatus.setText( String.format(placeholderStatus, viewHolder.newStatus.getText()));
        }
    }

    private void bindAdminComment(Context context, TimelineViewHolder viewHolder, Comment item) {
        bindComment(context, viewHolder, item);
        if (viewHolder.username != null) {
            viewHolder.username.setTypeface(viewHolder.username.getTypeface(), Typeface.BOLD);
        }
        if (viewHolder.adminBackground != null) {
            viewHolder.adminBackground.getBackground().setAlpha(13);
            if (Instabug.getTheme() == InstabugColorTheme.InstabugColorThemeLight) {
                DrawableUtils.setColor(viewHolder.adminBackground, ColorUtils.setAlphaComponent(SettingsManager.getInstance().getPrimaryColor(), 255));
            } else {
                DrawableUtils.setColor(viewHolder.adminBackground, ContextCompat.getColor(context, android.R.color.white));
            }
        }
    }

    protected void bindComment(final Context context, final TimelineViewHolder viewHolder, final Comment comment) {
        if (viewHolder.username != null) {
            viewHolder.username.setTypeface(viewHolder.username.getTypeface(), Typeface.NORMAL);
            viewHolder.username.setText(comment.getCommenterName() != null
                    && !comment.getCommenterName().equalsIgnoreCase("null")
                    && !TextUtils.isEmpty(comment.getCommenterName().trim())
                    ? comment.getCommenterName()
                    : LocaleUtils.getLocaleStringResource(Instabug.getLocale(context), R.string.feature_request_owner_anonymous_word, context));
        }
        if (viewHolder.avatar != null) {
            if (comment.getAvatarFilePath() == null) {
                loadImage(context, comment, viewHolder.avatar);
                viewHolder.avatar.setImageResource(R.drawable.ibg_core_ic_avatar);
            } else {
                try {
                    viewHolder.avatar.setImageBitmap(BitmapFactory.decodeStream(new FileInputStream
                            (new File(comment.getAvatarFilePath()))));
                } catch (FileNotFoundException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Can't set avatar image in feature detail comments", e);
                }
            }
        }
        if (viewHolder.date != null) {
            viewHolder.date.setText(DateUtils.getTimeAgo(context, comment.getCreatedAt()));
        }

        String expandText = LocaleUtils.getLocaleStringResource(Instabug.getLocale(context), R.string.feature_request_str_more, context);
        String collapseText = LocaleUtils.getLocaleStringResource(Instabug.getLocale(context), R.string.feature_request_str_less, context);

        if (viewHolder.comment != null && expandText != null && collapseText != null)
            TextViewUtils.makeTextViewResizable(
                    viewHolder.comment,
                    comment.getBody(),
                    expandText,
                    collapseText,
                    comment.isLastViewmoreState(),
                    new Runnable() {
                        @Override
                        public void run() {
                            comment.setLastViewmoreState(!comment.isLastViewmoreState());
                            viewCallback.updateListHeight();
                        }
                    });
    }

    private void loadImage(Context context, final Comment comment, final ImageView imageView) {
        AssetEntity assetEntity = AssetsCacheManager.createEmptyEntity(context, comment.getAvatar(), AssetEntity
                .AssetType.IMAGE);
        PoolProvider.postIOTask(() -> AssetsCacheManager.getAssetEntity(assetEntity, new AssetsCacheManager
                .OnDownloadFinished() {
            @Override
            public void onSuccess(AssetEntity assetEntity1) {
                PoolProvider.postMainThreadTask(new Runnable() {
                    @Override
                    public void run() {

                        comment.setAvatarFilePath(assetEntity1.getFile().getAbsolutePath());
                        if (!assetEntity1.getFile().exists()) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Asset Entity downloading got FileNotFoundException" +
                                    " error");
                        }

                        notifyDataSetChanged();
                    }
                });
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Asset Entity downloading got error", error);
            }
        }));
    }

    public void add(List<Comment> comments) {
        comments.addAll(comments);
    }

    public static class TimelineViewHolder {
        @Nullable
        public final RelativeLayout adminBackground;
        @Nullable
        public final ImageView avatar;
        @Nullable
        public final TextView username;
        @Nullable
        public final TextView date;
        @Nullable
        public final TextView comment;
        // status changed
        @Nullable
        public final TextView newStatus;
        @Nullable
        public final TextView newStatusDate;

        TimelineViewHolder(View view) {
            avatar = view.findViewById(R.id
                    .instabug_feature_requests_comment_avatar);
            username = view.findViewById(R.id
                    .instabug_feature_requests_comment_username);
            date = view.findViewById(R.id
                    .instabug_feature_requests_comment_date);
            comment = view.findViewById(R.id
                    .instabug_feature_feature_details_comment);
            adminBackground = view.findViewById(R.id
                    .instabug_feature_request_admin_comment_layout);

            newStatus = view.findViewById(R.id.instabug_feature_request_new_status);
            newStatusDate = view.findViewById(R.id.instabug_feature_request_new_status_date);
        }
    }

}
