package com.instabug.chat.ui.chats;

import android.annotation.SuppressLint;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.chat.cache.ChatsCacheManager;
import com.instabug.chat.model.Chat;
import com.instabug.chat.model.Message;
import com.instabug.chat.notification.NotificationManager;
import com.instabug.chat.synchronization.NewMessagesHandler;
import com.instabug.chat.synchronization.OnNewMessagesReceivedListener;
import com.instabug.library.Instabug;
import com.instabug.library.PresentationManager;
import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.CacheChangedListener;
import com.instabug.library.internal.storage.cache.CacheManager;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

import io.reactivexport.android.schedulers.AndroidSchedulers;
import io.reactivexport.disposables.Disposable;
import io.reactivexport.observers.DisposableObserver;
import io.reactivexport.subjects.PublishSubject;


class ChatsPresenter extends BasePresenter<ChatsContract.View> implements ChatsContract
        .Presenter, CacheChangedListener<Chat>, OnNewMessagesReceivedListener {

    private static final String TAG = "ChatsPresenter";

    @Nullable
    private PublishSubject<Long> chatsCacheUpdatePublishSubject;
    @Nullable
    private Disposable chatsCacheUpdateDisposable;

    ChatsPresenter(ChatsContract.View view) {
        super(view);
    }

    @Override
    public void start() {
        subscribeToChatsCacheUpdates();
        subscribeToChatsCache(this);
        NewMessagesHandler.getInstance().addOnNewMessagesReceivedListener(this);
        updateUI();
    }

    @Override
    public void stop() {
        CacheManager.getInstance().unSubscribe(ChatsCacheManager.CHATS_MEMORY_CACHE_KEY, this);
        NewMessagesHandler.getInstance().removeOnNewMessagesReceivedListener(this);
        unSubscribeFromCacheUpdates();
    }

    @Override
    @Nullable
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_RETURN_ANNOTATION")
    public List<Message> onNewMessagesReceived(@NonNull final List<Message> newMessages) {
        if (view != null) {
            final ChatsContract.View viewRef = view.get();
            if (viewRef != null) {
                if (viewRef.getViewContext().getActivity() != null) {
                    if (viewRef.isChatsFragmentVisible()) {
                        NotificationManager.getInstance().playNotificationSound(viewRef
                                .getViewContext()
                                .getActivity());
                    } else {
                        if (Instabug.getApplicationContext() != null) {
                            PresentationManager.getInstance().show(new Runnable() {
                                @Override
                                public void run() {
                                    if (Instabug.getApplicationContext() != null) {
                                        NotificationManager.getInstance()
                                                .showNotification(Instabug.getApplicationContext(), newMessages);
                                    }
                                }
                            });
                        }
                    }
                }
            }
        }
        return null;
    }

    @Override
    public void onCachedItemRemoved(final Chat removedElement) {
        updateChatsFromCache(System.currentTimeMillis());
    }

    @Override
    public void onCachedItemAdded(final Chat addedElement) {
        updateChatsFromCache(System.currentTimeMillis());
    }

    @Override
    public void onCachedItemUpdated(final Chat oldElement, final Chat updatedElement) {
        updateChatsFromCache(System.currentTimeMillis());
    }

    @Override
    public void onCacheInvalidated() {
        updateChatsFromCache(System.currentTimeMillis());
    }

    private void updateChatsFromCache(long currentMilliSecondsTime) {
        if (chatsCacheUpdatePublishSubject != null) {
            chatsCacheUpdatePublishSubject.onNext(currentMilliSecondsTime);
        }
    }

    private void subscribeToChatsCacheUpdates() {
        chatsCacheUpdatePublishSubject = PublishSubject.create();
        chatsCacheUpdateDisposable = chatsCacheUpdatePublishSubject
                .debounce(300, TimeUnit.MILLISECONDS)
                .observeOn(AndroidSchedulers.mainThread())
                .subscribeWith(new DisposableObserver<Long>() {
                    @Override
                    public void onComplete() {

                    }

                    @Override
                    public void onError(Throwable e) {

                    }

                    @Override
                    public void onNext(Long currentMilliSecondsTime) {
                        updateUI();
                    }
                });
    }

    private void unSubscribeFromCacheUpdates() {
        if (chatsCacheUpdateDisposable != null && !chatsCacheUpdateDisposable.isDisposed()) {
            chatsCacheUpdateDisposable.dispose();
        }
    }

    private void updateUI() {
        ArrayList<Chat> chats = getChatsFromCache();
        Collections.sort(chats, Collections.reverseOrder(new Chat.Comparator()));
        if (view != null) {
            ChatsContract.View v = view.get();
            if (v != null) {
                v.updateChats(chats);
                v.showChats();
            }
        }
    }

    private synchronized ArrayList<Chat> getChatsFromCache() {
        ArrayList<Chat> chats;
        if (ChatsCacheManager.getCache() != null)
            chats = new ArrayList<>(ChatsCacheManager.getValidChats());
        else
            chats = new ArrayList<>();
        Collections.sort(chats, Collections.reverseOrder(new Chat.Comparator()));
        return chats;
    }

    private void subscribeToChatsCache(CacheChangedListener<Chat> listener) {
        try {
            CacheManager.getInstance().subscribe(ChatsCacheManager.CHATS_MEMORY_CACHE_KEY, listener);
        } catch (IllegalArgumentException exception) {
            InstabugSDKLogger.e(TAG, "Couldn't subscribe to cache", exception);
            IBGDiagnostics.reportNonFatal(exception, "Couldn't subscribe to cache");
        }
    }
}
