package com.instabug.chat.ui.chat;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.os.Bundle;
import android.util.DisplayMetrics;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.webkit.URLUtil;
import android.widget.ProgressBar;
import android.widget.Toast;

import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;

import com.instabug.bug.R;
import com.instabug.library.model.AssetEntity;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.library.view.ScaleImageView;

import org.jetbrains.annotations.NotNull;

/**
 * Created by vezikon on 12/15/15.
 */
@SuppressLint({"ERADICATE_FIELD_NOT_INITIALIZED", "ERADICATE_FIELD_NOT_NULLABLE"})
public class ImageAttachmentViewerFragment extends Fragment {

    public static final String TAG = "image_attachment_viewer_fragment";
    private final static String PARAM_IMG_URL = "img_url";

    private String imgUrl;

    //UI
    private ProgressBar mProgressBar;
    private ScaleImageView mImageView;

    private float windowWidth;
    private float windowHeight;

    public static ImageAttachmentViewerFragment newInstance(String imgUrl) {
        ImageAttachmentViewerFragment fragment = new ImageAttachmentViewerFragment();
        Bundle bundle = new Bundle();
        bundle.putString(PARAM_IMG_URL, imgUrl);
        fragment.setArguments(bundle);
        return fragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        if (getArguments() != null) {
            imgUrl = getArguments().getString(PARAM_IMG_URL);

            //no need to continue the method, we already got some data
            return;
        }

        if (savedInstanceState != null) {
            imgUrl = savedInstanceState.getString(PARAM_IMG_URL);
        }
    }

    @Override
    public void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        outState.putString(PARAM_IMG_URL, imgUrl);
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle
            savedInstanceState) {
        View view = inflater.inflate(R.layout.instabug_fragment_image_attachment_viewer,
                container, false);
        mProgressBar = (ProgressBar) view.findViewById(R.id.instabug_attachment_progress_bar);
        mImageView = (ScaleImageView) view.findViewById(R.id.instabug_img_attachment);
        return view;
    }

    @Override
    public void onViewCreated(@NotNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        if (getActivity() == null) {
            return;
        }
        //getting window dimensions in px
        DisplayMetrics metrics = new DisplayMetrics();
        getActivity().getWindowManager().getDefaultDisplay().getMetrics(metrics);
        int pixelsToSubtract = (int) convertDpToPixel(24, getActivity());
        windowWidth = metrics.widthPixels - pixelsToSubtract;
        windowHeight = metrics.heightPixels - pixelsToSubtract;

        if (URLUtil.isValidUrl(imgUrl)) {
            PoolProvider.postIOTask(new Runnable() {
                @Override
                public void run() {
                    BitmapUtils.loadBitmapForAsset(getActivity(), imgUrl, AssetEntity.AssetType.IMAGE, new BitmapUtils.OnBitmapReady() {
                        @Override
                        public void onBitmapReady(@Nullable final Bitmap bitmap) {
                            PoolProvider.postMainThreadTask(new Runnable() {
                                @Override
                                public void run() {
                                    if (mImageView != null) {
                                        showAttachment(bitmap);
                                    }
                                }
                            });
                        }

                        @Override
                        public void onBitmapFailedToLoad() {

                        }
                    });
                }
            });
        } else {
            BitmapUtils.loadBitmap(imgUrl, mImageView, windowWidth, windowHeight);
        }
    }

    private void showAttachment(@Nullable Bitmap bitmap) {
        if (bitmap != null) {
            mImageView.setImageBitmap(bitmap);
        } else {
            Context currentContext = getContext();
            if (currentContext != null) {
                Toast.makeText(currentContext, R.string
                        .instabug_str_image_loading_error, Toast.LENGTH_SHORT).show();
            }
        }
        if (mProgressBar.getVisibility() == View.VISIBLE)
            mProgressBar.setVisibility(View.GONE);
    }

    /**
     * This method converts dp unit to equivalent pixels, depending on device density.
     *
     * @param dp      A value in dp (density independent pixels) unit. Which we need to convert
     *                into pixels
     * @param context Context to get resources and device specific display metrics
     * @return A float value to represent px equivalent to dp depending on device density
     */
    public float convertDpToPixel(float dp, Context context) {
        Resources resources = context.getResources();
        DisplayMetrics metrics = resources.getDisplayMetrics();
        float px = dp * (metrics.densityDpi / 160f);
        return px;
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        mImageView = null;
        mProgressBar = null;
    }
}
