package com.instabug.chat.annotation.recognition;

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PathMeasure;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Created by Tarek360 on 12/28/16.
 */

public class PathMatcher {

    private static final int INTERSECTION_COLOR = 0XFF808080;
    private static final int USER_PATH_COLOR = 0X80FFFFFF;
    private static final int ORIGINAL_PATH_COLOR = 0XFF000000;
    private static final int ARROW_ANGLE_STEP = 10;
    private static final int RECT_FIRST_ANGLE_STEP = 10;
    private static final int STROKE_WIDTH = 2;
    private int userPathPixelsCount;
    private int userPathLeftWeight;
    private int userPathTopWeight;
    private int userPathRightWeight;
    private int userPathBottomWeight;
    private float pathLength;
    private final Path userPath;
    private Paint paint;

    public PathMatcher(Path path) {
        paint = new Paint();
        paint.setStyle(Paint.Style.STROKE);
        paint.setStrokeWidth(STROKE_WIDTH);
        this.userPath = Utility.resizePath(path);
        analyzePath(this.userPath);
    }

    private void analyzePath(Path path) {

        PathMeasure pathMeasure = new PathMeasure(path, false);
        pathLength = pathMeasure.getLength();

        Bitmap bitmap = Bitmap.createBitmap(ShapeSpecs.DIMEN, ShapeSpecs.DIMEN, Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);

        paint.setColor(Color.RED);
        canvas.drawPath(path, paint);

        int[] pixels = new int[ShapeSpecs.DIMEN * ShapeSpecs.DIMEN];

        bitmap.getPixels(pixels, 0, ShapeSpecs.DIMEN, 0, 0, ShapeSpecs.DIMEN, ShapeSpecs.DIMEN);
        for (int i = 0; i < pixels.length; i++) {
            if (pixels[i] == Color.RED) {
                userPathPixelsCount++;
                if (i < pixels.length / 2) {
                    userPathTopWeight++;
                } else {
                    userPathBottomWeight++;
                }

                if (i % ShapeSpecs.DIMEN < ShapeSpecs.DIMEN / 2) {
                    userPathLeftWeight++;
                } else {
                    userPathRightWeight++;
                }

            }
        }
    }

    private Weight matchPaths(Path originalPath, Path userPath) {

        Bitmap bitmap = Bitmap.createBitmap(ShapeSpecs.DIMEN, ShapeSpecs.DIMEN, Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);

        paint.setColor(ORIGINAL_PATH_COLOR);
        canvas.drawPath(originalPath, paint);

        int[] originalPixels = new int[ShapeSpecs.DIMEN * ShapeSpecs.DIMEN];
        int originalPixelsCount = 0;
        bitmap.getPixels(originalPixels, 0, ShapeSpecs.DIMEN, 0, 0, ShapeSpecs.DIMEN, ShapeSpecs.DIMEN);
        for (int pixel : originalPixels) {
            if (pixel == ORIGINAL_PATH_COLOR) {
                originalPixelsCount++;
            }
        }

        paint.setColor(USER_PATH_COLOR);
        canvas.drawPath(userPath, paint);

        int[] pixels = new int[ShapeSpecs.DIMEN * ShapeSpecs.DIMEN];

        bitmap.getPixels(pixels, 0, bitmap.getWidth(), 0, 0, ShapeSpecs.DIMEN, ShapeSpecs.DIMEN);

        int intersectedPixelsCount = 0;
        float originalPathNotConsumedPixelsCount = 0;
        float userPathNotConsumedPixelsCount = 0;

        Weight weight = new Weight();

        for (int pixel : pixels) {
            if (pixel == INTERSECTION_COLOR) {
                intersectedPixelsCount++;
            } else if (pixel == USER_PATH_COLOR) {
                userPathNotConsumedPixelsCount++;
            } else if (pixel == ORIGINAL_PATH_COLOR) {
                originalPathNotConsumedPixelsCount++;
            }
        }

        weight.value = intersectedPixelsCount;
        weight.pathNotConsumedPercent = userPathNotConsumedPixelsCount / userPathPixelsCount;

        weight.originalPathNotConsumedPercent = originalPathNotConsumedPixelsCount / originalPixelsCount;

        weight.probability = (1 - weight.originalPathNotConsumedPercent
                + 1 - weight.pathNotConsumedPercent
                + (float) intersectedPixelsCount / originalPixelsCount) / 3;


        weight.leftWeight = userPathLeftWeight;
        weight.topWeight = userPathTopWeight;
        weight.rightWeight = userPathRightWeight;
        weight.bottomWeight = userPathBottomWeight;
        weight.pathLength = pathLength;

        return weight;
    }

    private List<Weight> getArrowWeights(Path userPath) {
        List<Weight> arrowWeights = new ArrayList<>();
        Path path = PathFactory.getArrowPath();
        for (int i = 0; i < 360 / ARROW_ANGLE_STEP; i++) {
            int angle = i * ARROW_ANGLE_STEP;
            Weight weight = matchPaths(Utility.rotatePath(path, angle), userPath);
            weight.angle = angle;
            arrowWeights.add(weight);
        }
        return arrowWeights;
    }

    private List<Weight> getRectWeights(ShapeSpecs.Type type, int degreesFrom, int degreesTo,
                                        int angleStep) {
        List<Weight> weights = new ArrayList<>();
        List<RectPath> rectPaths = PathFactory.getRectPaths(type);
        if (rectPaths != null && !rectPaths.isEmpty()) {
            for (RectPath rectPath : rectPaths) {
                for (int i = degreesFrom; i < degreesTo; i++) {
                    int angle = i * angleStep;
                    Weight weight = matchPaths(Utility.rotatePath(rectPath.path, angle),
                            userPath);
                    weight.angle = angle;
                    weight.rectPaddingPercent = rectPath.paddingPercent;
                    weights.add(weight);
                    //OVAL with padding = 0 is a circle and no need to rotate.
                    if (type == ShapeSpecs.Type.OVAL && rectPath.paddingPercent == 0) {
                        break;
                    }
                }
            }
        }
        return weights;
    }

    public Weight getFineWeightRect(ShapeSpecs.Type type) {

        List<Weight> weights = getRectWeights(type, 0, 180 / RECT_FIRST_ANGLE_STEP, RECT_FIRST_ANGLE_STEP);

        Weight weight = Collections.max(weights);

        weight.averageProbability = getAverageProbability(weights);

        return weight;
    }

    public Weight getFineWeightArrow() {

        List<Weight> weights = getArrowWeights(userPath);

        Weight weight = Collections.max(weights);

        if ((weight.angle >= 0 && weight.angle <= 20) ||
                (weight.angle <= 360 && weight.angle >= 340) ||
                (weight.angle >= 160 && weight.angle <= 200) ||
                (weight.angle > 180 && weight.getVerticalWeight() == Weight.DIRECTION.TOP)
                || (weight.angle < 180 && weight.getVerticalWeight() == Weight.DIRECTION.BOTTOM)) {
            return weight;
        } else if (weight.angle < 180) {
            weight = weights.get((weight.angle + 180) / ARROW_ANGLE_STEP);
        } else {
            weight = weights.get((weight.angle - 180) / ARROW_ANGLE_STEP);
        }
        weight.averageProbability = getAverageProbability(weights);
        return weight;

    }


    private float getAverageProbability(List<Weight> weights) {
        float sumProbability = 0;
        for (Weight w : weights) {
            sumProbability += w.probability;
        }
        return sumProbability / weights.size();
    }

}
