package com.instabug.bug.view.reporting;

import static android.app.Activity.RESULT_OK;
import static com.instabug.bug.Constants.BUG_REPORTING_EXECUTOR_TASK;
import static com.instabug.bug.Constants.USER_CONSENT_KEY_PREFIX;
import static com.instabug.library.Constants.LogPlaceHolders.EMPTY_EMAIL;
import static com.instabug.library.Constants.LogPlaceHolders.NON_EMPTY_EMAIL;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.text.Spanned;
import android.util.Pair;
import android.util.Patterns;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.annotation.VisibleForTesting;

import com.instabug.bug.BugPlugin;
import com.instabug.bug.Constants;
import com.instabug.bug.LiveBugManager;
import com.instabug.bug.StateCreatorEventBus;
import com.instabug.bug.model.Bug;
import com.instabug.bug.screenrecording.ExternalScreenRecordHelper;
import com.instabug.bug.screenshot.ScreenshotHelper;
import com.instabug.bug.screenshot.viewhierarchy.ViewHierarchyInspector;
import com.instabug.bug.screenshot.viewhierarchy.utilities.ViewHierarchyInspectorEventBus;
import com.instabug.bug.settings.BugSettings;
import com.instabug.bug.userConsent.ActionType;
import com.instabug.bug.userConsent.UserConsent;
import com.instabug.bug.userConsent.UserConsentsManager;
import com.instabug.bug.view.visualusersteps.VisualUserStepsDisclaimerHelper;
import com.instabug.library.IBGFeature;
import com.instabug.library.InstabugCustomTextPlaceHolder;
import com.instabug.library.R;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.internal.InstabugMediaProjectionIntent;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.internal.storage.cache.Cache;
import com.instabug.library.internal.storage.cache.CacheManager;
import com.instabug.library.internal.video.MediaProjectionHelper;
import com.instabug.library.internal.video.RequestPermissionActivity;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.State;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.PlaceHolderUtils;
import com.instabug.library.util.VideoManipulationUtils;
import com.instabug.library.util.threading.PoolProvider;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.reactivexport.disposables.CompositeDisposable;
import io.reactivexport.functions.Consumer;

/**
 * @author Hossam
 * DEPENDENCIES:
 * **************
 * 1- {@link LiveBugManager}
 * 2- {@link SettingsManager}
 **/
public abstract class BaseReportingPresenter extends BasePresenter<IBaseReportingView.View>
        implements IBaseReportingView.Presenter {

    public static final int PICK_IMAGE_REQUEST_CODE = 0xF16;
    public static final int REQUEST_SCREEN_RECORDING_CAPTURE_PERMISSIONS = 0xF32;
    public static final int SHOW_THANK_YOU_ACTIVITY = 0xF64;
    public static final long MAX_FILE_SIZE_IN_MB = 50L;
    public static final long MAX_VIDEO_LENGTH_MILLI = 60000;
    public static final String VIDEO_PATH = "video.path";
    @Nullable
    private CompositeDisposable compositeDisposable;
    private WaitingPreparationAction waitingPreparationAction;
    @Nullable
    private List<UserConsent> consents;
    /**
     * Int counter for counting scheduled events and avoid dismissing the waiting
     * dialog and sending the report with missing info
     */
    private int waiting = 0;
    @NonNull
    private final UserConsentsManager userConsentsManager;

    private boolean isCommittingBug = false;

    private final ArrayList<Attachment> beingRemovedAttachments = new ArrayList<>();

    private boolean shouldShowScreenRecording = false;

    public BaseReportingPresenter(
            final IBaseReportingView.View view,
            @NonNull final UserConsentsManager userConsentsManager
    ) {
        super(view);

        waitingPreparationAction = WaitingPreparationAction.NONE;
        this.userConsentsManager = userConsentsManager;

        evaluateShouldShowScreenRecording(view);
    }

    @VisibleForTesting
    void evaluateShouldShowScreenRecording(IBaseReportingView.View view) {
        if (view != null) {
            Context context = view.getContext();
            if (context != null) {
                shouldShowScreenRecording = BugSettings.getInstance().getAttachmentsTypesParams().isAllowScreenRecording() && MediaProjectionHelper.INSTANCE.isMediaProjectionServiceAvailable(context);
            } else {
                shouldShowScreenRecording = false;
            }
        }
    }

    @Constants.ReportType
    abstract protected String getReportType();


    @Override
    public void onStart() {
        // subscribe on view hierarchy inspection events
        compositeDisposable = new CompositeDisposable();

        Bug bug = LiveBugManager.getInstance().getBug();
        if (bug != null) {
            if (bug.isRequiredViewHierarchy()) {
                subscribeOnViewHierarchyInspectorEventBus();
            }
            if (bug.getState() == null) {
                subscribeOnStateBuilt();
            }
        }

        if (InstabugCore.isFeatureEnabled(IBGFeature.VIEW_HIERARCHY_V2))
            subscribeOnViewHierarchyInspectorEventBus();

        showUserConsents(view.get());
    }

    private void showUserConsents(@Nullable final IBaseReportingView.View viewRef) {
        PoolProvider.postOrderedIOTask(BUG_REPORTING_EXECUTOR_TASK, () -> {
            if (viewRef != null && viewRef.getViewContext() != null && viewRef.getViewContext().getActivity() != null) {
                if (consents == null)
                    consents = userConsentsManager.getConsents();
                if (consents == null || consents.isEmpty()) return;
                viewRef.getViewContext().getActivity().runOnUiThread(() -> viewRef.showUserConsents(consents));
            }
        });
    }

    @Override
    public void onStop() {
        // un subscribe on view hierarchy inspection events
        if (compositeDisposable != null)
            compositeDisposable.dispose();
    }

    @Override
    public void onEmailChanged(String email) {
        if (LiveBugManager.getInstance().getBug() != null
                && LiveBugManager.getInstance().getBug().getState() != null) {
            LiveBugManager.getInstance().getBug().getState().setUserEmail(email);
        }
    }

    @Override
    public void onMessageChanged(String message) {
        if (LiveBugManager.getInstance().getBug() != null) {
            LiveBugManager.getInstance().getBug().setMessage(message);
        }
    }

    @Override
    public void pickPhotoFromGallery() {
        if (isCommittingBug)
            return;
        LiveBugManager.getInstance().setIsInBackground(true);
        if (view != null) {
            final IBaseReportingView.View viewRef = view.get();
            LiveBugManager.getInstance().onSdkDismissedForAttachment();
            if (viewRef != null) {
                viewRef.startGalleryPicker();
            }
        }
    }


    @Override
    public void takeScreenshot() {
        if (isCommittingBug)
            return;
        if (view != null) {
            IBaseReportingView.View viewRef = view.get();
            if (LiveBugManager.getInstance().getBug() != null &&
                    LiveBugManager.getInstance().getBug().isRequiredViewHierarchy() &&
                    LiveBugManager.getInstance().getBug().getViewHierarchyInspectionState()
                            == Bug.ViewHierarchyInspectionState.IN_PROGRESS) {
                waitingPreparationAction = WaitingPreparationAction.TAKE_EXTRA_SCREENSHOT;
                if (viewRef != null) {
                    viewRef.showPreparingDialog();
                }
            } else {
                if (viewRef != null) {
                    if (SettingsManager.getInstance().isScreenshotByMediaProjectionEnabled()) {
                        viewRef.requestMediaProjectionPermission();
                    } else {
                        prepareForTakingScreenShot(viewRef);
                    }
                }
            }
        }
    }

    private void prepareForTakingScreenShot(IBaseReportingView.View viewRef) {
        if (LiveBugManager.getInstance().getBug() != null) {
            LiveBugManager.getInstance().getBug().setBugState(Bug.BugState.IN_PROGRESS);
        }

        BugPlugin plugin = (BugPlugin) InstabugCore.getXPlugin(BugPlugin.class);
        if (plugin != null && plugin.getAppContext() != null) {
            plugin.setState(Plugin.STATE_PROCESSING_ATTACHMENT);
            SettingsManager.getInstance().setProcessingForeground(false);
            ScreenshotHelper.getInstance().startScreenshotCapturing(plugin.getAppContext());
        }
        if (viewRef != null) {
            viewRef.finishActivity();
        }
        LiveBugManager.getInstance().onSdkDismissedForAttachment();
    }

    @Override
    public void openVideoRecorder() {
        if (isCommittingBug)
            return;
        if (view != null) {
            IBaseReportingView.View viewRef = view.get();
            if (LiveBugManager.getInstance().getBug() != null &&
                    LiveBugManager.getInstance().getBug().isRequiredViewHierarchy() &&
                    LiveBugManager.getInstance().getBug().getViewHierarchyInspectionState()
                            == Bug.ViewHierarchyInspectionState.IN_PROGRESS) {
                waitingPreparationAction = WaitingPreparationAction.RECORD_VIDEO;
                if (viewRef != null) {
                    viewRef.showPreparingDialog();
                }
            } else {
                LiveBugManager.getInstance().onSdkDismissedForAttachment();
                ExternalScreenRecordHelper.getInstance().start();
                if (viewRef != null) {
                    viewRef.finishActivity();
                }

                BugPlugin bugPlugin = (BugPlugin) InstabugCore.getXPlugin(BugPlugin.class);
                if (bugPlugin != null) {
                    bugPlugin.setState(Plugin.STATE_PROCESSING_ATTACHMENT);
                }
            }
        }
    }

    @Override
    public void onSaveInstanceState(Bundle state) {

    }

    @Override
    public void onRestoreInstanceState(@Nullable Bundle state) {

    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        if (requestCode == PICK_IMAGE_REQUEST_CODE) {
            if (resultCode == RESULT_OK && data != null && data.getData() != null && view != null) {
                IBaseReportingView.View viewRef = view.get();
                if (viewRef != null) {
                    //Todo post this to BG Thread to fix DiskWriteViolation
                    if (Build.VERSION.SDK_INT >= 21) {
                        handleGalleryFileAbove21(viewRef, data);
                    } else {
                        handleGalleryFileBelow21(viewRef, data);
                    }
                }
            }

        } else if (requestCode == REQUEST_SCREEN_RECORDING_CAPTURE_PERMISSIONS) {
            if (resultCode == Activity.RESULT_OK && data != null) {
                InstabugMediaProjectionIntent.setMediaProjectionIntent(data);
                InstabugMediaProjectionIntent.setStaticResultCode(resultCode);
                openVideoRecorder();
            }
        } else if (requestCode == RequestPermissionActivity.REQUEST_MEDIA_PROJECTION_PERMISSION) {
            if (view.get() != null && data != null &&
                    data.getBooleanExtra(RequestPermissionActivity.KEY_IS_PERMISSION_GRANDTED, false))
                prepareForTakingScreenShot(view.get());
        }
    }

    private void handleGalleryFileBelow21(IBaseReportingView.View viewRef, @NonNull Intent data) {
        String filePath = AttachmentsUtility.getGalleryImagePath(viewRef.getActivity(), data.getData());
        if (filePath == null && data.getData() != null) {
            filePath = data.getData().getPath();
        }

        if (filePath == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Null file path while adding attachment from gallery");
            return;
        }

        String extension = FileUtils.getExtension(filePath);

        if (FileUtils.isImageExtension(extension)) {
            LiveBugManager.getInstance().addImageAttachment(
                    viewRef.getContext(), Uri.fromFile(new File(filePath)), Attachment.Type.GALLERY_IMAGE
            );
        } else if (FileUtils.isVideoExtension(extension)) {
            File file = new File(filePath);
            long fileSizeInMb = (file.length() / 1024L / 1024L);
            if (fileSizeInMb > MAX_FILE_SIZE_IN_MB) {
                viewRef.showMediaFileSizeAlert();
            } else if (VideoManipulationUtils.extractVideoDuration(filePath) > MAX_VIDEO_LENGTH_MILLI) {
                viewRef.showVideoLengthAlert();
            } else {
                LiveBugManager.getInstance().addVideoAttachment(
                        viewRef.getContext(), Uri.fromFile(file), Attachment.Type.GALLERY_VIDEO
                );
            }
        }
        LiveBugManager.getInstance().setIsInBackground(false);
    }

    @RequiresApi(api = Build.VERSION_CODES.KITKAT)
    private void handleGalleryFileAbove21(IBaseReportingView.View viewRef, Intent data) {
        Pair<String, String> fileNameAndSize = AttachmentsUtility.getFileNameAndSize(viewRef.getActivity(), data.getData());
        if (fileNameAndSize != null) {
            String extension = null, size = "0";

            String fileName = fileNameAndSize.first;

            if (fileNameAndSize.first != null)
                extension = FileUtils.getExtension(fileName);
            if (fileNameAndSize.second != null)
                size = fileNameAndSize.second;

            if (extension != null && FileUtils.isImageExtension(extension)) {
                File file = AttachmentsUtility.getFileFromContentProvider(viewRef.getContext(), data.getData(), fileName);
                if (file != null)
                    LiveBugManager.getInstance().addExternalAttachmentWithoutCopying(
                            viewRef.getContext(), file, Attachment.Type.GALLERY_IMAGE
                    );
            } else if (extension != null && FileUtils.isVideoExtension(extension)) {
                try {
                    double fileSizeInMb = (Double.parseDouble(size) / 1024 / 1024);
                    if (fileSizeInMb > MAX_FILE_SIZE_IN_MB) {
                        viewRef.showMediaFileSizeAlert();
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Attached video size exceeded the limit");
                        return;
                    }

                    File file = AttachmentsUtility.getFileFromContentProvider(viewRef.getContext(), data.getData(), fileName);

                    if (file != null) {
                        long videoLength = VideoManipulationUtils.extractVideoDuration(file.getPath());
                        if (videoLength > MAX_VIDEO_LENGTH_MILLI) {
                            viewRef.showVideoLengthAlert();
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Attached video length exceeded the limit, deleting file");

                            if (file.delete()) {
                                InstabugSDKLogger.v(Constants.LOG_TAG, "Attachment deleted");
                            }
                        } else {
                            LiveBugManager.getInstance().addVideoAttachment(
                                    viewRef.getContext(), Uri.fromFile(file), Attachment.Type.GALLERY_VIDEO
                            );
                        }
                    } else {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't get video attachment, file is null");
                    }
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e.getMessage() + " while adding video attachment", e);
                }
            }
        }
    }

    @Override
    public void removeAttachment(@NonNull Attachment fileToRemove) {
        if (isAttachmentBeingRemoved(fileToRemove)) {
            return;
        }
        beingRemovedAttachments.add(fileToRemove);
        PoolProvider.postIOTask(() -> {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Removing attachment: " + fileToRemove.getName());
            if (LiveBugManager.getInstance().getBug() != null)
                LiveBugManager.getInstance().getBug().getAttachments().remove(fileToRemove);
            if (fileToRemove.getLocalPath() != null) {
                File attachment = new File(fileToRemove.getLocalPath());
                if (Attachment.Type.EXTRA_VIDEO.equals(fileToRemove.getType()) || Attachment.Type.GALLERY_VIDEO.equals(fileToRemove.getType())) {
                    InstabugSDKLogger.v(Constants.LOG_TAG, "Removing video attachment");
                    Cache cache = CacheManager.getInstance()
                            .getCache(CacheManager.DEFAULT_IN_MEMORY_CACHE_KEY);
                    if (cache != null) {
                        Object deletedObject = cache.delete(VIDEO_PATH);
                        if (deletedObject != null) {
                            InstabugSDKLogger.v(Constants.LOG_TAG, "video attachment removed successfully");
                        }
                    }
                    if (LiveBugManager.getInstance().getBug() != null) {
                        LiveBugManager.getInstance().getBug().setHasVideo(false);
                    }
                }
                boolean isDeleted = attachment.delete();
                if (isDeleted) {
                    InstabugSDKLogger.v(Constants.LOG_TAG, "attachment removed successfully");
                    notifyAttachmentRemoved(fileToRemove);
                }
            }
            fileToRemove.setRemoved(true);
            beingRemovedAttachments.remove(fileToRemove);
        });
    }

    @Override
    public void refreshAttachments() {
        Bug liveBug = LiveBugManager.getInstance().getBug();
        if (liveBug != null) {
            if (view != null) {
                IBaseReportingView.View viewRef = view.get();
                if (viewRef != null) {
                    viewRef.setAttachments(liveBug.getAttachments());
                }
            }
        }
    }

    @Override
    public void notifyAttachmentRemoved(Attachment fileToRemove) {
        PoolProvider.postMainThreadTask(() -> {
            if (view != null) {
                IBaseReportingView.View viewRef = view.get();
                if (viewRef != null) {
                    viewRef.notifyAttachmentRemoved(fileToRemove);
                }
            }
        });
    }

    @Override
    public void handleVisualUserStepsDisclaimer(String body, String link) {
        if (VisualUserStepsDisclaimerHelper.isDisclaimerAvailable(body)) {
            if (view != null) {
                Spanned spanned = VisualUserStepsDisclaimerHelper.getSpannedDisclaimer(body, link);
                IBaseReportingView.View view = this.view.get();
                if (view != null) {
                    view.showVisualUserStepDisclaimer(spanned, body);
                }
            }
        } else {
            if (view != null) {
                IBaseReportingView.View view = this.view.get();
                if (view != null) {
                    view.hideVisualUserStepDisclaimer();
                }
            }
        }
    }

    @Override
    public void handleInstabugDisclaimer() {
        if (BugSettings.getInstance().getDisclaimerText() != null &&
                BugSettings.getInstance().getDisclaimerText().length() > 0) {
            if (view != null) {
                IBaseReportingView.View view = this.view.get();
                if (view != null) {
                    view.showInstabugDisclaimer(BugSettings.getInstance().getDisclaimerText());
                }
            }
        } else {
            if (view != null) {
                IBaseReportingView.View view = this.view.get();
                if (view != null) {
                    view.hideInstabugDisclaimer();
                }
            }
        }
    }

    @Override
    public void onSendClicked() {
        if (isCommittingBug)
            return;
        if (view != null) {
            IBaseReportingView.View viewRef = view.get();
            if (viewRef != null) {
                if (LiveBugManager.getInstance().getBug() == null) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "BUG WAS NULL - Recreate a new bug");
                    if (viewRef.getViewContext().getContext() != null) {
                        LiveBugManager.getInstance().createBug(viewRef.getViewContext().getContext());
                    } else {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't create the Bug due to Null context");
                    }
                } else if (LiveBugManager.getInstance().getBug() != null
                        && LiveBugManager.getInstance().getBug().getState() != null
                        && !BugSettings.getInstance().isEmailFieldVisible()) {
                    LiveBugManager.getInstance().getBug().getState().updateIdentificationAttrs();
                }
                boolean isEmailValid = checkUserEmailValid();
                boolean isCommentValid = checkCommentValid();
                if (!isEmailValid || !isCommentValid) {
                    return;
                }

                fillBugModelWithUserConsentResponses(LiveBugManager.getInstance().getBug());
                fillBugModelWithActionableConsents(LiveBugManager.getInstance().getBug());


                if (LiveBugManager.getInstance().getBug() != null &&
                        LiveBugManager.getInstance().getBug().isRequiredViewHierarchy() &&
                        LiveBugManager.getInstance().getBug().getViewHierarchyInspectionState()
                                == Bug.ViewHierarchyInspectionState.IN_PROGRESS) {
                    waitingPreparationAction = WaitingPreparationAction.SEND_BUG;
                    viewRef.showPreparingDialog();
                    return;
                }

                if (LiveBugManager.getInstance().getBug() != null
                        && LiveBugManager.getInstance().getBug().getState() == null) {
                    waitingPreparationAction = WaitingPreparationAction.SEND_BUG;
                    viewRef.showPreparingDialog();
                    return;
                }


                if (BugSettings.getInstance().isEmailFieldVisible()) {
                    SettingsManager.getInstance().setEnteredEmail(viewRef.getEnteredEmail());
                }

                boolean hasNextScreen = hasExtendedReport();

                if (!hasNextScreen) {
                    if (LiveBugManager.getInstance().getBug() != null &&
                            LiveBugManager.getInstance().getBug().getState() == null) {
                        viewRef.showPreparingDialog();
                    } else {
                        if (viewRef.getViewContext().getContext() != null) {
                            LiveBugManager.getInstance().commit();
                            isCommittingBug = true;
                        } else {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't commit the Bug due to Null context");
                        }
                        viewRef.navigateToSuccessFragment();
                    }
                } else {
                    viewRef.navigateToExtraFieldsFragment();
                }

                viewRef.notifyFragmentVisibilityChanged(false);
            }
        }
    }

    private void fillBugModelWithUserConsentResponses(@Nullable Bug bug) {
        if (bug == null) return;
        Map<String, String> userConsentResponses = getUserConsentResponses();
        if (userConsentResponses != null && !userConsentResponses.isEmpty())
            bug.setUserConsentResponses(userConsentResponses);
    }

    private void fillBugModelWithActionableConsents(@Nullable Bug bug) {
        if (bug == null) return;
        List<@ActionType String> actionableConsent = getActionableConsents();
        if (actionableConsent != null && !actionableConsent.isEmpty())
            bug.setActionableConsents(actionableConsent);
    }

    @Nullable
    private Map<String, String> getUserConsentResponses() {
        if (consents == null || consents.isEmpty()) return null;
        Map<String, String> consentsMap = new HashMap<>();
        for (UserConsent consent : consents) {
            if (consent.getKey() != null)
                consentsMap.put(USER_CONSENT_KEY_PREFIX + consent.getKey(), consent.isChecked() + "");
        }
        return consentsMap;
    }

    @Nullable
    private List<@ActionType String> getActionableConsents() {
        if (consents == null || consents.isEmpty()) return null;
        List<@ActionType String> actionableConsent = new ArrayList<>();
        for (UserConsent consent : consents) {
            if (consent.getKey() != null && consent.getActionType() != null && !consent.isChecked())
                actionableConsent.add(consent.getActionType());
        }
        return actionableConsent;
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    boolean checkUserEmailValid() {
        IBaseReportingView.View viewRef = view.get();
        Bug bug = LiveBugManager.getInstance().getBug();
        String email = null;
        if (bug != null && bug.getState() != null) {
            email = bug.getState().getUserEmail();
            if (email != null) {
                email = email.trim();
                String emailLog = email.isEmpty() ? EMPTY_EMAIL : NON_EMPTY_EMAIL;
                InstabugSDKLogger.v(Constants.LOG_TAG, "checkUserEmailValid :" + emailLog);
            }
        }
        if ((email == null || email.isEmpty()) && viewRef != null) {
            email = viewRef.getEnteredEmail().trim();
            onEmailChanged(email);
        }

        boolean isEmailValid = true;
        if (BugSettings.getInstance().isEmailFieldVisible()) {
            if (BugSettings.getInstance().isEmailFieldRequired()
                    && (email == null || email.isEmpty())) {
                isEmailValid = false;
            }
            if (email != null && !email.isEmpty()
                    && !Patterns.EMAIL_ADDRESS.matcher(email.trim()).matches()) {
                isEmailValid = false;
            }

            if (!isEmailValid) {
                if (viewRef != null) {
                    String message = PlaceHolderUtils.getPlaceHolder(InstabugCustomTextPlaceHolder.Key
                                    .INVALID_EMAIL_MESSAGE,
                            viewRef.getLocalizedString(R.string
                                    .instabug_err_invalid_email));
                    String emailLog = email == null || email.isEmpty() ? EMPTY_EMAIL : NON_EMPTY_EMAIL;
                    InstabugSDKLogger.v(Constants.LOG_TAG, "checkUserEmailValid failed with " + emailLog + " email");
                    viewRef.showEmailError(message);
                }
            }
        }
        return isEmailValid;
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private boolean checkCommentValid() {
        IBaseReportingView.View viewRef = view.get();

        String comment = null;
        if (LiveBugManager.getInstance().getBug() != null) {
            comment = LiveBugManager.getInstance().getBug().getMessage();
        }
        final int userMinCharLimit = BugSettings.getInstance().getReportTypeCommentMinCharLimit(getReportType());
        final int minCharLimit = Math.max(2, userMinCharLimit);
        final boolean isCommentRequired = BugSettings.getInstance().isCommentFieldRequired() || userMinCharLimit != 0;
        if (isCommentRequired
                && (comment == null || comment.trim().length() < minCharLimit)) {
            if (viewRef != null) {
                String message = PlaceHolderUtils.getPlaceHolder(InstabugCustomTextPlaceHolder.Key.COMMENT_FIELD_INSUFFICIENT_CONTENT,
                        viewRef.getLocalizedString(R.string.instabug_err_invalid_comment));
                message = String.format(message, minCharLimit);
                String commentLog = comment == null || comment.isEmpty() ? "empty-comment" : "non-empty-comment";
                InstabugSDKLogger.v(Constants.LOG_TAG, "checkCommentValid comment field is invalid : " + commentLog);
                viewRef.showCommentError(message);
                return false;
            }
        }

        return true;
    }

    @Override
    @Nullable
    public Attachment getLastVideoAttachment(List<Attachment> attachments) {
        //getting last video attachment
        for (int i = attachments.size() - 1; i >= 0; i--) {
            Attachment attachment = attachments.get(i);
            if (attachment.getType() == Attachment.Type.EXTRA_VIDEO || attachment.getType() == Attachment.Type.GALLERY_VIDEO) {
                return attachment;
            }
        }
        return null;
    }

    @Override
    public void updateLastVideoAttachment(List<Attachment> attachments, String videoFilePath) {
        //update video attachment with encoded video file with audio
        for (int i = attachments.size() - 1; i >= 0; i--) {
            Attachment attachment = attachments.get(i);
            if (attachment.getType() == Attachment.Type.EXTRA_VIDEO || attachment.getType() == Attachment.Type.GALLERY_VIDEO) {
                attachment.setLocalPath(videoFilePath);
                // set video encoded
                attachment.setVideoEncoded(true);
                break;
            }
        }
    }

    @Override
    public boolean hasValidConsents() {
        if (consents == null) return true;
        for (UserConsent consent : consents) {
            if (consent.isMandatory() && !consent.isChecked()) return false;
        }
        return true;
    }

    private void subscribeOnViewHierarchyInspectorEventBus() {
        ++waiting;
        if (compositeDisposable != null) {
            compositeDisposable.add(ViewHierarchyInspectorEventBus.getInstance().getEventObservable()
                    .subscribe(new Consumer<ViewHierarchyInspector.Action>() {
                        @Override
                        public void accept(ViewHierarchyInspector.Action action) {
                            // decrease waiting count
                            --waiting;
                            InstabugSDKLogger.d(Constants.LOG_TAG, "Received a view hierarchy inspection action, action " +
                                    "value: " + action);
                            if (action == ViewHierarchyInspector.Action.COMPLETED
                                    || action == ViewHierarchyInspector.Action.FAILED) {
                                if (view != null) {
                                    final IBaseReportingView.View viewRef = view.get();
                                    executeWaitingPreparationAction(viewRef);
                                }
                            }
                        }
                    }, new Consumer<Throwable>() {
                        @Override
                        public void accept(Throwable throwable) {
                            // decrease waiting count
                            --waiting;

                            if (view != null) {
                                final IBaseReportingView.View viewRef = view.get();
                                executeWaitingPreparationAction(viewRef);
                            }
                        }
                    }));
        }
    }

    /**
     * Subscribing on state built event.
     * <p>
     * This method adds a new disposable to listen for state created/built event.
     */
    private void subscribeOnStateBuilt() {
        // increase waiting count.
        ++waiting;
        if (compositeDisposable != null) {
            compositeDisposable.add(StateCreatorEventBus.getInstance()
                    .getEventObservable().subscribe(new Consumer<State.Action>() {
                        @Override
                        public void accept(State.Action action) throws Exception {
                            --waiting;
                            if (view != null) {
                                final IBaseReportingView.View viewRef = view.get();
                                executeWaitingPreparationAction(viewRef);
                            }
                            InstabugSDKLogger.v(Constants.LOG_TAG, "State Building finished action");
                            // release the bus to optmize memory
                            StateCreatorEventBus.release();
                        }
                    }, new Consumer<Throwable>() {
                        @Override
                        public void accept(Throwable throwable) throws Exception {
                            // decrease waiting count
                            --waiting;
                            if (view != null) {
                                final IBaseReportingView.View viewRef = view.get();
                                executeWaitingPreparationAction(viewRef);
                            }
                            InstabugSDKLogger.e(Constants.LOG_TAG, "State Building got error: " + throwable.getMessage());
                            // release the bus to optmize memory
                            StateCreatorEventBus.release();
                        }
                    }));
        }
    }

    private void executeWaitingPreparationAction(@Nullable final IBaseReportingView.View viewRef) {
        if (viewRef != null && viewRef.getViewContext().getActivity() != null) {
            viewRef.getViewContext().getActivity().runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    // this to only execute sending but when all scheduled jobs finished.
                    // If State was still being created and used tried to send bug he will be blocked
                    // till the state is created, but if he wants to take extra screenshot,,,etc he will be able
                    // to do so.
                    if (waitingPreparationAction == WaitingPreparationAction.SEND_BUG
                            && waiting != 0)
                        return;

                    viewRef.dismissPreparingDialog();
                    switch (waitingPreparationAction) {
                        case SEND_BUG:
                            onSendClicked();
                            break;
                        case TAKE_EXTRA_SCREENSHOT:
                            takeScreenshot();
                            break;
                        case RECORD_VIDEO:
                            openVideoRecorder();
                            break;
                        default:
                            break;
                    }
                }
            });
        }
    }

    @Override
    public boolean isAttachmentBeingRemoved(Attachment attachment) {
        return beingRemovedAttachments.contains(attachment) || attachment.isRemoved();
    }


    @Override
    public boolean shouldShowScreenRecording() {
        return shouldShowScreenRecording;
    }

    @Override
    public boolean shouldShowExtraScreenshot() {
        boolean isExtraScreenshotAllowed = BugSettings.getInstance().getAttachmentsTypesParams().isAllowTakeExtraScreenshot();
        boolean isExtraScreenshotAllowedByMediaProjection = isExtraScreenshotAllowed && SettingsManager.getInstance().isScreenshotByMediaProjectionEnabled();

        IBaseReportingView.View viewRef = view.get();
        Context context = viewRef == null ? null : viewRef.getContext();

        if (isExtraScreenshotAllowedByMediaProjection) {
            return context != null && MediaProjectionHelper.INSTANCE.isMediaProjectionServiceAvailable(context);
        } else {
            return isExtraScreenshotAllowed;
        }
    }


    private enum WaitingPreparationAction {
        NONE, SEND_BUG, TAKE_EXTRA_SCREENSHOT, RECORD_VIDEO
    }

}
