package com.instabug.bug.invocation.invocationdialog;

import static android.os.Build.VERSION.SDK_INT;
import static android.os.Build.VERSION_CODES.JELLY_BEAN;
import static androidx.core.view.accessibility.AccessibilityNodeInfoCompat.ACTION_SCROLL_FORWARD;
import static com.instabug.library.util.ThemeApplier.applyCtaStyle;
import static com.instabug.library.util.ThemeApplier.applyTitleStyle;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.util.DisplayMetrics;
import android.view.View;
import android.view.WindowManager;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.AdapterView;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.ListView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.core.content.ContextCompat;
import androidx.core.view.AccessibilityDelegateCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;

import com.instabug.bug.R;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.InstabugBaseFragment;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.AttrResolver;
import com.instabug.library.util.Colorizer;
import com.instabug.library.util.DrawableUtils;
import com.instabug.library.util.ListUtils;
import com.instabug.library.util.ThemeApplier;
import com.instabug.library.view.ViewUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Created by mzelzoghbi on 4/18/18.
 */
public class InstabugDialogFragment extends InstabugBaseFragment<InstabugDialogFragmentPresenter>
        implements InstabugDialogFragmentContract.View, AdapterView.OnItemClickListener {

    @Nullable
    private TextView titleTextView;
    @Nullable
    private InstabugDialogAdapter promptOptionsAdapter;
    @Nullable
    private ArrayList<InstabugDialogItem> dialogItems;
    @Nullable
    private InstabugDialogAction chatsListAction = null;
    @Nullable
    private Callbacks listener;
    @Nullable
    private InstabugDialogActivityContract.AnimationProvider animationProvider;
    @Nullable
    private ListView promptOptionsListView;

    public static InstabugDialogFragment newInstance(@Nullable String title,
                                                     boolean shouldOverrideTitleContentDescription,
                                                     @Nullable ArrayList<InstabugDialogItem> dialogItems) {
        InstabugDialogFragment fragment = new InstabugDialogFragment();
        Bundle bundle = new Bundle();
        bundle.putString(InstabugDialogActivity.KEY_DIALOG_TITLE, title);
        bundle.putBoolean(InstabugDialogActivity.KEY_DIALOG_SHOULD_OVERRIDE_TITLE_DESC,
                shouldOverrideTitleContentDescription);
        bundle.putSerializable(InstabugDialogActivity.KEY_DIALOG_ITEMS, dialogItems);
        fragment.setArguments(bundle);
        return fragment;
    }

    @SuppressWarnings("unchecked")
    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        setRetainInstance(true);
        super.onCreate(savedInstanceState);
//        TODO uncomment this block when https://issuetracker.google.com/issues/188457866 is fixed.
//        Context context = getContext();
//        if (SDK_INT >= LOLLIPOP && context != null) {
//            setSharedElementEnterTransition(TransitionInflater.from(context).inflateTransition(android.R.transition.move));
//        }
        if (presenter == null) {
            presenter = createPresenter();
        }

        ArrayList<InstabugDialogItem> originalDialogItem = null;
        if (getArguments() != null) {
            originalDialogItem = (ArrayList<InstabugDialogItem>) getArguments().getSerializable(InstabugDialogActivity.KEY_DIALOG_ITEMS);
        }
        if (originalDialogItem != null)
            setChatsListAction(originalDialogItem);
    }


    /**
     * This method to clone the current dialog items into anther list to delete the chat item if
     * found and set chat item to handle the configuration change
     *
     * @param dialogItemsWithoutChatOption list item to be cloned.
     */
    private void setChatsListAction(@NonNull List<InstabugDialogItem> dialogItemsWithoutChatOption) {
        dialogItems = new ArrayList<>(dialogItemsWithoutChatOption);
        Collections.copy(dialogItems, dialogItemsWithoutChatOption);
        int index = -1;
        for (int i = 0; i < dialogItems.size(); i++) {
            InstabugDialogItem instabugDialogItem = dialogItems.get(i);
            if (instabugDialogItem instanceof InstabugDialogAction) {
                index = i;
                break;
            }
        }
        if (index != -1) {
            chatsListAction = (InstabugDialogAction) dialogItems.remove(index);
        }
    }

    @Override
    public void onDestroyView() {
        playExitAnimation();
        titleTextView = null;
        promptOptionsListView = null;
        promptOptionsAdapter = null;
        super.onDestroyView();
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        if (context instanceof InstabugDialogFragment.Callbacks && context instanceof InstabugDialogActivityContract.AnimationProvider) {
            listener = (InstabugDialogFragment.Callbacks) context;
            animationProvider = (InstabugDialogActivityContract.AnimationProvider) context;
        } else {
            throw new RuntimeException(context.toString()
                    + " must implement InstabugDialogFragment.Callbacks and AnimationProvider");
        }
    }

    @Override
    public void onDetach() {
        super.onDetach();
        listener = null;
        animationProvider = null;
    }


    @Override
    protected int getLayout() {
        return R.layout.ib_core_lyt_dialog_fragment;
    }


    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        final View viewContainer = findViewById(R.id.instabug_main_prompt_container);
        if (viewContainer != null && getContext() != null) {
            resizeDialog(viewContainer);
            DrawableUtils.setColor(viewContainer,
                    AttrResolver.resolveAttributeColor(getContext(), R.attr.instabug_background_color));
        }
        titleTextView = findViewById(R.id.instabug_fragment_title);
        TextView textView = titleTextView;
        if (textView != null) {
            ViewCompat.setTransitionName(textView, "title");
            if (AccessibilityUtils.isTalkbackEnabled() && getArguments() != null &&
                    getArguments().getBoolean(InstabugDialogActivity.KEY_DIALOG_SHOULD_OVERRIDE_TITLE_DESC)) {
                textView.setContentDescription(getLocalizedString(R.string.ibg_prompt_options_title_content_description));
            }
        }
        if (chatsListAction != null) {
            // container
            View chatsListContainer = findViewById(R.id.instabug_chats_list_icon_container);
            if (chatsListContainer != null) {
                chatsListContainer.setVisibility(View.VISIBLE);
                if (listener != null) {
                    chatsListContainer.setOnClickListener(v -> {
                        if (chatsListAction != null && listener != null) {
                            listener.removeScreenshotIfNeeded(chatsListAction);
                            listener.onDialogItemClicked(
                                    chatsListAction,
                                    findViewById(R.id.instabug_main_prompt_container), // dialog container
                                    findViewById(R.id.instabug_pbi_container)); // powerByInstabug container
                        }
                    });
                }
            }
            // icon
            ImageView chatsListIcon = findViewById(R.id.instabug_chats_list_icon);
            if (chatsListIcon != null) {
                chatsListIcon.getDrawable().setColorFilter(SettingsManager.getInstance().getPrimaryColor(), PorterDuff.Mode.SRC_IN);
            }
            // counter
            TextView notificationCount = findViewById(R.id.instabug_notification_count);
            if (chatsListAction.getBadgeCount() > 0) {
                if (notificationCount != null) {
                    notificationCount.setVisibility(View.VISIBLE);
                    notificationCount.setContentDescription(getLocalizedString(R.string.ibg_prompt_options_notification_count_content_description, chatsListAction.getBadgeCount()));
                }
                int tintColor = getResources().getColor(R.color.ib_core_notification_dot_color);
                if (notificationCount != null && getContext() != null) {
                    Drawable drawable = ContextCompat.getDrawable(getContext(), R.drawable.ibg_core_bg_white_oval);
                    Drawable inboxIconBackground = null;
                    if (drawable != null) {
                        inboxIconBackground = Colorizer.getTintedDrawable(tintColor, drawable);
                    }
                    notificationCount.setBackgroundDrawable(inboxIconBackground);
                }
                if (notificationCount != null) {
                    notificationCount.setText(String.valueOf(chatsListAction.getBadgeCount()));
                }
            } else {
                if (notificationCount != null) {
                    notificationCount.setVisibility(View.GONE);
                }
            }
        }
        promptOptionsListView = findViewById(R.id
                .instabug_prompt_options_list_view);
        ListView listView = promptOptionsListView;
        if (listView != null) {
            listView.setOnItemClickListener(this);
            promptOptionsAdapter = new InstabugDialogAdapter(theme);
            listView.setAdapter(promptOptionsAdapter);

            if (AccessibilityUtils.isTalkbackEnabled()) {
                ViewCompat.setAccessibilityDelegate(listView, new AccessibilityDelegateCompat() {
                    @Override
                    public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                        super.onInitializeAccessibilityNodeInfo(host, info);
                        info.addAction(new AccessibilityNodeInfoCompat.AccessibilityActionCompat(ACTION_SCROLL_FORWARD,
                                getLocalizedString(R.string.ibg_prompt_options_list_view_scroll_description)));
                    }
                });
            }
        }
        Button cancelBtn = findViewById(R.id.instabug_prompt_cancel_btn);
        cancelBtn.setTextColor(SettingsManager.getInstance().getPrimaryColor());
        cancelBtn.setOnClickListener((view) -> finishActivity());
        showPromptOptions();
        playEnterAnimation();

        applyTheme(titleTextView, cancelBtn, viewContainer);
    }

    @VisibleForTesting
    void applyTheme(@Nullable TextView titleTextView, @Nullable Button cancelButton, View viewContainer) {
        if (theme != null) {
            applyTitleStyle(titleTextView, theme);
            applyCtaStyle(cancelButton, theme);

            int bgColor = theme.getBackgroundColor();
            if (bgColor != ThemeApplier.DEFAULT_COLOR) {
                viewContainer.setBackgroundColor(bgColor);
            }
        }
    }


    // this method is for resizing dialog
    private void resizeDialog(View viewContainer) {
        if (getActivity() != null) {
            WindowManager wm = (WindowManager) getActivity().getSystemService(Context.WINDOW_SERVICE);
            final DisplayMetrics metrics = new DisplayMetrics();
            if (wm != null) {
                wm.getDefaultDisplay().getMetrics(metrics);
            }

            Context context = Instabug.getApplicationContext();
            if (dialogItems != null && context != null && ((ViewUtils.convertDpToPx(context, 56) * dialogItems.size())
                    + ViewUtils.convertDpToPx(context, 200)) > metrics.heightPixels) {
                RelativeLayout.LayoutParams layoutPrams
                        = new RelativeLayout.LayoutParams(WindowManager.LayoutParams.WRAP_CONTENT,
                        metrics.heightPixels - ViewUtils.convertDpToPx(context, 110));
                layoutPrams.addRule(RelativeLayout.CENTER_IN_PARENT);
                viewContainer.setLayoutParams(layoutPrams);
            }
        }

    }

    @Override
    public void onStart() {
        super.onStart();
        if (presenter != null) {
            presenter.onViewStarted();
        }
    }

    @Override
    public void onStop() {
        super.onStop();
        if (presenter != null) {
            presenter.onViewStopped();
        }
    }

    @Override
    public void handlePoweredByFooter() {
        InstabugCore.handlePbiFooter(rootView);

        View container = findViewById(R.id.instabug_pbi_container);
        if (container != null
                && container.getVisibility() == View.VISIBLE
                && AccessibilityUtils.isTalkbackEnabled()) {
            container.setImportantForAccessibility(
                    View.IMPORTANT_FOR_ACCESSIBILITY_NO_HIDE_DESCENDANTS
            );
        }
    }

    private InstabugDialogFragmentPresenter createPresenter() {
        return new InstabugDialogFragmentPresenter(this);
    }

    @Override
    public void showTitle() {
        TextView textView = titleTextView;
        if (textView != null && getArguments() != null && getArguments().getString(InstabugDialogActivity.KEY_DIALOG_TITLE) != null) {
            textView.setText(getArguments().getString(InstabugDialogActivity.KEY_DIALOG_TITLE));
        }
    }

    @Override
    public void showPromptOptions() {
        if (dialogItems != null && promptOptionsAdapter != null && dialogItems.size() > 0) {
            promptOptionsAdapter.setPluginPromptOptions(dialogItems);
            promptOptionsAdapter.notifyDataSetChanged();
        }
    }

    @Override
    public void playEnterAnimation() {
        Context context = getContext();
        if (context != null && animationProvider != null) {
            // title
            final View title = findViewById(R.id.layout_title_container);
            if (title != null) {
                Animation titleAnimation = AnimationUtils.loadAnimation(context, animationProvider.getEnterAnimation());
                titleAnimation.setStartOffset(100);
                title.setAnimation(titleAnimation);
            }
            ListView listView = promptOptionsListView;
            if (listView != null) {
                Animation contentAnimation = AnimationUtils.loadAnimation(context, animationProvider.getEnterAnimation());
                contentAnimation.setStartOffset(100);
                contentAnimation.setAnimationListener(new Animation.AnimationListener() {
                    @Override
                    public void onAnimationStart(Animation animation) {
                        listView.setVerticalScrollBarEnabled(false);
                    }

                    @Override
                    public void onAnimationEnd(Animation animation) {
                        listView.setVerticalScrollBarEnabled(true);

                    }

                    @Override
                    public void onAnimationRepeat(Animation animation) {

                    }
                });
                if (SDK_INT >= JELLY_BEAN)
                    listView.setScrollBarDefaultDelayBeforeFade(0);
                listView.setAnimation(contentAnimation);
            }
        }
    }

    @Override
    public void playExitAnimation() {
        Context context = getContext();
        if (context != null && animationProvider != null) {
            // title
            final View title = findViewById(R.id.layout_title_container);
            if (title != null) {
                Animation titleAnimation = AnimationUtils.loadAnimation(context, animationProvider.getExitAnimation());
                title.setAnimation(titleAnimation);
            }
            ListView listView = promptOptionsListView;
            if (listView != null) {
                Animation contentAnimation = AnimationUtils.loadAnimation(context, animationProvider.getExitAnimation());
                contentAnimation.setAnimationListener(new Animation.AnimationListener() {
                    @Override
                    public void onAnimationStart(Animation animation) {
                        listView.setVerticalScrollBarEnabled(false);
                    }

                    @Override
                    public void onAnimationEnd(Animation animation) {
                        listView.setVerticalScrollBarEnabled(true);
                    }

                    @Override
                    public void onAnimationRepeat(Animation animation) {

                    }
                });
                listView.setAnimation(contentAnimation);
            }
        }
    }

    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
        if (promptOptionsListView != null) {
            promptOptionsListView.setOnItemClickListener(null);
        }
        if (listener != null) {
            listener.onDialogItemClicked(
                    ListUtils.safeGet(dialogItems, position),
                    findViewById(R.id.instabug_main_prompt_container), // dialog container
                    findViewById(R.id.instabug_pbi_container)); // powerByInstabug container
        }
    }


    interface Callbacks {

        void onDialogItemClicked(@Nullable InstabugDialogItem item, View... sharedViews);

        void removeScreenshotIfNeeded(InstabugDialogItem chatsListAction);
    }
}
