package com.instabug.bug.userConsent

import com.instabug.bug.Constants
import com.instabug.bug.Constants.INVALID_USER_CONSENT_DESCRIPTION_LENGTH
import com.instabug.bug.Constants.INVALID_USER_CONSENT_KEY_LENGTH
import com.instabug.bug.Constants.MAX_CONSENT_NUMBER_REACHED
import com.instabug.bug.Constants.NULL_OR_EMPTY_USER_CONSENT_KEY
import com.instabug.bug.configurations.BugReportingConfigurationsProvider
import com.instabug.library.util.InstabugSDKLogger

interface UserConsentValidator {
    operator fun invoke(consent: UserConsent, currentConsentKeys: Set<String>): UserConsent?
}

class UserConsentValidatorImpl(private val configs: BugReportingConfigurationsProvider) :
    UserConsentValidator {

    override fun invoke(consent: UserConsent, currentConsentKeys: Set<String>): UserConsent? =
        consent.takeUnless { it.key.isNullOrBlank() }
            ?.let(::removedEmptyPrefixAndSuffix)
            ?.let(::trimKeyIfRequired)
            ?.let(::validateDescription)
            ?.also { logWarningIfPassedConsentCountLimit(currentConsentKeys, it) }
            .also(::logInvalidKeyErrorIfRequired)

    private fun removedEmptyPrefixAndSuffix(consent: UserConsent): UserConsent = with(consent) {
        val trimmedKey = key?.trim()
        val trimmedDescription = description?.trim()
        takeIf { it.key == trimmedKey && it.description == trimmedDescription }
            ?: copy(key = trimmedKey, description = trimmedDescription)
    }

    private fun trimKeyIfRequired(consent: UserConsent): UserConsent = consent.key
        ?.takeIf { it.length > configs.consentKeyLimitCharacterLimit }
        ?.let {
            logWarning(INVALID_USER_CONSENT_KEY_LENGTH.format(configs.consentKeyLimitCharacterLimit))
            consent.copy(key = it.substring(0, configs.consentKeyLimitCharacterLimit))
        } ?: consent

    private fun logWarningIfPassedConsentCountLimit(
        currentConsentKeys: Set<String>,
        it: UserConsent
    ) {
        if (!isBelowConsentCountLimit(currentConsentKeys, it))
            logWarning(
                MAX_CONSENT_NUMBER_REACHED.format(currentConsentKeys.first(), configs.consentsLimit)
            )
    }

    private fun logInvalidKeyErrorIfRequired(userConsent: UserConsent?) {
        if (userConsent == null)
            InstabugSDKLogger.e(Constants.LOG_TAG, NULL_OR_EMPTY_USER_CONSENT_KEY)
    }

    private fun isBelowConsentCountLimit(
        currentConsentKeys: Set<String>,
        it: UserConsent
    ) = currentConsentKeys.size < configs.consentsLimit || currentConsentKeys.contains(it.key)

    private fun validateDescription(consent: UserConsent): UserConsent = consent.description
        ?.let {
            when {
                it.isBlank() -> consent.copy(description = null)
                it.length > configs.consentDescriptionChaLimit -> {
                    logWarning(INVALID_USER_CONSENT_DESCRIPTION_LENGTH.format(configs.consentDescriptionChaLimit))
                    consent.copy(description = it.substring(0, configs.consentDescriptionChaLimit))
                }

                else -> consent
            }
        } ?: consent

    private fun logWarning(message: String) = InstabugSDKLogger.w(Constants.LOG_TAG, message)

}