package com.instabug.bug.proactivereporting.ui

import android.content.Context
import android.content.Intent
import android.os.Build
import android.os.Bundle
import androidx.appcompat.app.AlertDialog
import com.instabug.bug.R
import com.instabug.bug.ReportingActivityLauncher
import com.instabug.library.Instabug
import com.instabug.library.core.ui.BaseFragmentActivity
import com.instabug.library.frustratingexperience.FrustratingExperienceType
import com.instabug.library.ui.custom.InstabugAlertDialog
import com.instabug.library.util.LocaleUtils
import com.instabug.library.util.StatusBarUtils

private const val FRUSTRATING_EXPERIENCE_TYPE_KEY = "frustrating_experience_type"
private const val FRUSTRATING_EXPERIENCE_ID_KEY = "frustrating_experience_id"

class ProactiveReportingDialogActivity :
    BaseFragmentActivity<ProactiveReportingDialogActivityPresenter>(),
    ProactiveReportingDialogActivityContract.View {

    companion object {

        fun getIntent(context: Context, type: @FrustratingExperienceType String, id: Long) =
            Intent(context, ProactiveReportingDialogActivity::class.java).apply {
                putExtra(FRUSTRATING_EXPERIENCE_TYPE_KEY, type)
                putExtra(FRUSTRATING_EXPERIENCE_ID_KEY, id)
            }
    }

    private var frustratingExperienceType: String? = null
    private var frustratingExperienceId: Long? = null

    private var dialog: AlertDialog? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
            overrideActivityTransition(OVERRIDE_TRANSITION_OPEN, R.anim.ib_core_anim_fade_in, R.anim.ib_core_anim_fade_out)
        } else {
            overridePendingTransition(R.anim.ib_core_anim_fade_in, R.anim.ib_core_anim_fade_out)
        }

        super.onCreate(savedInstanceState)
        StatusBarUtils.setStatusBarForDialog(this)

        presenter = ProactiveReportingDialogActivityPresenter(this)

        frustratingExperienceType = intent?.getStringExtra(FRUSTRATING_EXPERIENCE_TYPE_KEY)
        frustratingExperienceId = intent?.getLongExtra(FRUSTRATING_EXPERIENCE_ID_KEY, 0L)


        dialog = InstabugAlertDialog.Builder(this)
            .setTitle(getLocalizedString(presenter?.getTitleResId(frustratingExperienceType)))
            .setMessage(getLocalizedString(R.string.ib_frustrating_experience_dialog_message))
            .setCancellable(false)
            .setNegativeButton(getLocalizedString(R.string.ib_frustrating_experience_dialog_negative_btn)) { _, _ ->
                presenter?.onNegativeButtonClicked()
            }
            .setPositiveButton(getLocalizedString(R.string.ib_frustrating_experience_dialog_positive_btn)) { _, _ ->
                presenter?.onPositiveButtonClicked(frustratingExperienceType, frustratingExperienceId)
            }.show()
        presenter?.onDialogShown()

    }
    override fun onPause() {
        super.onPause()
        dialog?.takeIf { dialog ->
            dialog.isShowing
        }?.cancel()
    }

    override fun onResume() {
        super.onResume()
        dialog?.takeUnless { dialog ->
            dialog.isShowing
        }?.show()
    }

    override fun finishActivity() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
            overrideActivityTransition(OVERRIDE_TRANSITION_CLOSE, R.anim.ib_core_anim_fade_in, R.anim.ib_core_anim_fade_out)
        }
        finish()
    }

    override fun getLayout(): Int = R.layout.ibg_proactive_reporting_dialog

    override fun initViews() {
        //We don't need it
    }

    private fun getLocalizedString(stringResId: Int?): String =
        stringResId?.let {
            LocaleUtils.getLocaleStringResource(Instabug.getLocale(this), stringResId, this)
        } ?: ""

    override fun navigateToBugReportingActivity() {
        startActivity(ReportingActivityLauncher.newFrustratingExperienceIntent(this))
    }
}