package com.instabug.chat.annotation;

import android.annotation.SuppressLint;
import android.graphics.Canvas;
import android.graphics.Path;
import android.graphics.PointF;

import com.instabug.chat.annotation.shape.ArrowShape;
import com.instabug.chat.annotation.shape.Shape;

import java.io.Serializable;
import java.util.Stack;

public class MarkUpDrawable implements Serializable {

    protected volatile Shape shape;
    protected Shape firstShape;
    protected DirectionRectF bounds = new DirectionRectF();
    protected DirectionRectF lastBounds = new DirectionRectF();
    private Stack<DirectionRectF> translationsStack = new Stack<>();

    public MarkUpDrawable(Shape shape) {
        this.shape = shape;
        this.firstShape = shape;
    }

    public void setBounds(DirectionRectF bounds) {
        this.bounds = bounds;
        lastBounds.set(bounds);
    }

    public boolean isTouched(PointF touchedPoint) {
        if (lastBounds.isVisible()) {
            return shape.isTouched(touchedPoint, bounds);
        }
        return false;
    }

    protected void draw(Canvas canvas) {
        if (lastBounds.isVisible()) {
            canvas.save();
            shape.draw(canvas, bounds, lastBounds);
            canvas.restore();
        }
    }

    public float getWidth() {
        return bounds.width();
    }

    public float getHeight() {
        return bounds.height();
    }

    private void addLastPosition(DirectionRectF rect) {

        if (lastBounds != null) {

            translationsStack.push(new DirectionRectF(lastBounds));
        }
        lastBounds = rect;
    }

    public void onActionUp() {
        addLastPosition(new DirectionRectF(bounds));
    }

    public void setVisibility(boolean visibility) {
        DirectionRectF directionRectF = new DirectionRectF(bounds);
        directionRectF.setVisibility(visibility);
        addLastPosition(directionRectF);
    }

    /**
     * @return true when you can go back or false if you can not.
     */
    public boolean back() {
        if (translationsStack.size() > 0) {

            lastBounds = translationsStack.pop();

            if (translationsStack.size() == 0) {
                shape = firstShape;
            }
            shape.adjustBounds(lastBounds, bounds, true);

            return true;
        } else {
            return false;
        }
    }

    public void translate(float x, float y) {
        bounds.right += (int) x - bounds.left;
        bounds.bottom += (int) y - bounds.top;
        bounds.left = (int) x;
        bounds.top = (int) y;
    }

    public void translateBy(int x, int y) {
        shape.translateBy(bounds, lastBounds, x, y);
    }

    public Shape getShape() {
        return shape;
    }

    public void adjustBounds(DirectionRectF newBounds) {
        shape.adjustBounds(newBounds, bounds, false);
    }

    public void drawControlButtons(Canvas canvas, ControlButton... controlButtons) {
        shape.drawControlButtons(canvas, bounds, controlButtons);
    }

    public void drawBorder(Canvas canvas) {
        shape.drawBorder(canvas, bounds.getTopLeftPoint(), bounds.getTopRightPoint(),
                bounds.getBottomRightPoint(), bounds.getBottomLeftPoint());
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void setShape(Shape shape, DirectionRectF bounds) {
        addLastPosition(new DirectionRectF(bounds));
        this.shape = shape;
        if (shape instanceof ArrowShape) {
            this.bounds = bounds;
        }
    }

    public boolean isVisible() {
        return lastBounds.isVisible();
    }

    public Path getPathOfShape() {
        return shape.getPath(bounds);
    }

}
