package com.instabug.bug.screenshot;

import android.content.Context;
import android.net.Uri;

import androidx.annotation.Nullable;

import com.instabug.bug.BugPlugin;
import com.instabug.bug.Constants;
import com.instabug.bug.LiveBugManager;
import com.instabug.bug.ReportingActivityLauncher;
import com.instabug.bug.model.Bug;
import com.instabug.library.InstabugState;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.InstabugStateEventBus;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.model.Attachment;
import com.instabug.library.screenshot.ExtraScreenshotHelper;
import com.instabug.library.util.InstabugSDKLogger;

import java.lang.ref.WeakReference;

import io.reactivexport.functions.Consumer;

/**
 * Created by mohamedzakaria on 6/6/17.
 */

public class ScreenshotHelper implements ExtraScreenshotHelper.OnCaptureListener {

    private static ScreenshotHelper INSTANCE;
    @Nullable
    private WeakReference<Context> contextWeakReference;
    private final ExtraScreenshotHelper extraScreenshotHelper;

    private ScreenshotHelper() {
        extraScreenshotHelper = new ExtraScreenshotHelper();

        InstabugStateEventBus.getInstance().subscribe(new Consumer<InstabugState>() {
            @Override
            public void accept(InstabugState instabugState) {
                if (instabugState == InstabugState.DISABLED) {
                    cancel();
                }
            }
        });
    }

    public synchronized static ScreenshotHelper getInstance() {
        if (INSTANCE == null)
            INSTANCE = new ScreenshotHelper();
        return INSTANCE;
    }

    public synchronized void startScreenshotCapturing(Context context) {
        this.contextWeakReference = new WeakReference<>(context);
        extraScreenshotHelper.init(this);
    }

    public void release() {
        ScreenshotHelper helper = getInstance();
        helper = null;
    }

    private synchronized void cancel() {
        LiveBugManager.getInstance().removeBug();
        LiveBugManager.getInstance().releaseBug();
        extraScreenshotHelper.cancel();
        BugPlugin plugin = (BugPlugin) InstabugCore.getXPlugin(BugPlugin.class);
        if (plugin != null) {
            plugin.setState(Plugin.STATE_BACKGROUND);
        }
    }

    @Override
    public synchronized void onExtraScreenshotCaptured(Uri extraScreenshotUri) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Extra screenshot captured, Uri: " + extraScreenshotUri);
        extraScreenshotHelper.release();
        Bug bug = LiveBugManager.getInstance().getBug();
        if (bug != null) {
            bug.addAttachment(extraScreenshotUri, Attachment.Type.EXTRA_IMAGE);
            if (contextWeakReference != null) {
                Context context = contextWeakReference.get();
                if (context != null) {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "starting feedback activity");
                    startBugFeedbackActivity(context, extraScreenshotUri);
                }
            }
        } else {
            InstabugSDKLogger.w(Constants.LOG_TAG, "Bug has been released");
        }
    }

    @Override
    public synchronized void onExtraScreenshotError(Throwable throwable) {
        extraScreenshotHelper.release();
        if (contextWeakReference != null) {
            Context context = contextWeakReference.get();
            if (context != null) {
                startBugFeedbackActivity(context, null);
            }
        }
    }

    private void startBugFeedbackActivity(Context context, @Nullable Uri bitmap) {
        context.startActivity(ReportingActivityLauncher.newHangingBugIntent(context, bitmap));
    }
}