package com.instabug.bug.invocation.invoker;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.app.Activity;
import android.content.Context;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.LayerDrawable;
import android.graphics.drawable.ShapeDrawable;
import android.graphics.drawable.shapes.OvalShape;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.view.GestureDetector;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.ImageView;

import androidx.annotation.MainThread;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;

import com.instabug.bug.invocation.InvocationListener;
import com.instabug.bug.invocation.InvocationManager;
import com.instabug.library.invocation.util.InstabugFloatingButtonEdge;
import com.instabug.library.Constants;
import com.instabug.library.InstabugState;
import com.instabug.library.InstabugStateProvider;
import com.instabug.bug.R;
import com.instabug.library._InstabugActivity;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.threading.PoolProvider;

import java.lang.ref.WeakReference;
import java.util.concurrent.Callable;

public class FloatingButtonInvoker implements AbstractInvoker<Void>, View.OnClickListener {

    private static final int MIN_TOP_LOCATION = 50;
    private static final int LANDSCAPE_MODE = 2;
    private static final int NAVIGATION_BAR_SIZE = 48;
    @Nullable
    private FrameLayout.LayoutParams layoutParams;

    int x, y = 0;
    private int width, height = 0;
    int realWidth = 0, realHeight = 0;
    int orientation = 0;
    float densityFactor;

    private InvocationListener invocationListener;
    @Nullable
    private WeakReference<FloatingButtonFrameLayout> floatingButtonFrameLayoutWeakReference;
    @Nullable
    private WeakReference<FloatingButton> floatingButtonWeakReference;
    private int floatingButtonSize;

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public FloatingButtonInvoker(InvocationListener invocationListener) {
        this.invocationListener = invocationListener;
    }

    @Override
    public void listen() {
        final Activity currentActivity = InstabugInternalTrackingDelegate.getInstance()
                .getCurrentRealActivity();
        if (currentActivity != null && !(currentActivity instanceof _InstabugActivity)
                && !(currentActivity.getClass().getName().contains("PlayCoreDialogWrapperActivity"))) {
            PoolProvider.postMainThreadTask(new Runnable() {
                @Override
                public void run() {
                    showFAB(currentActivity);
                }
            });
        }
    }

    @Override
    public void handle(Void input) {

    }

    @Override
    public void sleep() {
        PoolProvider.postMainThreadTask(new Runnable() {
            @Override
            public void run() {
                hideFAB();
            }
        });
    }

    @Override
    public boolean isActive() {
        try {
            if (Looper.myLooper() == Looper.getMainLooper()) {
                return getIsActiveCallable().call();
            } else {
                return PoolProvider.submitMainThreadTask(getIsActiveCallable()).get();
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            InstabugSDKLogger.e(Constants.LOG_TAG, "InterruptedException happened while checking floating button visibility", e);
            return false;
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error happened while checking floating button visibility", e);
            return false;
        }
    }

    private Callable<Boolean> getIsActiveCallable() {
        return () -> {
            final Activity currentActivity = InstabugInternalTrackingDelegate.getInstance()
                    .getCurrentActivity();
            if (currentActivity != null) {
                Window window = currentActivity.getWindow();
                if (window != null) {
                    View view = window.findViewById(R.id.instabug_fab_container);
                    return view != null;
                }
            }
            return false;
        };
    }

    @MainThread
    private void showFAB(final Activity activity) {
        hideFAB();

        FloatingButtonFrameLayout floatingButtonFrameLayout = new FloatingButtonFrameLayout(activity);
        orientation = activity.getResources().getConfiguration().orientation;

        floatingButtonFrameLayout.setId(R.id.instabug_fab_container);
        int tWidth, tHeight;
        densityFactor = activity.getResources().getDisplayMetrics().density;
        DisplayMetrics metrics = new DisplayMetrics();

        //adding the old screen dimensions to temp variables
        tWidth = width;
        tHeight = height;

        height = activity.getResources().getDisplayMetrics().heightPixels;
        width = activity.getResources().getDisplayMetrics().widthPixels;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            activity.getWindowManager().getDefaultDisplay().getRealMetrics(metrics);
            realHeight = metrics.heightPixels;
            realWidth = metrics.widthPixels;
        }

        floatingButtonSize = (int) (56 * densityFactor);
        FloatingButton floatingButton = new FloatingButton(activity);

        ShapeDrawable border = new ShapeDrawable(new OvalShape());
        ShapeDrawable background = new ShapeDrawable(new OvalShape());

        background.getPaint().setColor(InstabugCore.getPrimaryColor());
        border.getPaint().setColor(InstabugCore.getPrimaryColor());

        Drawable[] layers = {background, border};
        LayerDrawable layerDrawable = new LayerDrawable(layers);

        layerDrawable.setLayerInset(0, 0, 0, 0, 0);
        layerDrawable.setLayerInset(1, 2, 2, 2, 2);

        floatingButton.setBackgroundDrawable(layerDrawable);
        Drawable instabugLogo = activity.getResources().getDrawable(R.drawable.ibg_core_ic_floating_btn);
        assert instabugLogo != null;
        floatingButton.setImageDrawable(instabugLogo);
        floatingButton.setScaleType(ImageView.ScaleType.CENTER);
        floatingButton.setContentDescription(
                getLocalizedString(activity, R.string.floating_button_invoker_content_description)
        );

        if (layoutParams == null) {
            if (InvocationManager.getInstance().getCurrentInvocationSettings().getFloatingButtonParams().edge == InstabugFloatingButtonEdge.LEFT) {
                layoutParams = new FrameLayout.LayoutParams(floatingButtonSize, floatingButtonSize, Gravity.TOP | Gravity.LEFT);
                floatingButton.setLayoutParams(layoutParams);
                floatingButton.setLocation(-10, InvocationManager.getInstance().getCurrentInvocationSettings().getFloatingButtonParams().offsetFromTop);
            } else {
                layoutParams = new FrameLayout.LayoutParams(floatingButtonSize, floatingButtonSize, Gravity.TOP | Gravity.RIGHT);
                floatingButton.setLayoutParams(layoutParams);
                floatingButton.setLocation(width + 10, InvocationManager.getInstance().getCurrentInvocationSettings().getFloatingButtonParams().offsetFromTop);
            }
        } else {
            float tempX, tempY;
            tempX = ((float) (x * width)) / (float) tWidth;
            tempY = ((float) (y * height)) / (float) tHeight;

            //adding new values according to the change in width and height
            //in the event of screen rotation
            x = Math.round(tempX);
            y = Math.round(tempY);

            //changing layout params in according to the new position
            layoutParams.leftMargin = x;
            layoutParams.rightMargin = width - x;
            layoutParams.topMargin = y;
            layoutParams.bottomMargin = height - y;

            floatingButton.setLayoutParams(layoutParams);

            //help the floating button go to the nearest wall
            floatingButton.goToWall();
        }


        if (floatingButton != null) {
            floatingButton.setOnClickListener(this);
            floatingButton.setScaleType(ImageView.ScaleType.CENTER_INSIDE);
            if (floatingButtonFrameLayout != null)
                floatingButtonFrameLayout.addView(floatingButton);
        }

        ((FrameLayout) activity.getWindow().getDecorView()).addView(floatingButtonFrameLayout, new ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT));
        floatingButtonFrameLayoutWeakReference = new WeakReference<>(floatingButtonFrameLayout);
        floatingButtonWeakReference = new WeakReference<>(floatingButton);
    }

    private String getLocalizedString(Context context, @StringRes int resourceId) {
        return LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(context),
                resourceId,
                context);
    }

    @MainThread
    private void hideFAB() {
        if (floatingButtonFrameLayoutWeakReference != null) {
            FloatingButtonFrameLayout floatingButtonFrameLayout = floatingButtonFrameLayoutWeakReference.get();
            if (floatingButtonFrameLayout != null) {
                floatingButtonFrameLayout.removeAllViews();
                floatingButtonWeakReference = null;
                if (floatingButtonFrameLayout.getParent() != null
                        && floatingButtonFrameLayout.getParent() instanceof ViewGroup) {
                    ViewGroup vg = (ViewGroup) (floatingButtonFrameLayout.getParent());
                    vg.removeView(floatingButtonFrameLayout);
                    floatingButtonFrameLayoutWeakReference = null;
                }
            }
        }
    }

    public void updateButtonLocation() {
        PoolProvider.postMainThreadTask(new Runnable() {
            @Override
            public void run() {
                if (InstabugStateProvider.getInstance().getState() == InstabugState.ENABLED
                        && InstabugCore.getRunningSession() != null
                ) {
                    sleep();
                    layoutParams = null;
                    listen();
                } else {
                    PoolProvider.postMainThreadTask(new Runnable() {
                        @Override
                        public void run() {
                            hideFAB();
                        }
                    });
                }
            }
        });
    }

    @Override
    public void onClick(View v) {
        hideFAB();
        PoolProvider.postIOTask(() -> {
            invocationListener.onInvocationRequested();
            InvocationManager.getInstance().setLastUsedInvoker(FloatingButtonInvoker.this);
        });
    }

    public Rect getButtonBounds() {
        if (floatingButtonWeakReference == null) {
            return new Rect();
        }
        FloatingButton floatingButton = floatingButtonWeakReference.get();
        if (floatingButton == null || floatingButton.lastXPose == 0 || floatingButton.lastYPose == 0) {
            return new Rect();
        }
        float left = floatingButton.lastXPose;
        float top = floatingButton.lastYPose;
        float right = floatingButton.lastXPose + floatingButton.getWidth();
        float bottom = floatingButton.lastYPose + floatingButton.getHeight();
        return new Rect((int) left, (int) top, (int) right, (int) bottom);
    }

    public static class FloatingButtonFrameLayout extends FrameLayout {

        public FloatingButtonFrameLayout(Context context) {
            super(context);
        }

        public FloatingButtonFrameLayout(Context context, AttributeSet attrs) {
            super(context, attrs);
        }

        public FloatingButtonFrameLayout(Context context, AttributeSet attrs, int defStyleAttr) {
            super(context, attrs, defStyleAttr);
        }

        @TargetApi(Build.VERSION_CODES.LOLLIPOP)
        public FloatingButtonFrameLayout(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes) {
            super(context, attrs, defStyleAttr, defStyleRes);
        }
    }

    @SuppressLint("AppCompatCustomView")
    public class FloatingButton extends ImageButton {

        private static final int TOUCH_TIME_THRESHOLD = 200;
        @Nullable
        private GestureDetector mGestureDetector;
        private boolean shouldFlingAway = true;
        @Nullable
        private MoveAnimator mAnimator;
        private long lastTouchDown;
        float lastXPose, lastYPose;
        private boolean isBeingDragged = false;


        public FloatingButton(Context context) {
            super(context);
            mGestureDetector = new GestureDetector(context, new FlingListener());
            mAnimator = new MoveAnimator();
            setId(R.id.instabug_floating_button);
        }

        public FloatingButton(Context context, AttributeSet attrs) {
            super(context, attrs);
        }

        public FloatingButton(Context context, AttributeSet attrs, int defStyleAttr) {
            super(context, attrs, defStyleAttr);
        }

        @TargetApi(Build.VERSION_CODES.LOLLIPOP)
        public FloatingButton(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes) {
            super(context, attrs, defStyleAttr, defStyleRes);
        }

        @Override
        public void setLayoutParams(ViewGroup.LayoutParams params) {
            layoutParams = ((FrameLayout.LayoutParams) params);
            super.setLayoutParams(params);
        }


        @Override
        public boolean onTouchEvent(MotionEvent event) {
            boolean fling = false;
            if (shouldFlingAway) {
                if (mGestureDetector != null) {
                    fling = mGestureDetector.onTouchEvent(event);
                }
            }
            if (fling) {
                goToWall();
            } else {
                float x = event.getRawX();
                float y = event.getRawY();
                int action = event.getAction();
                if (action == MotionEvent.ACTION_DOWN) {
                    lastTouchDown = System.currentTimeMillis();
                    if (mAnimator != null) {
                        mAnimator.stop();
                    }
                    isBeingDragged = true;
                } else if (action == MotionEvent.ACTION_UP) {
                    if (System.currentTimeMillis() - lastTouchDown < TOUCH_TIME_THRESHOLD) {
                        performClick();
                    }
                    isBeingDragged = false;
                    goToWall();
                } else if (action == MotionEvent.ACTION_MOVE) {
                    if (isBeingDragged) {
                        move(x - lastXPose, y - lastYPose);
                    }
                }
                lastXPose = x;
                lastYPose = y;
            }
            return true;
        }

        private void goToWall() {
            if (InvocationManager.getInstance().getCurrentInvocationSettings().getFloatingButtonParams().edge == InstabugFloatingButtonEdge.LEFT) {
                float nearestXWall = x >= width / 2f ? width - floatingButtonSize + 10 : -10;
                if (mAnimator != null) {
                    mAnimator.start(nearestXWall, y > (height - floatingButtonSize) ? height - (floatingButtonSize * 2) : y);
                }
            } else {
                float nearestXWall = x >= width / 2f ? width + 10 : -10 + floatingButtonSize;
                if (mAnimator != null) {
                    mAnimator.start(nearestXWall, y > (height - floatingButtonSize) ? height - (floatingButtonSize * 2) : y);
                }
            }
        }

        void setLocation(int pX, int pY) {
            x = pX;
            y = pY;

            if (layoutParams != null) {
                layoutParams.leftMargin = x;
                layoutParams.rightMargin = width - x;
                if (orientation == LANDSCAPE_MODE && realWidth > width) {
                    layoutParams.rightMargin = (int) (layoutParams.rightMargin + (NAVIGATION_BAR_SIZE * densityFactor));
                }
                layoutParams.topMargin = y;
                layoutParams.bottomMargin = height - y;
                setLayoutParams(layoutParams);
            }
        }


        void move(float deltaX, float deltaY) {
            if (y + deltaY > MIN_TOP_LOCATION) {
                setLocation((int) (x + deltaX), (int) (y + deltaY));
            }
            if (layoutParams != null && shouldFlingAway && !isBeingDragged && Math.abs(layoutParams.rightMargin) < 50
                    && Math.abs(layoutParams.topMargin - (getContext().getResources().getDisplayMetrics().heightPixels / 2)) < 250) {
                goToWall();
            }
        }

        private class MoveAnimator implements Runnable {
            private Handler handler = new Handler(Looper.getMainLooper());
            private float destinationX;
            private float destinationY;
            private long startingTime;

            private void start(float x, float y) {
                this.destinationX = x;
                this.destinationY = y;
                startingTime = System.currentTimeMillis();
                handler.post(this);
            }

            @Override
            public void run() {
                if (FloatingButton.this.getParent() != null) {
                    float progress = Math.min(1, (System.currentTimeMillis() - startingTime) / 400f);
                    float deltaX = (destinationX - x) * progress;
                    float deltaY = (destinationY - y) * progress;
                    setLocation((int) (x + deltaX), (int) (y + deltaY));
                    if (progress < 1) {
                        handler.post(this);
                    }
                }
            }

            private void stop() {
                handler.removeCallbacks(this);
            }
        }
    }

    static class FlingListener extends GestureDetector.SimpleOnGestureListener {
        private static final float FLING_THRESHOLD_VELOCITY = 90f;

        @Override
        public boolean onFling(@Nullable MotionEvent e1, @androidx.annotation.NonNull MotionEvent e2, float velocityX, float velocityY) {
            return (Math.abs(e2.getX() - (e1 != null ? e1.getX() : 0)) < FLING_THRESHOLD_VELOCITY && e2.getY() - (e1 != null ? e1.getY() : 0) > FLING_THRESHOLD_VELOCITY);
        }
    }

    public static class FloatingButtonParams {
        public InstabugFloatingButtonEdge edge = InstabugFloatingButtonEdge.RIGHT;
        public int offsetFromTop = 250;
    }

}
