package com.instabug.chat.ui.chats;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.drawable.Drawable;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.TextView;

import androidx.annotation.Nullable;
import androidx.annotation.StringRes;
import androidx.core.content.ContextCompat;
import androidx.core.view.AccessibilityDelegateCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;

import com.instabug.bug.R;
import com.instabug.chat.model.Attachment;
import com.instabug.chat.model.Chat;
import com.instabug.chat.model.Message;
import com.instabug.library.InstabugCustomTextPlaceHolder;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.model.AssetEntity;
import com.instabug.library.model.IBGTheme;
import com.instabug.library.ui.custom.CircularImageView;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.Colorizer;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.PlaceHolderUtils;
import com.instabug.library.util.ThemeApplier;
import com.instabug.library.util.threading.PoolProvider;

import java.util.Collections;
import java.util.List;

class ChatsAdapter extends BaseAdapter {
    private List<Chat> chats;
    @Nullable
    private final IBGTheme ibgTheme;

    ChatsAdapter(List<Chat> chats, @Nullable IBGTheme ibgTheme) {
        this.chats = chats;
        this.ibgTheme = ibgTheme;
    }

    @Override
    public int getCount() {
        return chats.size();
    }

    @Override
    public Chat getItem(int position) {
        return chats.get(position);
    }

    @Override
    public long getItemId(int position) {
        String id = getItem(position).getId();
        return id != null ? id.hashCode() : position;
    }

    @Override
    public View getView(final int position, View convertView, ViewGroup parent) {
        if (parent != null) {
            ViewHolder viewHolder;
            if (convertView == null) {
                convertView = LayoutInflater.from(parent.getContext())
                        .inflate(R.layout.instabug_conversation_list_item,
                                parent, false);
                viewHolder = new ViewHolder(convertView);
                convertView.setTag(viewHolder);
            } else {
                viewHolder = (ViewHolder) convertView.getTag();
            }

            bind(convertView.getContext(), viewHolder, getItem(position));

            // Handle Accessibility
            final View finalConvertView = convertView;
            ViewCompat.setAccessibilityDelegate(convertView, new AccessibilityDelegateCompat() {

                @Override
                public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                    super.onInitializeAccessibilityNodeInfo(host, info);
                    if (getItem(position).getTitle() != null) {
                        info.setContentDescription(String.format(
                                getLocalizedString(R.string.ibg_chat_conversation_with_name_content_description, finalConvertView.getContext()),
                                getItem(position).getTitle()));
                    } else {
                        info.setContentDescription(getLocalizedString(R.string.ibg_chat_conversation_content_description, finalConvertView.getContext()));
                    }
                }
            });
        }
        return convertView;
    }

    private String getLocalizedString(Context context, @StringRes int resourceId) {
        return LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(context),
                resourceId, context);
    }

    private void bind(final Context context, final ViewHolder viewHolder, final Chat item) {
        Collections.sort(item.getMessages(), new Message.Comparator());
        Message lastMessage = item.getLastMessage();
        if (lastMessage == null || lastMessage.getBody() == null
                || TextUtils.isEmpty(lastMessage.getBody().trim())
                || lastMessage.getBody().equals("null")) {
            if (lastMessage != null && lastMessage.getAttachments().size() > 0) {
                @Attachment.AttachmentType String lastMessageLastAttachmentType = lastMessage
                        .getAttachments()
                        .get(lastMessage.getAttachments().size() - 1).getType();
                if (lastMessageLastAttachmentType != null && viewHolder.messageSnippetTextView != null) {
                    switch (lastMessageLastAttachmentType) {
                        case Attachment.AttachmentType.TYPE_IMAGE_GALLERY:
                        case Attachment.AttachmentType.TYPE_IMAGE_SCREENSHOT:
                            viewHolder.messageSnippetTextView.setText(
                                    PlaceHolderUtils.getPlaceHolder(InstabugCustomTextPlaceHolder.Key.CHATS_TYPE_IMAGE
                                            , getLocalizedString(context, R.string.instabug_str_image)));
                            break;
                        case Attachment.AttachmentType.TYPE_AUDIO:
                            viewHolder.messageSnippetTextView.setText(
                                    PlaceHolderUtils.getPlaceHolder(InstabugCustomTextPlaceHolder.Key.CHATS_TYPE_AUDIO
                                            , getLocalizedString(context, R.string.instabug_str_audio)));
                            break;
                        case Attachment.AttachmentType.TYPE_VIDEO_GALLERY:
                        case Attachment.AttachmentType.TYPE_VIDEO_RECORD:
                        case Attachment.AttachmentType.TYPE_VIDEO:
                            viewHolder.messageSnippetTextView.setText(
                                    PlaceHolderUtils.getPlaceHolder(InstabugCustomTextPlaceHolder.Key.CHATS_TYPE_VIDEO
                                            , getLocalizedString(context, R.string.instabug_str_video)));
                            break;
                        default:
                            break;
                    }
                }
            }
        } else {
            if (viewHolder.messageSnippetTextView != null) {
                viewHolder.messageSnippetTextView.setText(lastMessage.getBody());
            }
        }

        String senderName = item.getSenderName();
        if (viewHolder.senderTextView != null) {
            if (senderName != null && !senderName.equals("") && !senderName.equals("null")
                    && lastMessage != null && !lastMessage.isInbound()) {
                viewHolder.senderTextView.setText(senderName);
            } else {
                viewHolder.senderTextView.setText(item.getTitle());
            }
        }

        if (viewHolder.messageTimeTextView != null) {
            viewHolder.messageTimeTextView.setText(InstabugDateFormatter
                    .formatConversationLastMessageDate(context, item.getLastMessageDate()));
        }
        if (item.getUnreadCount() != 0) {
            TypedValue typedValue = new TypedValue();
            Resources.Theme theme = context.getTheme();
            theme.resolveAttribute(com.instabug.library.R.attr
                    .instabug_unread_message_background_color, typedValue, true);
            Drawable drawable = ContextCompat.getDrawable(context,
                    com.instabug.library.R.drawable.ibg_core_bg_white_oval);
            if (drawable != null) {
                Drawable unreadMessagesCountBackground = Colorizer.getPrimaryColorTintedDrawable
                        (drawable);
                if (viewHolder.unreadMessagesCount != null) {
                    viewHolder.unreadMessagesCount.setBackgroundDrawable(unreadMessagesCountBackground);
                }
            }
            if (viewHolder.unreadMessagesCount != null) {
                viewHolder.unreadMessagesCount.setText(String.valueOf(item.getUnreadCount()));
                viewHolder.unreadMessagesCount.setVisibility(View.VISIBLE);
            }
        } else {
            if (viewHolder.unreadMessagesCount != null) {
                viewHolder.unreadMessagesCount.setVisibility(View.GONE);
            }
        }

        if (item.getSenderAvatarUrl() != null) {
            PoolProvider.postIOTask(() -> {
                if (item.getSenderAvatarUrl() != null) {
                    BitmapUtils.loadBitmapForAsset(context, item.getSenderAvatarUrl(), AssetEntity.AssetType.IMAGE,
                            new BitmapUtils.OnBitmapReady() {
                                @Override
                                public void onBitmapReady(@Nullable final Bitmap senderAvatarBitmap) {
                                    PoolProvider.postMainThreadTask(() -> {
                                        if (viewHolder.senderAvatar != null) {
                                            viewHolder.senderAvatar.setImageBitmap(senderAvatarBitmap);
                                        }
                                    });
                                }

                                @Override
                                public void onBitmapFailedToLoad() {

                                }
                            });
                }
            });
        } else {
            if (viewHolder.senderAvatar != null) {
                viewHolder.senderAvatar.setImageResource(R.drawable.ibg_core_ic_avatar);
            }
        }
        renderIBGTheme(viewHolder);
    }

    private void renderIBGTheme(final ViewHolder viewHolder) {
        ThemeApplier.applyPrimaryTextStyle(viewHolder.senderTextView, ibgTheme);
        ThemeApplier.applySecondaryTextStyle(viewHolder.messageSnippetTextView, ibgTheme);
        ThemeApplier.applySecondaryTextStyle(viewHolder.messageTimeTextView, ibgTheme);
    }

    public void setChats(List<Chat> chats) {
        this.chats = chats;
    }

    private static class ViewHolder {
        @Nullable
        private final TextView senderTextView;
        @Nullable
        private final CircularImageView senderAvatar;
        @Nullable
        private final TextView messageTimeTextView;
        @Nullable
        private final TextView messageSnippetTextView;
        @Nullable
        private final TextView unreadMessagesCount;

        ViewHolder(View view) {
            senderTextView = view.findViewById(R.id
                    .instabug_txt_message_sender);
            senderAvatar = view.findViewById(R.id
                    .instabug_message_sender_avatar);
            messageTimeTextView = view.findViewById(R.id
                    .instabug_txt_message_time);
            unreadMessagesCount = view.findViewById(R.id
                    .instabug_unread_messages_count);
            messageSnippetTextView = view.findViewById(R.id
                    .instabug_txt_message_snippet);
        }

    }


    public String getLocalizedString(@StringRes int resourceId, Context context) {
        return LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(context), resourceId,
                context);
    }
}
