package com.instabug.bug.invocation;

import static com.instabug.library.core.plugin.PromptOptionManager.getPluginByIdentifier;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.bug.invocation.invocationdialog.InstabugDialogItem;
import com.instabug.bug.invocation.invoker.AbstractInvoker;
import com.instabug.bug.invocation.invoker.FloatingButtonInvoker;
import com.instabug.bug.invocation.invoker.ScreenshotGestureInvoker;
import com.instabug.bug.invocation.invoker.ShakeInvoker;
import com.instabug.bug.invocation.invoker.ThreeFingerSwipeLeftInvoker;
import com.instabug.bug.invocation.invoker.TwoFingerSwipeLeftInvoker;
import com.instabug.bug.ui.promptoptions.PromptOptionsLauncher;
import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.core.InitialScreenshotHelper;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.plugin.PluginPromptOption;
import com.instabug.library.internal.video.MediaProjectionHelper;
import com.instabug.library.invocation.InvocationMode;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;


/**
 * A class to Handle invocations requests, those request might come directly from the invokers such as
 * {@link FloatingButtonInvoker}
 * {@link ShakeInvoker}
 * {@link ScreenshotGestureInvoker}
 * {@link TwoFingerSwipeLeftInvoker}
 * {@link ThreeFingerSwipeLeftInvoker}
 * or directly from {@link Instabug#show()}
 * or directly from any module#show()
 * <p>
 * <p>
 * DEPENDENCIES
 * ************
 * 1- {@link InstabugCore#getPluginsPromptOptions()}
 * 2- {@link InitialScreenshotHelper#captureScreenshot(InitialScreenshotHelper.InitialScreenshotCapturingListener)}
 * 3- {@link InstabugInternalTrackingDelegate} this to get the current activity to start the activity from.
 * 4- {@link PluginPromptOption}
 * 5- {@link SettingsManager#isInitialScreenShotAllowed()}
 * 6- {@link PromptOptionsLauncher} to launch prompt option
 **/
public class InvocationRequestListenerImp implements InvocationListener {
    /**
     * A callback from the {@link ScreenshotGestureInvoker}
     */
    @Override
    public void onInvocationRequested(@Nullable Uri screenshotUri) {
        handleInvocationRequested(screenshotUri);
    }

    /**
     * A callback from the {@link AbstractInvoker}
     */
    @Override
    public void onInvocationRequested() {
        handleInvocationRequested(null);
    }


    /**
     * A method to get the invocation mode based on each plugin's prompt options.
     *
     * @return invocation mode {@link InvocationMode}
     */
    @InvocationMode
    @VisibleForTesting
    int getInvocationMode() {
        ArrayList<PluginPromptOption> pluginsPromptOptions = InstabugCore.getPluginsPromptOptions();
        if (pluginsPromptOptions.size() > 1) {
            return InvocationMode.PROMPT_OPTIONS;
        }
        if (!pluginsPromptOptions.isEmpty()) {
            PluginPromptOption pluginPromptOption = pluginsPromptOptions.get(0);
            int promptOptionIdentifier = pluginPromptOption.getPromptOptionIdentifier();
            switch (promptOptionIdentifier) {
                case PluginPromptOption.PromptOptionIdentifier.ASK_QUESTION:
                    return InvocationMode.ASK_QUESTION;
                case PluginPromptOption.PromptOptionIdentifier.BUG_REPORT:
                    return InvocationMode.NEW_BUG;
                case PluginPromptOption.PromptOptionIdentifier.CHAT_LIST:
                    return InvocationMode.CHATS;
                case PluginPromptOption.PromptOptionIdentifier.FEEDBACK:
                    return InvocationMode.NEW_FEEDBACK;
                default:
                    return InvocationMode.UNDEFINED;
            }
        } else return InvocationMode.UNDEFINED;
    }

    /**
     * A method to check whether the initial screenshot is required or not
     *
     * @return true if required, false otherwise.
     */
    private boolean isInitialScreenshotIsRequired() {
        if (SettingsManager.getInstance().isScreenshotByMediaProjectionEnabled()) {
            Context context = Instabug.getApplicationContext();
            if (context != null) {
                return SettingsManager.isInitialScreenShotAllowed() && MediaProjectionHelper.INSTANCE.isMediaProjectionServiceAvailable(context);
            } else {
                return false;
            }
        } else {
            return SettingsManager.isInitialScreenShotAllowed();
        }
    }

    /**
     * A method to handle any requested invocation from the {@link AbstractInvoker}
     * first it get the available prompt options then capture screenshot if it's required
     * screenshot is required if for the below prompt option given that {@link SettingsManager#isInitialScreenShotAllowed() is trye}
     * 1- BugReport
     * 2- Feedback
     * 3- Ask a question
     * For feature request and chat list it's not required
     * launch the target option based on {@link InvocationMode}
     *
     * @param screenShotUri screenshot if provided else null
     */
    @SuppressLint("SwitchIntDef")
    @VisibleForTesting
    void handleInvocationRequested(@Nullable Uri screenShotUri) {

        if (InstabugCore.getRunningSession() == null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "handleInvocationRequested() called with: screenShotUri = [" + screenShotUri + "] but session is not started yet!");
            return;
        } else if (!InstabugCore.isForegroundNotBusy()) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "handleInvocationRequested() called with: screenShotUri = [" + screenShotUri + "] but SDK is Busy");
            return;
        }
        handleInvocationRequest(screenShotUri);
    }

    private void handleInvocationRequest(@Nullable Uri screenShotUri) {
        @InvocationMode int invocationMode = getInvocationMode();
        if (invocationMode == InvocationMode.CHATS) {
            invokeWithoutScreenshot(PluginPromptOption.PromptOptionIdentifier.CHAT_LIST);
            return;
        }

        if (screenShotUri == null && isInitialScreenshotIsRequired()) {
            switch (invocationMode) {
                case InvocationMode.PROMPT_OPTIONS:
                    startPreInvocationRunnable();
                    captureScreenshotThenLaunchPromptOptions();
                    break;

                case InvocationMode.ASK_QUESTION:
                case InvocationMode.NEW_FEEDBACK:
                case InvocationMode.NEW_BUG:
                    startPreInvocationRunnable();
                    captureScreenshotThenInvoke(InstabugCore.getPluginsPromptOptions().get(0));
                    break;
                case InvocationMode.UNDEFINED:
                    break;

            }
        } else {
            switch (invocationMode) {
                case InvocationMode.PROMPT_OPTIONS:
                    startPreInvocationRunnable();
                    launchPromptOptionActivity(screenShotUri);
                    break;

                case InvocationMode.ASK_QUESTION:
                case InvocationMode.NEW_FEEDBACK:
                case InvocationMode.NEW_BUG:
                    startPreInvocationRunnable();
                    invoke(screenShotUri, InstabugCore.getPluginsPromptOptions().get(0));
                    break;
                case InvocationMode.UNDEFINED:
                    break;
            }
        }
    }

    /**
     * A method to invoke feature request immediately without capturing a screenshot
     */
    private void invokeWithoutScreenshot(@PluginPromptOption.PromptOptionIdentifier int identifier) {
        PluginPromptOption promptItemByIdentifier =
                getPluginByIdentifier(identifier, false);
        if (promptItemByIdentifier != null) {
            invoke(null, promptItemByIdentifier);
        }
    }


    /**
     * A method to start any pre invocation runnables if found
     */
    private void startPreInvocationRunnable() {
        if (SettingsManager.getInstance().getOnInvokeCallback() != null) {
            SettingsManager.getInstance().getOnInvokeCallback().onInvoke();
        }
    }


    /**
     * A method to invoke the plugin
     *
     * @param screenshotUri screenshot if found
     * @param option        requested plugin option
     */
    @VisibleForTesting
    void invoke(@Nullable Uri screenshotUri, @NonNull PluginPromptOption option) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationRequestListenerImp#invoke] invoking...");
        Activity activity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        if (activity == null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationRequestListenerImp#invoke] CurrentActivity is null, returning...");
            return;
        }
        // map PluginPromptOption to InstabugDialogItem
        InstabugDialogItem item = PromptOptionsLauncher.getInstance().getItem(option, null);
        // launch prompt options if it has sub-items or reporting screen otherwise.
        if (item == null || item.getSubItems() == null || item.getSubItems().isEmpty()) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationRequestListenerImp#invoke] invoking directly");
            option.invoke(screenshotUri);
        } else {
            InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationRequestListenerImp#invoke] Launching prompt options");
            PromptOptionsLauncher.getInstance().launchPromptOptionsActivity(
                    activity, screenshotUri, option.getTitle(), item.getSubItems());
        }
    }

    /**
     * A method to capture screen shot then launch the prompt option regardless the screenshot is captured successfully or not
     */
    @VisibleForTesting
    void captureScreenshotThenLaunchPromptOptions() {

        InitialScreenshotHelper.captureScreenshot(new InitialScreenshotHelper.InitialScreenshotCapturingListener() {
            @Override
            public void onScreenshotCapturedSuccessfully(Uri screenshotUri) {
                launchPromptOptionActivity(screenshotUri);
            }

            @Override
            public void onScreenshotCapturingFailed(Throwable throwable) {
                launchPromptOptionActivity(null);
            }
        });

    }


    /**
     * A method to invoke given its mode directly this method usually get triggered after invoking
     * {@link Instabug#show()} or Module#show()
     * For feature request it will invoke immediately, others will check if screenshot is required
     * before invocation {@link SettingsManager#isInitialScreenShotAllowed()}
     *
     * @param invocationMode requested invocation mode
     */
    @SuppressLint("SwitchIntDef")
    @VisibleForTesting
    void invokeWithMode(@InvocationMode int invocationMode) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationRequestListenerImp#invokeWithMode] Invoking with mode: " + invocationMode);
        if (InstabugCore.getRunningSession() == null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "invokeWithMode() called but session is not started yet!");
            return;
        }

        PluginPromptOption pluginPromptOption = null;
        switch (invocationMode) {
            case InvocationMode.ASK_QUESTION:
                pluginPromptOption = getPluginByIdentifier(PluginPromptOption.PromptOptionIdentifier.ASK_QUESTION, false);
                break;
            case InvocationMode.NEW_BUG:
                pluginPromptOption = getPluginByIdentifier(PluginPromptOption.PromptOptionIdentifier.BUG_REPORT, false);
                break;
            case InvocationMode.NEW_FEEDBACK:
                pluginPromptOption = getPluginByIdentifier(PluginPromptOption.PromptOptionIdentifier.FEEDBACK, false);
                break;
            case InvocationMode.CHATS:
                invokeWithoutScreenshot(PluginPromptOption.PromptOptionIdentifier.CHAT_LIST);
                break;
            case InvocationMode.UNDEFINED:
                break;
        }
        if (pluginPromptOption == null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationRequestListenerImp#invokeWithMode] PluginPromptOption is null");
            return;
        }
        InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationRequestListenerImp#invokeWithMode] Proceeding with PluginPromptOption: " + pluginPromptOption.getPromptOptionIdentifier());
        boolean isInitialScreenshotRequired = isInitialScreenshotIsRequired();
        InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationRequestListenerImp#invokeWithMode] isInitialScreenshotRequired: " + isInitialScreenshotRequired);
        if (isInitialScreenshotIsRequired())
            captureScreenshotThenInvoke(pluginPromptOption);
        else invoke(null, pluginPromptOption);
    }


    /**
     * A method to capture screenshot then invoke the plugin option
     *
     * @param pluginPromptOption the option which will be invoked.
     */
    @VisibleForTesting
    void captureScreenshotThenInvoke(final PluginPromptOption pluginPromptOption) {

        InitialScreenshotHelper.captureScreenshot(new InitialScreenshotHelper.InitialScreenshotCapturingListener() {
            @Override
            public void onScreenshotCapturedSuccessfully(Uri screenshotUri) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationRequestListenerImp#capturingScreenshotThenInvoke] Screenshot captured successfully.");
                invoke(screenshotUri, pluginPromptOption);
            }

            @Override
            public void onScreenshotCapturingFailed(Throwable throwable) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationRequestListenerImp#capturingScreenshotThenInvoke] Screenshot capturing failed with throwable." + throwable.getMessage());
                invoke(null, pluginPromptOption);
            }
        });

    }


    /**
     * A method to launch prompt option dialog
     *
     * @param screenshotUri screenshot if available
     */
    @VisibleForTesting
    void launchPromptOptionActivity(@Nullable Uri screenshotUri) {
        Activity currentActivity = InstabugInternalTrackingDelegate.getInstance()
                .getCurrentActivity();
        if (currentActivity != null) {
            PromptOptionsLauncher.getInstance().launchPromptOptionsActivity(currentActivity, screenshotUri);
        }
    }


    /**
     * A method to force show the option without asking for feature permission internally,
     * <p>
     * This method was intentional implemented as a workaround until we remove deprecated chats api.
     *
     * @param invocationMode
     */
    void forceInvoke(@InvocationMode int invocationMode) {
        if (InstabugCore.getRunningSession() == null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "forceInvoke() called but session is not started yet!");
            return;
        }

        PluginPromptOption pluginPromptOption = null;
        switch (invocationMode) {
            case InvocationMode.ASK_QUESTION:
                pluginPromptOption = getPluginByIdentifier(PluginPromptOption.PromptOptionIdentifier.ASK_QUESTION,
                        true);
                break;
            case InvocationMode.NEW_BUG:
                pluginPromptOption = getPluginByIdentifier(PluginPromptOption.PromptOptionIdentifier.BUG_REPORT,
                        true);
                break;
            case InvocationMode.NEW_FEEDBACK:
                pluginPromptOption = getPluginByIdentifier(PluginPromptOption.PromptOptionIdentifier.FEEDBACK,
                        true);
                break;
            case InvocationMode.CHATS:
                invokeWithoutScreenshot(PluginPromptOption.PromptOptionIdentifier.CHAT_LIST);
                break;
            case InvocationMode.UNDEFINED:
                break;
        }
        if (pluginPromptOption != null) {
            if (isInitialScreenshotIsRequired())
                captureScreenshotThenInvoke(pluginPromptOption);
            else invoke(null, pluginPromptOption);
        }
    }
}
