package com.instabug.bug.cache;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.bug.model.Bug;
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues;

import org.json.JSONException;

import java.util.List;

/**
 * Bugs Module database helper class <br>
 * Supports old(V1) database which is plain DB through {@link BugReportsPlainDbHelperImpl}
 *     using the normal android {@link android.content.ContentValues}  <br>
 * And new(V2) <b>DEFAULT</b> database which is encrypted DB through {@link BugReportsDbHelperImpl}
 *     using Instabug's encrypt-able {@link IBGContentValues}<br>
 */

public interface BugReportsDbHelper<T> {
    long NOT_FOUND = -1L;

    /**
     * Inserting a bug report in the database
     *
     * @param bug instance of {@link com.instabug.bug.model.Bug}
     * @return the row ID of the newly inserted row, or -1 if an error occurred
     * @throws JSONException
     */
    long insert(Bug bug) throws JSONException;

    /**
     * Inserting a bug report in the database
     *
     * @param id            of an instance of {@link com.instabug.bug.model.Bug} to be updated
     * @param contentValues the content to update the bug record
     *                      type <T> could be of {@link IBGContentValues} for encrypt-able database,
     *                      or {@link android.content.ContentValues}  for plain text database.
     */
    void update(String id, T contentValues);


    void updateConnectionError(@Nullable String id, String connectionError);

    /**
     * Gets list of all cached bugs
     *
     * @return list of Bugs
     */
    @NonNull
    List<Bug> retrieve(Context context);

    /**
     * Deletes all cached bugs, drop the table
     */
    void deleteAll();


    /**
     * Drops the table for the target DB
     */
    void dropTable();

    /**
     * Deletes bugs by id
     *
     * @param bugID id of the target bug to be deleted
     */
    void delete(String bugID);


    boolean updateFrustratingExperienceExternalId(Long internalId, String externalId);

    List<Bug> retrieveProactiveReportingBugs(Context context);

    List<Bug> retrieveNormalBugs(Context context);
}
