package com.instabug.chat.ui.chats;


import android.annotation.SuppressLint;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ImageButton;
import android.widget.ListView;

import androidx.annotation.Nullable;

import com.instabug.bug.R;
import com.instabug.chat.model.Chat;
import com.instabug.chat.ui.ChatActivity;
import com.instabug.chat.ui.base.ToolbarFragment;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.PlaceHolderUtils;

import java.util.ArrayList;

import static com.instabug.library.InstabugCustomTextPlaceHolder.Key;

@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public class ChatsFragment extends ToolbarFragment<ChatsContract.Presenter> implements
        ChatsContract.View, AdapterView.OnItemClickListener {

    public static final String TAG = "chats_fragment";
    private static final String KEY_COMPOSE = "compose_key";
    private ChatsAdapter chatsAdapter;
    private ArrayList<Chat> chats;
    private Callbacks callbacks;

    public static ChatsFragment newInstance(boolean composeEnabled) {
        ChatsFragment chatsFragment = new ChatsFragment();
        Bundle bundle = new Bundle();
        bundle.putBoolean(KEY_COMPOSE, composeEnabled);
        chatsFragment.setArguments(bundle);
        return chatsFragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        setRetainInstance(true);
        if (getActivity() != null && getActivity() instanceof ChatActivity) {
            callbacks = (Callbacks) getActivity();
        }
        presenter = new ChatsPresenter(this);
        chats = new ArrayList<>();
    }

    @Override
    protected int getContentLayout() {
        return R.layout.instabug_fragment_chats;
    }

    @Override
    protected void initContentViews(View rootView, @Nullable Bundle savedInstanceState) {

        // hide top right button
        ImageButton toolbarRightButton = rootView.findViewById(R.id.instabug_btn_toolbar_right);
        if (toolbarRightButton != null) {
            toolbarRightButton.setVisibility(View.GONE);
        }
        // init chats listView
        ListView chatsListView = rootView.findViewById(R.id.instabug_lst_chats);
        if (chatsListView != null) {
            chatsListView.setOnItemClickListener(this);
            chatsAdapter = new ChatsAdapter(chats, getIBGTheme());
            chatsListView.setAdapter(chatsAdapter);
        }
        ImageButton closeConversationsToolbarLeftButton = rootView.findViewById(R.id.instabug_btn_toolbar_left);
        if (closeConversationsToolbarLeftButton != null) {
            // set a tag for the close button for testing purposes
            closeConversationsToolbarLeftButton.setTag(R.id.TAG_BTN_CLOSE, "instabug_btn_close");
            closeConversationsToolbarLeftButton.setContentDescription(getLocalizedString(R.string.ibg_chats_conversations_close_content_description));
        }
    }

    @Override
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    protected String getTitle() {
        return PlaceHolderUtils.getPlaceHolder(Key.CONVERSATIONS_LIST_TITLE, getLocalizedString(R.string
                .instabug_str_conversations));
    }

    @Override
    protected void onDoneButtonClicked() {

    }

    @Override
    public void onStart() {
        super.onStart();
        if (presenter != null) {
            presenter.start();
        }

        if (AccessibilityUtils.isTalkbackEnabled() && Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            new Handler(Looper.getMainLooper()).postDelayed(() -> {
                if (getView() != null) {
                    getView().announceForAccessibility(getLocalizedString(R.string.ibg_chats_conversations_screen_content_description));
                }
            }, 300);
        }
    }

    @Override
    public void onStop() {
        super.onStop();
        if (presenter != null) {
            presenter.stop();
        }
    }

    @Override
    public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
        if (callbacks != null) {
            callbacks.openChat(((Chat) parent.getItemAtPosition(position)).getId());
        }
    }


    @Override
    public void updateChats(ArrayList<Chat> chats) {
        this.chats = chats;
    }

    @Override
    public void showChats() {
        chatsAdapter.setChats(chats);
        chatsAdapter.notifyDataSetChanged();
    }

    @Override
    public boolean isChatsFragmentVisible() {
        if (getFragmentManager() != null) {
            return getFragmentManager().findFragmentById(R.id.instabug_fragment_container) instanceof
                    ChatsFragment;
        }
        return false;
    }

    public interface Callbacks {
        void openChat(String chatNumber);

    }
}
