package com.instabug.chat.ui.chat;

import static com.instabug.library.view.ViewUtils.convertDpToPx;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.ColorFilter;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.drawable.Drawable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.ProgressBar;
import android.widget.TextView;

import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.instabug.chat.Constants;
import com.instabug.bug.R;
import com.instabug.chat.model.FlatMessage;
import com.instabug.chat.model.MessageAction;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.media.AudioPlayer;
import com.instabug.library.internal.storage.cache.AssetsCacheManager;
import com.instabug.library.model.AssetEntity;
import com.instabug.library.model.IBGTheme;
import com.instabug.library.ui.custom.CircularImageView;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.Colorizer;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.OnVideoFrameReady;
import com.instabug.library.util.ThemeApplier;
import com.instabug.library.util.VideoManipulationUtils;
import com.instabug.library.util.threading.PoolProvider;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Created by vezikon on 8/15/16.
 */
public class MessagesListAdapter extends BaseAdapter {

    private final static String TAG = "MessagesListAdapter";

    private final static int ITEM_MESSAGE_INBOUND = 0;
    private final static int ITEM_MESSAGE_OUTBOUND = 1;
    private final static int ITEM_ATTACHMENT_INBOUND_IMG = 2;
    private final static int ITEM_ATTACHMENT_OUTBOUND_IMG = 3;
    private final static int ITEM_ATTACHMENT_INBOUND_VOICE = 4;
    private final static int ITEM_ATTACHMENT_OUTBOUND_VOICE = 5;
    private final static int ITEM_ATTACHMENT_INBOUND_VIDEO = 6;
    private final static int ITEM_ATTACHMENT_OUTBOUND_VIDEO = 7;
    private final static int ITEM_TYPE_COUNT = 8;
    private final AudioPlayer audioPlayer;
    private List<FlatMessage> flatMessages;
    private ColorFilter colorFilter;

    private Context context;
    private ListView listView;
    private OnMessageItemClickListener listener;
    private boolean isFirstImage = true;

    @Nullable
    private final IBGTheme ibgTheme;

    public MessagesListAdapter(
            List<FlatMessage> flatMessages,
            Context context,
            ListView listView,
            OnMessageItemClickListener listener,
            @Nullable IBGTheme ibgTheme
    ) {
        this.flatMessages = flatMessages;
        this.listView = listView;
        this.context = context;
        this.listener = listener;
        this.ibgTheme = ibgTheme;
        audioPlayer = new AudioPlayer();

        colorFilter = new PorterDuffColorFilter(InstabugCore.getPrimaryColor(), PorterDuff.Mode
                .SRC_IN);
    }

    @Override
    public int getViewTypeCount() {
        return ITEM_TYPE_COUNT;
    }

    @Override
    public int getCount() {
        return flatMessages.size();
    }

    @Override
    public int getItemViewType(int position) {
        FlatMessage flatMessage = getItem(position);
        if (flatMessage.getType() != null) {
            switch (flatMessage.getType()) {
                case MESSAGE:
                    return flatMessage.isInbound() ? ITEM_MESSAGE_INBOUND : ITEM_MESSAGE_OUTBOUND;
                case IMAGE:
                    return flatMessage.isInbound() ? ITEM_ATTACHMENT_INBOUND_IMG :
                            ITEM_ATTACHMENT_OUTBOUND_IMG;
                case AUDIO:
                    return flatMessage.isInbound() ? ITEM_ATTACHMENT_INBOUND_VOICE :
                            ITEM_ATTACHMENT_OUTBOUND_VOICE;
                case VIDEO:
                    return flatMessage.isInbound() ? ITEM_ATTACHMENT_INBOUND_VIDEO :
                            ITEM_ATTACHMENT_OUTBOUND_VIDEO;
                default:
                    return -1;
            }
        }
        return -1;
    }

    @Override
    public FlatMessage getItem(int position) {
        return flatMessages.get(position);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @Override
    public View getView(int position, View convertView, ViewGroup parent) {
        if (parent == null) {
            return convertView;
        }
        ViewHolder viewHolder;
        int type = getItemViewType(position);
        if (convertView == null) {
            int layoutResource = getLayoutId(type);
            convertView = LayoutInflater.from(parent.getContext()).inflate(layoutResource, parent, false);
            viewHolder = new ViewHolder(convertView);
            convertView.setTag(viewHolder);

        } else {
            viewHolder = (ViewHolder) convertView.getTag();
        }

        try {
            bind(viewHolder, getItem(position));
        } catch (Exception e) {
        }
        return convertView;
    }

    private static int getLayoutId(int type) {
        int layoutResource;

        switch (type) {
            case ITEM_MESSAGE_INBOUND:
                layoutResource = R.layout.instabug_message_list_item_me;
                break;
            case ITEM_MESSAGE_OUTBOUND:
                layoutResource = R.layout.instabug_message_list_item;
                break;
            case ITEM_ATTACHMENT_INBOUND_IMG:
                layoutResource = R.layout.instabug_message_list_item_img_me;
                break;
            case ITEM_ATTACHMENT_OUTBOUND_IMG:
                layoutResource = R.layout.instabug_message_list_item_img;
                break;
            case ITEM_ATTACHMENT_INBOUND_VOICE:
                layoutResource = R.layout.instabug_message_list_item_voice_me;
                break;
            case ITEM_ATTACHMENT_OUTBOUND_VOICE:
                layoutResource = R.layout.instabug_message_list_item_voice;
                break;
            case ITEM_ATTACHMENT_INBOUND_VIDEO:
                layoutResource = R.layout.instabug_message_list_item_video_me;
                break;
            case ITEM_ATTACHMENT_OUTBOUND_VIDEO:
                layoutResource = R.layout.instabug_message_list_item_video;
                break;
            default:
                layoutResource = R.layout.instabug_message_list_item_me;
                break;
        }
        return layoutResource;
    }

    private void bind(final ViewHolder viewHolder, FlatMessage flatMessage) {
        if (viewHolder == null) {
            return;
        }
        if (flatMessage.getType() != null) {
            switch (flatMessage.getType()) {
                case MESSAGE:
                    renderMessageItem(viewHolder, flatMessage);
                    break;
                case IMAGE:
                    renderImageItem(viewHolder, flatMessage);
                    break;
                case AUDIO:
                    renderAudioItem(viewHolder, flatMessage);
                    break;
                case VIDEO:
                    renderVideoItem(viewHolder, flatMessage);
                    break;
            }
        }
        ThemeApplier.applySecondaryTextStyle(viewHolder.messageTimeTextView, ibgTheme);
    }

    private void renderMessageItem(ViewHolder viewHolder, FlatMessage flatMessage) {
        if (flatMessage.isInbound()) {
            if (viewHolder.messageSnippetTextView != null) {
                ThemeApplier.setViewBackground(
                        viewHolder.messageSnippetTextView,
                        getInboundBubbleDrawable(viewHolder.messageSnippetTextView)
                );
            }
        } else {
            if (viewHolder.messageActionContainer != null) {
                viewHolder.messageActionContainer.removeAllViews();
            }
            if (flatMessage.hasActions()) {
                bindMessageActions(flatMessage, viewHolder);
            }
        }
        if (viewHolder.messageTimeTextView != null) {
            viewHolder.messageTimeTextView.setText(InstabugDateFormatter.formatMessageDate
                    (context, flatMessage.getMessagedAt()));
        }
        if (flatMessage.getBody() != null && viewHolder.messageSnippetTextView != null) {
            viewHolder.messageSnippetTextView.setText(flatMessage.getBody());
        }
        if (viewHolder.senderImageView != null && flatMessage.getSenderAvatarUrl() != null) {
            loadImage(flatMessage.getSenderAvatarUrl(), viewHolder.senderImageView, false);

        }
        ThemeApplier.applyPrimaryTextStyle(viewHolder.messageSnippetTextView, ibgTheme);
    }

    private Drawable getInboundBubbleDrawable(View view) {
        int color = InstabugCore.getPrimaryColor();
        return Colorizer.getTintedDrawable(color, view.getBackground());
    }

    private void renderImageItem(ViewHolder viewHolder, FlatMessage flatMessage) {
        if (flatMessage.isInbound() && viewHolder.messageImageView != null) {
            ThemeApplier.setViewBackground(
                    viewHolder.messageImageView,
                    getInboundBubbleDrawable(viewHolder.messageImageView)
            );
        }

        if (viewHolder.messageTimeTextView != null) {
            viewHolder.messageTimeTextView.setText(InstabugDateFormatter.formatMessageDate
                    (context, flatMessage.getMessagedAt()));
        }
        bindImageAttachments(flatMessage, viewHolder);
        if (viewHolder.senderImageView != null && flatMessage.getSenderAvatarUrl() !=
                null) {
            loadImage(flatMessage.getSenderAvatarUrl(), viewHolder.senderImageView, false);
        }
    }

    private void renderAudioItem(ViewHolder viewHolder, FlatMessage flatMessage) {
        if (flatMessage.isInbound()) {
            if (viewHolder.messageAudioAttachmentBackground != null) {
                ThemeApplier.setViewBackground(
                        viewHolder.messageAudioAttachmentBackground,
                        getInboundBubbleDrawable(viewHolder.messageAudioAttachmentBackground)
                );
            }

            if (viewHolder.messageAudioPlayImageView != null)
                viewHolder.messageAudioPlayImageView.setColorFilter(colorFilter);
        }
        if (viewHolder.messageTimeTextView != null) {
            viewHolder.messageTimeTextView.setText(InstabugDateFormatter.formatMessageDate
                    (context, flatMessage.getMessagedAt()));
        }
        bindVoiceAttachments(flatMessage, viewHolder);
        if (viewHolder.senderImageView != null && flatMessage.getSenderAvatarUrl() !=
                null) {
            loadImage(flatMessage.getSenderAvatarUrl(), viewHolder.senderImageView, false);
        }

    }

    private void renderVideoItem(ViewHolder viewHolder, FlatMessage flatMessage) {
        if (flatMessage.isInbound()) {
            if (viewHolder.messageVideoFrameHolderBackground != null) {
                ThemeApplier.setViewBackground(
                        viewHolder.messageVideoFrameHolderBackground,
                        getInboundBubbleDrawable(viewHolder.messageVideoFrameHolderBackground)
                );
            }

            if (viewHolder.messageVideoPlayImageView != null)
                viewHolder.messageVideoPlayImageView.setColorFilter(colorFilter);
        }
        final FlatMessage videoAttachment = flatMessage;
        if (viewHolder.messageTimeTextView != null) {
            viewHolder.messageTimeTextView
                    .setText(InstabugDateFormatter.formatMessageDate(context, flatMessage
                            .getMessagedAt()));
        }

        bindVideoAttachments(videoAttachment, viewHolder);

        if (viewHolder.senderImageView != null && videoAttachment.getUrl() != null) {
            loadImage(videoAttachment.getUrl(), viewHolder.senderImageView, false);
        }
    }

    private void bindMessageActions(FlatMessage flatMessage, ViewHolder viewHolder) {
        ArrayList<MessageAction> actions = flatMessage.getMessageActions();
        if (actions != null && actions.size() > 0) {
            for (int i = 0; i < actions.size(); i++) {
                final MessageAction action = actions.get(i);
                Button actionButton = new Button(context);
                actionButton.setLayoutParams(new ViewGroup.LayoutParams(ViewGroup.LayoutParams
                        .WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT));
                actionButton.setPadding(convertDpToPx(context, 8), 0, convertDpToPx(context, 8), 0);
                actionButton.setText(action.getTitle());
                actionButton.setTextColor(ContextCompat.getColor(context, android.R.color.white));
                actionButton.setBackgroundColor(InstabugCore.getPrimaryColor());
                actionButton.setMaxEms(30);
                actionButton.setMaxLines(1);
                actionButton.setId(i);
                actionButton.setOnClickListener(view -> listener.onMessageActionClicked(action.getUrl()));
                if (viewHolder.messageActionContainer != null) {
                    viewHolder.messageActionContainer.addView(actionButton);
                }
                ThemeApplier.applyCtaStyle(actionButton, ibgTheme);
            }
        }
    }

    private void bindImageAttachments(final FlatMessage flatMessage, final ViewHolder viewHolder) {
        if (flatMessage.getPath() != null && viewHolder.messageImageView != null) {
            BitmapUtils.loadBitmap(flatMessage.getPath(), viewHolder.messageImageView);
        } else if (flatMessage.getUrl() != null && viewHolder.messageImageView != null) {
            loadImage(flatMessage.getUrl(), viewHolder.messageImageView, true);
        }

        if (viewHolder.messageImageView != null)
            viewHolder.messageImageView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    if (listener != null) {
                        if (flatMessage.getPath() != null) {
                            listener.onImageAttachmentClicked(flatMessage.getPath());
                        } else if (flatMessage.getUrl() != null) {
                            listener.onImageAttachmentClicked(flatMessage.getUrl());
                        }
                    }
                }
            });
    }

    private void bindVoiceAttachments(final FlatMessage flatMessage, final ViewHolder viewHolder) {

        final String path;
        if (flatMessage.getUrl() != null) {
            path = flatMessage.getUrl();
        } else {
            path = flatMessage.getPath();
        }

        if (viewHolder.messageAudioAttachmentProgressBar != null
                && viewHolder.messageAudioAttachmentProgressBar.getVisibility() == View.VISIBLE)
            viewHolder.messageAudioAttachmentProgressBar.setVisibility(View.GONE);

        if (viewHolder.messageAudioPlayImageView != null && viewHolder.messageAudioPlayImageView.getVisibility() == View.GONE)
            viewHolder.messageAudioPlayImageView.setVisibility(View.VISIBLE);

        if (viewHolder.messageAudioAttachmentBackground != null) {
            viewHolder.messageAudioAttachmentBackground.setOnClickListener(view -> {
                if (flatMessage.getMode() == FlatMessage.Mode.NONE) {
                    audioPlayer.start(path);
                    flatMessage.setMode(FlatMessage.Mode.PLAYING);
                    if (viewHolder.messageAudioPlayImageView != null) {
                        viewHolder.messageAudioPlayImageView.setImageResource(R.drawable.ibg_chat_ic_pause);
                    }
                } else {
                    audioPlayer.pause();
                    flatMessage.setMode(FlatMessage.Mode.NONE);
                    if (viewHolder.messageAudioPlayImageView != null) {
                        viewHolder.messageAudioPlayImageView.setImageResource(R.drawable.ibg_chat_ic_play);
                    }
                }
            });
        }

        audioPlayer.addOnStopListener(new AudioPlayer.OnStopListener(path) {
            @Override
            public void onStop() {
                flatMessage.setMode(FlatMessage.Mode.NONE);
                if (viewHolder.messageAudioPlayImageView != null) {
                    viewHolder.messageAudioPlayImageView.setImageResource(R.drawable.ibg_chat_ic_play);
                }
            }
        });

    }

    private void bindVideoAttachments(final FlatMessage flatMessage, final ViewHolder viewHolder) {
        if (flatMessage.getPath() != null) {
            bindLocalVideoAttachment(flatMessage, viewHolder);
        } else {
            bindRemoteVideoAttachments(flatMessage, viewHolder);
        }
    }

    private void bindLocalVideoAttachment(final FlatMessage flatMessage, final ViewHolder
            viewHolder) {
        if (viewHolder.messageVideoAttachmentProgressBar != null) {
            viewHolder.messageVideoAttachmentProgressBar.setVisibility(View.GONE);
        }
        if (viewHolder.messageVideoPlayImageView != null) {
            viewHolder.messageVideoPlayImageView.setVisibility(View.VISIBLE);
        }

        if (viewHolder.messageVideoAttachmentBackground != null) {
            viewHolder.messageVideoAttachmentBackground.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    if (listener != null && flatMessage.getPath() != null) {
                        listener.onVideoAttachmentClicked(flatMessage.getPath());
                    }
                }
            });
        }
        if (flatMessage.getPath() != null) {
            VideoManipulationUtils.extractFirstVideoFrame(flatMessage
                    .getPath(), new OnVideoFrameReady() {
                @Override
                public void onReady(@Nullable Bitmap frame) {
                    if (frame != null && viewHolder.messageVideoFrameHolderBackground != null) {
                        viewHolder.messageVideoFrameHolderBackground.setImageBitmap(frame);
                    }
                }
            });
        }
    }

    private void bindRemoteVideoAttachments(final FlatMessage flatMessage, final ViewHolder
            viewHolder) {
        if (flatMessage.getUrl() != null) {
            PoolProvider.postIOTask(new Runnable() {
                @Override
                public void run() {
                    String url = flatMessage.getUrl();
                    if (url != null) {
                        AssetEntity assetEntity = AssetsCacheManager.createEmptyEntity(context, url,
                                AssetEntity.AssetType.VIDEO);
                        AssetsCacheManager.getAssetEntity(assetEntity, new AssetsCacheManager
                                .OnDownloadFinished() {
                            @Override
                            public void onSuccess(final AssetEntity assetEntity) {
                                InstabugSDKLogger.v(Constants.LOG_TAG, "Asset Entity download succeeded: ");
                                PoolProvider.postMainThreadTask(new Runnable() {
                                    @Override
                                    public void run() {
                                        if (viewHolder.messageVideoAttachmentProgressBar != null) {
                                            viewHolder.messageVideoAttachmentProgressBar.setVisibility(View.GONE);
                                        }
                                        if (viewHolder.messageVideoPlayImageView != null) {
                                            viewHolder.messageVideoPlayImageView.setVisibility(View.VISIBLE);
                                        }
                                        VideoManipulationUtils.extractFirstVideoFrame
                                                (assetEntity.getFile().getPath(), new OnVideoFrameReady() {
                                                    @Override
                                                    public void onReady(@Nullable Bitmap frame) {
                                                        if (frame != null && viewHolder.messageVideoFrameHolderBackground != null) {
                                                            viewHolder.messageVideoFrameHolderBackground.setImageBitmap(frame);
                                                        }
                                                    }
                                                });


                                        if (viewHolder.messageVideoAttachmentBackground != null)
                                            viewHolder.messageVideoAttachmentBackground.setOnClickListener(new View
                                                    .OnClickListener() {
                                                @Override
                                                public void onClick(View view) {
                                                    listener.onVideoAttachmentClicked(assetEntity.getFile().getPath());
                                                }
                                            });
                                    }
                                });
                            }

                            @Override
                            public void onFailed(Throwable error) {
                                InstabugSDKLogger.e(Constants.LOG_TAG, "Asset Entity downloading got error: " + error.getMessage());
                            }
                        });
                    }
                }
            });
        }

    }

    private void loadImage(final String url, final ImageView imageView, final boolean isAttachment) {
        PoolProvider.postIOTask(new Runnable() {
            @Override
            public void run() {
                BitmapUtils.loadBitmapForAsset(context, url, AssetEntity.AssetType.IMAGE,
                        new BitmapUtils.OnBitmapReady() {
                            @Override
                            public void onBitmapReady(@Nullable final Bitmap avatarBitmap) {
                                PoolProvider.postMainThreadTask(new Runnable() {
                                    @Override
                                    public void run() {
                                        imageView.setImageBitmap(avatarBitmap);
                                        if (isAttachment && isFirstImage) {
                                            listView.setSelection(getCount() - 1);
                                            isFirstImage = false;
                                        }
                                    }
                                });
                            }

                            @Override
                            public void onBitmapFailedToLoad() {

                            }
                        });
            }
        });
    }

    public List<FlatMessage> getFlatMessageItems() {
        return flatMessages;
    }

    public void setFlatMessageItems(List<FlatMessage> flatMessages) {
        Iterator<FlatMessage> flatMessageIterator = flatMessages.iterator();
        while (flatMessageIterator.hasNext()) {
            FlatMessage flatMessage = flatMessageIterator.next();
            if (flatMessage.getType() == null)
                flatMessageIterator.remove();
        }
        this.flatMessages = flatMessages;
    }

    public void onDestroyView() {
        audioPlayer.release();
    }

    public interface OnMessageItemClickListener {

        void onVideoAttachmentClicked(String videoPath);

        void onImageAttachmentClicked(String imgPath);

        void onMessageActionClicked(String url);
    }

    public static class ViewHolder {
        @Nullable
        public CircularImageView senderImageView;
        @Nullable
        public TextView messageTimeTextView;
        @Nullable
        public TextView messageSnippetTextView;

        @Nullable
        public LinearLayout receivedBubble;

        //Image
        @Nullable
        public ImageView messageImageView;

        //Audio
        @Nullable
        public FrameLayout messageAudioAttachmentBackground;
        @Nullable
        public ImageView messageAudioPlayImageView;
        @Nullable
        public ProgressBar messageAudioAttachmentProgressBar;

        //Video
        @Nullable
        public ImageView messageVideoPlayImageView;
        @Nullable
        public ImageView messageVideoFrameHolderBackground;
        @Nullable
        public FrameLayout messageVideoAttachmentBackground;
        @Nullable
        public ProgressBar messageVideoAttachmentProgressBar;

        //MessageAction
        @Nullable
        public LinearLayout messageActionContainer;

        public ViewHolder(View view) {
            senderImageView = (CircularImageView) view.findViewById(R.id
                    .instabug_img_message_sender);
            messageTimeTextView = (TextView) view.findViewById(R.id.instabug_txt_message_time);
            messageSnippetTextView = (TextView) view.findViewById(R.id.instabug_txt_message_body);

            messageImageView = (ImageView) view.findViewById(R.id.instabug_img_attachment);

            messageAudioPlayImageView = (ImageView) view.findViewById(R.id.instabug_btn_play_audio);
            messageAudioAttachmentBackground = (FrameLayout) view.findViewById(R.id
                    .instabug_audio_attachment);
            messageAudioAttachmentProgressBar = (ProgressBar) view.findViewById(R.id
                    .instabug_audio_attachment_progress_bar);

            messageVideoFrameHolderBackground = (ImageView) view.findViewById(R.id
                    .instabug_img_video_attachment);
            messageVideoPlayImageView = (ImageView) view.findViewById(R.id.instabug_btn_play_video);
            messageVideoAttachmentBackground = (FrameLayout) view.findViewById(R.id
                    .instabug_video_attachment);
            messageVideoAttachmentProgressBar = (ProgressBar) view.findViewById(R.id
                    .instabug_video_attachment_progress_bar);
            messageActionContainer = (LinearLayout) view.findViewById(R.id
                    .instabug_message_actions_container);

            receivedBubble = (LinearLayout) view.findViewById(R.id.bubble_received_container);
        }
    }
}
