package com.instabug.chat.cache;


import android.annotation.SuppressLint;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.chat.Constants;
import com.instabug.chat.model.Message;
import com.instabug.chat.model.ReadMessage;
import com.instabug.library.internal.storage.cache.Cache;
import com.instabug.library.internal.storage.cache.CacheManager;
import com.instabug.library.internal.storage.cache.InMemoryCache;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * @author vezikon .
 */
public class ReadQueueCacheManager {

    public static final String READ_QUEUE_DISK_CACHE_FILE_NAME = "/read_queue.cache";
    public static final String READ_QUEUE_DISK_CACHE_KEY = "read_queue_disk_cache_key";
    public static final String READ_QUEUE_MEMORY_CACHE_KEY = "read_queue_memory_cache_key";

    @Nullable
    private static ReadQueueCacheManager mReadQueueCacheManager;

    private ReadQueueCacheManager() {
        InMemoryCache<Integer, ReadMessage> readQueueInMemoryCache = new InMemoryCache<>
                (READ_QUEUE_MEMORY_CACHE_KEY);
        CacheManager.getInstance().addCache(readQueueInMemoryCache);
    }

    public static ReadQueueCacheManager getInstance() {
        if (mReadQueueCacheManager == null) {
            mReadQueueCacheManager = new ReadQueueCacheManager();
        }
        return mReadQueueCacheManager;
    }

    /**
     * Loads Read Messages cache from disk if it's not in memory
     *
     * @return in-memory cache for {@link com.instabug.chat.model.ReadMessage}
     */
    @Nullable
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public static InMemoryCache<String, ReadMessage> getCache() {
        if (!CacheManager.getInstance().cacheExists(ReadQueueCacheManager
                .READ_QUEUE_MEMORY_CACHE_KEY)) {

            CacheManager.getInstance().migrateCache(ReadQueueCacheManager
                    .READ_QUEUE_DISK_CACHE_KEY, ReadQueueCacheManager
                    .READ_QUEUE_MEMORY_CACHE_KEY, new CacheManager.KeyExtractor<String,
                    ReadMessage>() {
                @Override
                @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
                public String extractKey(ReadMessage value) {
                    return value.getChatNumber();
                }
            });
            Cache readQueueMemoryCache = CacheManager.getInstance().getCache
                    (READ_QUEUE_MEMORY_CACHE_KEY);
            if (readQueueMemoryCache != null) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "In-memory cache restored from " +
                        "disk, "
                        + readQueueMemoryCache.getValues().size() + " elements restored");
            }
        }
        return (InMemoryCache<String, ReadMessage>) CacheManager.getInstance().getCache
                (ReadQueueCacheManager.READ_QUEUE_MEMORY_CACHE_KEY);
    }

    /**
     * Saves all cached read messages from {@code ReadQueueCacheManager
     * .READ_QUEUE_MEMORY_CACHE_KEY} to disk cache
     */
    public static void saveCacheToDisk() {

        final Cache readQueueMemoryCache = CacheManager.getInstance().getCache
                (READ_QUEUE_MEMORY_CACHE_KEY);
        final Cache readQueueDiskCache = CacheManager.getInstance().getCache
                (READ_QUEUE_DISK_CACHE_KEY);
        if (readQueueMemoryCache == null || readQueueDiskCache == null) {
            return;
        }

        InstabugSDKLogger.d(Constants.LOG_TAG, "Saving In-memory cache to disk, no. of " +
                "items to save is "
                + readQueueMemoryCache.getValues());
        CacheManager.getInstance().migrateCache(readQueueMemoryCache
                , readQueueDiskCache, new CacheManager.KeyExtractor<String,
                        ReadMessage>() {
                    @Override
                    public String extractKey(ReadMessage value) {
                        return String.valueOf(value.getChatNumber());
                    }
                });
    }

    public void add(ReadMessage message) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Adding message to read queue in-memory cache");
        Cache cache =
                CacheManager.getInstance().getCache(READ_QUEUE_MEMORY_CACHE_KEY);
        if (cache != null && message != null && message.getChatNumber() != null) {
            cache.put(message.getChatNumber(), message);
            InstabugSDKLogger.v(Constants.LOG_TAG,
                    "Added message to read queue in-memory cache " + cache.size());
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public JSONArray getReadMessagesArray() {
        JSONArray jsonArray = new JSONArray();

        List<ReadMessage> readMessages = getAll();
        for (ReadMessage readMessage : readMessages) {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("chat_number", readMessage.getChatNumber());
                jsonObject.put("message_id", readMessage.getMessageId());
                jsonObject.put("read_at", readMessage.getReadAt());
                jsonArray.put(jsonObject);
            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e.getMessage() + " occurred while getting read messages");
            }
        }
        return jsonArray;
    }

    public List<ReadMessage> getAll() {
        Cache cache = CacheManager.getInstance().getCache(READ_QUEUE_MEMORY_CACHE_KEY);
        if (cache != null) {
            return cache.getValues();
        }
        return new ArrayList<>();

    }

    private void remove(String key) {
        Cache cache = CacheManager.getInstance().getCache(READ_QUEUE_MEMORY_CACHE_KEY);
        if (cache != null) {
            cache.delete(key);
        }
    }

    public void notify(List<ReadMessage> readMessages) {
        for (ReadMessage temp : getAll()) {
            for (ReadMessage readMessage : readMessages) {
                if (temp.getChatNumber() != null
                    && temp.getChatNumber().equals(readMessage.getChatNumber())
                    && temp.getMessageId() != null
                    && temp.getMessageId().equals(readMessage.getMessageId())
                    && readMessage.getChatNumber() != null) {
                    remove(readMessage.getChatNumber());
                }
            }
        }
    }

    public void markAsRead(@NonNull Message message) {
        ReadMessage readMessage = map(message);
        add(readMessage);
    }

    @VisibleForTesting
    ReadMessage map(@NonNull Message message) {
        ReadMessage readMessage = new ReadMessage();
        readMessage.setChatNumber(message.getChatId());
        readMessage.setMessageId(message.getId());
        readMessage.setReadAt(InstabugDateFormatter.getCurrentUTCTimeStampInSeconds());
        return readMessage;
    }

    @VisibleForTesting
    static void tearDown() {
        mReadQueueCacheManager = null;
    }
}
