package com.instabug.bug.screenshot.viewhierarchy.utilities;

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Rect;
import android.opengl.GLSurfaceView;
import android.text.InputType;
import android.text.method.TransformationMethod;
import android.view.TextureView;
import android.view.View;
import android.view.ViewGroup;
import android.webkit.WebView;
import android.widget.EditText;

import androidx.annotation.Nullable;

import com.instabug.bug.Constants;
import com.instabug.bug.screenshot.viewhierarchy.ViewHierarchy;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.instacapture.screenshot.ScreenshotTaker;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.Collection;
import java.util.IllegalFormatCodePointException;


public class BitmapUtils {

    public static ViewHierarchy captureViewHierarchy(final ViewHierarchy viewHierarchy) {
        return captureViewWithoutChildren(viewHierarchy);
    }

    private static ViewHierarchy captureViewWithoutChildren(ViewHierarchy viewHierarchy) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "staring capture viewHierarchy: " + viewHierarchy.getId());
        Bitmap bitmap;
        if (viewHierarchy.getView() instanceof ViewGroup) {
            boolean[] childrenVisibility = hideViewChildren((ViewGroup) viewHierarchy.getView());
            bitmap = captureView(viewHierarchy);
            showViewChildren((ViewGroup) viewHierarchy.getView(), childrenVisibility);
        } else {
            bitmap = captureView(viewHierarchy);
        }
        viewHierarchy.setImage(bitmap);
        InstabugSDKLogger.v(Constants.LOG_TAG, "capture viewHierarchy done successfully: " + viewHierarchy.getId());
        return viewHierarchy;
    }

    private static boolean[] hideViewChildren(ViewGroup parent) {
        boolean[] childrenVisibility = new boolean[parent.getChildCount()];
        for (int i = 0; i < parent.getChildCount(); i++) {
            if (parent.getChildAt(i).getVisibility() == View.VISIBLE) {
                childrenVisibility[i] = true;
                parent.getChildAt(i).setVisibility(View.INVISIBLE);
            } else {
                childrenVisibility[i] = false;
            }
        }
        return childrenVisibility;
    }

    private static void showViewChildren(ViewGroup parent, boolean[] childrenVisibility) {
        for (int i = 0; i < childrenVisibility.length; i++) {
            if (childrenVisibility[i]) {
                parent.getChildAt(i).setVisibility(View.VISIBLE);
            }
        }
    }

    @Nullable
    private static Bitmap captureView(ViewHierarchy viewHierarchy) {
        View view = viewHierarchy.getView();
        if (view != null && viewHierarchy.getVisibleRect() != null
                && view.getHeight() > 0
                && view.getWidth() > 0
                && viewHierarchy.getVisibleRect().height() > 0
                && viewHierarchy.getVisibleRect().width() > 0) {
            // capture view original bitmap
            Bitmap originalBitmap = captureView(view);

            // crop original bitmap to visible area
            if (originalBitmap != null && viewHierarchy.getOriginalRect() != null) {
                Bitmap croppedBitmap = cropImageToVisiblyArea(originalBitmap,
                        viewHierarchy.getVisibleRect(), viewHierarchy.getOriginalRect());
                if (croppedBitmap != null) {
                    // scale bitmap to half then return
                    return scaleBitmap(croppedBitmap, viewHierarchy.getScale());
                }
            }
        }
        return null;
    }

    @Nullable
    private static Bitmap captureView(@Nullable View view) {
        // Securing edit test
        Bitmap originalBitmap = null;
        if(view != null) {
            try {
                if (view instanceof EditText) {

                    TransformationMethod oldTransformation = ((EditText) view).getTransformationMethod();
                    int inputType = ((EditText) view).getInputType();
                    ((EditText) view).setInputType(InputType.TYPE_CLASS_TEXT);
                    ((EditText) view).setTransformationMethod(new AsteriskPasswordTransformationMethod());
                    view.buildDrawingCache();

                    // draw un secured EditText
                    originalBitmap = Bitmap.createBitmap(view.getDrawingCache());

                    ((EditText) view).setInputType(inputType);
                    ((EditText) view).setTransformationMethod(oldTransformation);

                } else {
                    originalBitmap = Bitmap.createBitmap(view.getWidth(),
                            view.getHeight(), Bitmap.Config.ARGB_8888);
                    final Canvas canvas = new Canvas(originalBitmap);
                    view.setDrawingCacheEnabled(true);
                    view.buildDrawingCache(true);
                    // draw bitmap
                    view.draw(canvas);
                    view.setDrawingCacheEnabled(false);
                    // draw special views
                    int[] viewLocationOnScreen = new int[2];
                    if (view instanceof TextureView) {
                        ScreenshotTaker.drawTextureView((TextureView) view, viewLocationOnScreen, canvas);
                    }
                    if (view instanceof GLSurfaceView) {
                        ScreenshotTaker.drawGLSurfaceView((GLSurfaceView) view, viewLocationOnScreen, canvas);
                    }
                    if (view instanceof WebView) {
                        ScreenshotTaker.drawWebView((WebView) view, canvas);
                    }

                    Collection<View> privateViews = SettingsManager.getInstance().getPrivateViews();
                    for (View privateView : privateViews) {
                        if (privateView != null && ScreenshotTaker.isVisible(privateView)
                                && privateView.getId() == view.getId()) {
                            canvas.drawColor(Color.BLACK);
                        }
                    }
                }
            } catch (Exception e) {
                //IllegalArgumentException is thrown when trying to draw HW accelerated view
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e.getMessage() + " occurred while capturing view");
            }
        }
        return originalBitmap;
    }

    @Nullable
    private static Bitmap cropImageToVisiblyArea(Bitmap bitmap, Rect visibleRect, Rect originalRect) throws IllegalFormatCodePointException {
        if (bitmap != null && visibleRect != null && visibleRect.width() > 0 && visibleRect.height() > 0) {
            int x = visibleRect.left - originalRect.left;
            int y = visibleRect.top - originalRect.top;
            if (x >= 0 && y >= 0 && visibleRect.width() <= bitmap.getWidth() && visibleRect.height() <= bitmap.getHeight()) {
                Bitmap croppedBitmap;
                try {
                    croppedBitmap = Bitmap.createBitmap(bitmap, x, y, visibleRect.width(), visibleRect.height());
                } catch (OutOfMemoryError outOfMemoryError) {
                    InstabugCore.reportError(outOfMemoryError, "Cropping bitmap throws an OOM");
                    croppedBitmap = null;
                }
                return croppedBitmap;
            } else {
                return null;
            }
        } else {
            return null;
        }
    }

    @Nullable
    private static Bitmap scaleBitmap(Bitmap bitmap, int scaleFactor) {
        if (bitmap != null && bitmap.getWidth() > scaleFactor && bitmap.getHeight() > scaleFactor) {
            return Bitmap.createScaledBitmap(bitmap, bitmap.getWidth() / scaleFactor,
                    bitmap.getHeight() / scaleFactor, false);
        } else {
            return null;
        }
    }
}
