package com.instabug.bug.screenrecording;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.bug.LiveBugManager;
import com.instabug.bug.ReportingActivityLauncher;
import com.instabug.bug.internal.video.InternalScreenRecordHelper;
import com.instabug.library.core.eventbus.ScreenRecordingEventBus;
import com.instabug.library.internal.video.ScreenRecordingContract;
import com.instabug.library.internal.video.ScreenRecordingEvent;
import com.instabug.library.model.Attachment;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;

import io.reactivexport.disposables.Disposable;
import io.reactivexport.functions.Consumer;

/**
 * Created by vezikon on 7/10/17.
 */

public class ExternalScreenRecordHelper implements ScreenRecordingContract {

    private static ExternalScreenRecordHelper INSTANCE;
    @Nullable
    private Disposable disposable;

    public static ExternalScreenRecordHelper getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new ExternalScreenRecordHelper();
        }

        return INSTANCE;
    }

    public void start() {
        InternalScreenRecordHelper.getInstance().init();

        if (disposable == null || disposable.isDisposed()) {
            disposable = ScreenRecordingEventBus.getInstance().subscribe(screenRecordEvent -> {
                if (screenRecordEvent.getStatus() == ScreenRecordingEvent.RECORDING_FINISHED) {
                    LiveBugManager.getInstance().setRecordingDuration(screenRecordEvent.getVideoDuration());
                } else if (screenRecordEvent.getStatus() == ScreenRecordingEvent.RECORDING_FILE_READY) {
                    backToBugReport(screenRecordEvent.getVideoUri());
                    clear();
                } else if (screenRecordEvent.getStatus() == ScreenRecordingEvent.RECORDING_PERMISSION_DENIED) {
                    InternalScreenRecordHelper.getInstance().release();
                    backToBugReport(screenRecordEvent.getVideoUri());
                    clear();
                } else if (screenRecordEvent.getStatus() == ScreenRecordingEvent.RECORDING_ERROR) {
                    InternalScreenRecordHelper.getInstance().release();
                    backToBugReport(null);
                    clear();
                } else if (screenRecordEvent.getStatus() == ScreenRecordingEvent.RECORDING_CANCELED) {
                    InternalScreenRecordHelper.getInstance().release();
                    clear();
                }
            });
        }
    }

    private void backToBugReport(@Nullable Uri videoUri) {
        addVideoAttachment(videoUri);
        openReport();
    }

    @VisibleForTesting
    private void addVideoAttachment(@Nullable Uri videoUri) {
        if (videoUri != null && LiveBugManager.getInstance().getBug() != null) {
            LiveBugManager.getInstance().getBug().addAttachment(videoUri,
                    Attachment.Type.EXTRA_VIDEO);
        }
    }

    private void openReport() {
        Activity activity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        if (activity != null) {
            Intent intent = ReportingActivityLauncher.newHangingBugIntent(activity
                    .getApplicationContext());
            activity.startActivity(intent);
        }
    }

    public boolean isRecording() {
        return InternalScreenRecordHelper.getInstance().isRecording();
    }

    @Nullable
    @Override
    public Uri getAutoScreenRecordingFileUri() {
        return InternalScreenRecordHelper.getInstance().getAutoScreenRecordingFileUri();
    }

    @Override
    public void delete() {
        InternalScreenRecordHelper.getInstance().delete();
    }

    @Override
    public void clear() {
        unsubscribe();
        InternalScreenRecordHelper.getInstance().clear();
    }

    private void unsubscribe() {
        if (disposable != null && !disposable.isDisposed()) {
            disposable.dispose();
        }
    }

    @Override
    public boolean isEnabled() {
        return InternalScreenRecordHelper.getInstance().isEnabled();
    }
}
