/*
 * Copyright (c) 2017 Yrom Wang <http://www.yrom.net>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.instabug.bug.internal.video.customencoding;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.media.MediaCodec;
import android.media.MediaFormat;
import android.os.Build;
import android.os.Looper;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Objects;

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
abstract class BaseEncoder implements Encoder {

    @Nullable
    private final String mCodecName;
    @Nullable
    private MediaCodec mEncoder;
    @Nullable
    private Callback mCallback;

    abstract static class Callback implements Encoder.Callback {
        void onInputBufferAvailable(BaseEncoder encoder, int index) {
        }

        void onOutputFormatChanged(BaseEncoder encoder, MediaFormat format) {
        }

        void onOutputBufferAvailable(BaseEncoder encoder, int index, MediaCodec.BufferInfo info) {
        }
    }

    BaseEncoder(@Nullable String codecName) {
        this.mCodecName = codecName;
    }

    @Override
    public void setCallback(Encoder.Callback callback) {
        if (!(callback instanceof Callback)) {
            throw new IllegalArgumentException("The passed callback parameter is not instance of BaseEncoder.Callback");
        }
        this.setCallback((Callback) callback);
    }

    void setCallback(Callback callback) {
        if (this.mEncoder != null) {
            throw new IllegalStateException("mEncoder is not null");
        }
        this.mCallback = callback;
    }

    /**
     * Must call in a worker handler thread!
     */
    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void prepare() throws IOException {
        if (Looper.myLooper() == null || Looper.myLooper() == Looper.getMainLooper()) {
            throw new IllegalStateException("should run in a HandlerThread");
        }
        if (mEncoder != null) {
            throw new IllegalStateException("prepared!");
        }
        MediaFormat format = createMediaFormat();

        String mimeType = format.getString(MediaFormat.KEY_MIME);
        final MediaCodec encoder = createEncoder(mimeType);
        try {
            if (this.mCallback != null) {
                // NOTE: MediaCodec maybe crash on some devices due to null callback
                encoder.setCallback(mCodecCallback);
            }
            encoder.configure(format, null, null, MediaCodec.CONFIGURE_FLAG_ENCODE);
            onEncoderConfigured(encoder);
            encoder.start();
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Configure codec failure!\n  with format" + format, e);
            throw e;
        }
        mEncoder = encoder;
    }

    /**
     * call immediately after {@link #getEncoder() MediaCodec}
     * configure with {@link #createMediaFormat() MediaFormat} success
     *
     * @param encoder
     */
    protected void onEncoderConfigured(MediaCodec encoder) {
    }

    /**
     * create a new instance of MediaCodec
     */
    private MediaCodec createEncoder(String type) throws IOException {
        try {
            // use codec name first
            if (this.mCodecName != null) {
                return MediaCodec.createByCodecName(mCodecName);
            }
        } catch (IOException | IllegalArgumentException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Create MediaCodec by name '" + mCodecName + "' failure! " + e.getMessage());
        }
        return MediaCodec.createEncoderByType(type);
    }

    /**
     * create {@link MediaFormat} for {@link MediaCodec}
     */
    protected abstract MediaFormat createMediaFormat();

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    protected final MediaCodec getEncoder() {
        return Objects.requireNonNull(mEncoder, "doesn't prepare()");
    }

    /**
     * @throws NullPointerException if prepare() not call
     * @see MediaCodec#getOutputBuffer(int)
     */
    @Nullable
    public final ByteBuffer getOutputBuffer(int index) {
        try {
            return getEncoder().getOutputBuffer(index);
        } catch (Exception exception) {
            InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Something went wrong while calling getOutputBuffer. " + exception.getMessage(),
                    exception
            );
        }
        return null;
    }

    /**
     * @throws NullPointerException if prepare() not call
     * @see MediaCodec#getInputBuffer(int)
     */
    @Nullable
    public final ByteBuffer getInputBuffer(int index) {
        try {
            return getEncoder().getInputBuffer(index);
        } catch (Exception exception) {
            InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Something went wrong while calling getInputBuffer. " + exception.getMessage(),
                    exception
            );
        }
        return null;
    }

    /**
     * @throws NullPointerException if prepare() not call
     * @see MediaCodec#queueInputBuffer(int, int, int, long, int)
     * @see MediaCodec#getInputBuffer(int)
     */
    public final void queueInputBuffer(int index, int offset, int size, long pstTs, int flags) {
        try {
            getEncoder().queueInputBuffer(index, offset, size, pstTs, flags);
        } catch (Exception exception) {
            InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Something went wrong while calling queueInputBuffer. " + exception.getMessage(),
                    exception
            );
        }
    }

    /**
     * @throws NullPointerException if prepare() not call
     * @see MediaCodec#releaseOutputBuffer(int, boolean)
     */
    public final void releaseOutputBuffer(int index) {
        try {
            getEncoder().releaseOutputBuffer(index, false);
        } catch (Exception exception) {
            InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Something went wrong while calling releaseOutputBuffer. " + exception.getMessage(),
                    exception
            );
        }
    }

    /**
     * @see MediaCodec#stop()
     */
    @Override
    public void stop() {
        if (mEncoder != null) {
            try {
                mEncoder.stop();
            } catch (IllegalStateException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while stopping the encoder. " + e.getMessage(), e);
            }
        }
    }

    /**
     * @see MediaCodec#release()
     */
    @Override
    public void release() {
        if (mEncoder != null) {
            mEncoder.release();
            mEncoder = null;
        }
    }

    /**
     * let media codec run async mode if mCallback != null
     */
    private final MediaCodec.Callback mCodecCallback = new MediaCodec.Callback() {
        @Override
        public void onInputBufferAvailable(@NonNull MediaCodec codec, int index) {
            if (mCallback != null) {
                mCallback.onInputBufferAvailable(BaseEncoder.this, index);
            }
        }

        @Override
        public void onOutputBufferAvailable(@NonNull MediaCodec codec, int index, @NonNull MediaCodec.BufferInfo info) {
            if (mCallback != null) {
                mCallback.onOutputBufferAvailable(BaseEncoder.this, index, info);
            }
        }

        @Override
        public void onError(@NonNull MediaCodec codec, @NonNull MediaCodec.CodecException e) {
            if (mCallback != null) {
                mCallback.onError(BaseEncoder.this, e);
            }
        }

        @Override
        public void onOutputFormatChanged(@NonNull MediaCodec codec, @NonNull MediaFormat format) {
            if (mCallback != null) {
                mCallback.onOutputFormatChanged(BaseEncoder.this, format);
            }
        }
    };


}
