package com.instabug.bug.internal.video;

import static com.instabug.library.util.NotificationUtils.cancelNotification;
import static com.instabug.library.util.NotificationUtils.createAndShowForegroundNotification;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Service;
import android.content.ComponentCallbacks2;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ServiceInfo;
import android.os.Build;
import android.os.IBinder;
import android.widget.Toast;

import androidx.annotation.Nullable;

import com.instabug.bug.internal.video.customencoding.ScreenRecorder;
import com.instabug.bug.utils.PermissionsUtils;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugState;
import com.instabug.library.R;
import com.instabug.library.core.eventbus.AutoScreenRecordingEventBus;
import com.instabug.library.core.eventbus.InstabugStateEventBus;
import com.instabug.library.core.eventbus.ScreenRecordingEventBus;
import com.instabug.library.core.eventbus.SessionStateEventBus;
import com.instabug.library.internal.InstabugMediaProjectionIntent;
import com.instabug.library.internal.video.ScreenRecordingEvent;
import com.instabug.library.internal.video.ScreenRecordingServiceAction;
import com.instabug.library.internal.video.ServiceUtils;
import com.instabug.library.model.session.SessionState;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import io.reactivexport.disposables.Disposable;
import io.reactivexport.functions.Consumer;

public class ScreenRecordingService extends Service implements ComponentCallbacks2 {

    static final int SCREEN_RECORDING_NOTIFICATION_ID = 8743;
    private static final String TAG = "ScreenRecordingService";
    private static final String EXTRA_RESULT_CODE = "result-code";
    private static final String EXTRA_DATA = "data";
    private static final String EXTRA_IS_MANUAL_SR = "is.manual.screen.recording";
    private Disposable sessionDisposable;
    private Disposable sdkStateDisposable;
    @Nullable
    private ScreenRecordingSession recordingSession;
    private boolean isManualRecording;
    private final ScreenRecordingSession.Listener listener = new ScreenRecordingSession.Listener() {
        @Override
        public void onStart() {

        }

        @Override
        public void onStop() {

        }

        @Override
        public void onError() {
            if (isManualRecording) {
                InternalScreenRecordHelper.getInstance().onRecordingError();
            }
        }

        @Override
        public void onEnd() {
            stopForeground(true);
            stopSelf();
        }
    };

    private final ScreenRecorder.Callback startRecordingCallBack = new ScreenRecorder.Callback() {

        @Override
        public void onStop(Throwable error) {
            InstabugSDKLogger.e(TAG, "Error while starting screen recorder", error);
            if (recordingSession != null) {
                recordingSession.deleteVideoFile();
            }
            if (isManualRecording) {
                InternalScreenRecordHelper.getInstance().onRecordingFinished();
                PoolProvider.postMainThreadTask(() -> {
                    InternalScreenRecordHelper.getInstance().release();
                    Toast.makeText(ScreenRecordingService.this, R.string.instabug_str_recording_video_error, Toast.LENGTH_SHORT).show();
                });
            }
            listener.onEnd();
            stopForeground(true);
            stopSelf();
        }

        @Override
        public void onStart() {

        }

        @Override
        public void onRecording(long presentationTimeUs) {

        }
    };

    @Nullable
    private Disposable manualScreenRecordingDisposable;
    @Nullable
    private Disposable autoScreenRecordingDisposable;

    public ScreenRecordingService() {
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static Intent newIntent(Context context, int resultCode, @Nullable Intent data, boolean isManualRecording) {
        Intent intent = new Intent(context, ScreenRecordingService.class);
        intent.putExtra(EXTRA_RESULT_CODE, resultCode);
        intent.putExtra(EXTRA_IS_MANUAL_SR, isManualRecording);
        intent.putExtra(EXTRA_DATA, data);
        return intent;
    }

    public static void startScreenRecordingService(Context context, Intent intent) {
        if (android.os.Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            context.startService(intent);
        } else if (PermissionsUtils.isPermissionGranted(context, Manifest.permission.FOREGROUND_SERVICE) && InstabugMediaProjectionIntent.isMediaProjectionIntentReadyToUse()) {
            context.startForegroundService(intent);
        } else {
            ScreenRecordingEventBus.getInstance().post(new ScreenRecordingEvent(ScreenRecordingEvent.RECORDING_ERROR, null));
        }
    }

    private void subscribeToSessionEvents() {
        if (sessionDisposable == null || sessionDisposable.isDisposed()) {
            sessionDisposable = SessionStateEventBus.getInstance()
                    .subscribe(sessionState -> {
                        if (Instabug.getApplicationContext() != null) {
                            if (sessionState == SessionState.FINISH
                                    && ServiceUtils.isServiceRunningInForeground(
                                    getApplicationContext(), ScreenRecordingService.class)) {
                                cancelManualScreenRecording();
                            }
                        }
                    });
        }
    }

    @Override
    public void onCreate() {
        super.onCreate();

        subscribeToSessionEvents();
        subscribeToSdkStateEvents();
        InstabugMediaProjectionIntent.setMediaProjectionIntentUsed(true);

        // creating notification for a mediaprojection service is mandatory in android 10
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            createAndShowForegroundNotification(
                    this,
                    R.string.ibg_screen_recording_notification_title,
                    SCREEN_RECORDING_NOTIFICATION_ID,
                    ServiceInfo.FOREGROUND_SERVICE_TYPE_MEDIA_PROJECTION);
        }
    }

    private void subscribeToSdkStateEvents() {
        sdkStateDisposable = InstabugStateEventBus.getInstance().subscribe(new Consumer<InstabugState>() {
            @Override
            public void accept(InstabugState instabugState) {
                ScreenRecordingServiceAction.CustomeActions action = null;
                if (instabugState == InstabugState.DISABLED) {
                    action = ScreenRecordingServiceAction.CustomeActions.STOP_DELETE;
                } else if (instabugState == InstabugState.INVOKED) {
                    action = ScreenRecordingServiceAction.CustomeActions.STOP_TRIM_KEEP;
                }
                if (action != null) {
                    AutoScreenRecordingEventBus.getInstance().post(action);
                }
            }
        });
    }

    @Override
    @SuppressLint("STRICT_MODE_VIOLATION")
    public int onStartCommand(Intent intent, int flags, int startId) {

        if (intent == null) {
            stopForeground(true);
            stopSelf();
        } else {
            final int resultCode = intent.getIntExtra(EXTRA_RESULT_CODE, 0);
            final Intent data = intent.getParcelableExtra(EXTRA_DATA);
            if (resultCode == 0 || data == null) {
                InstabugSDKLogger.w("ScreenRecordingService", "Can't start service. "
                        + "Result code: " + resultCode + ", Data: " + data);
                Toast.makeText(this, R.string.feature_requests_error_state_title, Toast.LENGTH_SHORT).show();
                ScreenRecordingEventBus.getInstance().post(new ScreenRecordingEvent(ScreenRecordingEvent.RECORDING_ERROR, null));
                stopForeground(true);
                stopSelf();
            }

            isManualRecording = intent.getBooleanExtra(EXTRA_IS_MANUAL_SR, true);

            registerEventBus();

            if (!SettingsManager.getInstance().isScreenCurrentlyRecorded() && Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                if (data != null) {
                    recordingSession = new ScreenRecordingSession(Instabug.getApplicationContext(), listener, startRecordingCallBack, resultCode, data);
                    SettingsManager.getInstance().setScreenCurrentlyRecorded(true);
                }
            } else {
                //terminate service
                stopForeground(true);
                stopSelf();
            }
        }
        return super.onStartCommand(intent, flags, startId);
    }


    private void registerEventBus() {
        if (isManualRecording) {
            registerManualScreenRecordingEventBus();
        } else {
            registerAutoScreenRecordingEventBus();
        }
    }

    private void registerManualScreenRecordingEventBus() {
        //Register manual screen recording event subscriber

        if (manualScreenRecordingDisposable == null
                || manualScreenRecordingDisposable.isDisposed()) {
            manualScreenRecordingDisposable
                    = ScreenRecordingEventBus.getInstance().subscribe(
                    event -> {
                        if (event.getStatus() == ScreenRecordingEvent.RECORDING_CANCELED) {
                            cancelManualScreenRecording();
                        } else if (SettingsManager.getInstance().isScreenCurrentlyRecorded()
                                || event.getStatus() == ScreenRecordingEvent.RECORDING_FINISHED) {
                            SettingsManager.getInstance().setScreenCurrentlyRecorded(false);

                            if (recordingSession != null) {
                                recordingSession.destroy(new ScreenRecorder.Callback() {
                                    @Override
                                    public void onStop(Throwable error) {
                                        if (event.getStatus() == ScreenRecordingEvent.RECORDING_FINISHED) {
                                            if (recordingSession != null) {
                                                recordingSession.hold();
                                            }
                                        }
                                    }

                                    @Override
                                    public void onStart() {

                                    }

                                    @Override
                                    public void onRecording(long presentationTimeUs) {

                                    }
                                });
                            }
                        }
                    });
        }

    }

    private void registerAutoScreenRecordingEventBus() {
        if (autoScreenRecordingDisposable == null || autoScreenRecordingDisposable.isDisposed()) {
            //Register auto screen recording event subscriber
            autoScreenRecordingDisposable
                    = AutoScreenRecordingEventBus.getInstance().subscribe(action -> PoolProvider.postIOTask(() -> {
                if (SettingsManager.getInstance().isScreenCurrentlyRecorded()) {
                    SettingsManager.getInstance().setScreenCurrentlyRecorded(false);
                    if (recordingSession != null) {
                        recordingSession.destroy(new ScreenRecorder.Callback() {
                            @Override
                            public void onStop(Throwable error) {
                                if (recordingSession == null) {
                                    stopForeground(true);
                                    stopSelf();
                                    return;
                                }

                                switch (action) {
                                    case STOP_DELETE:
                                        recordingSession.deleteVideoFile();
                                        break;
                                    case STOP_KEEP:
                                        recordingSession.hold();
                                        break;
                                    case STOP_TRIM_KEEP:
                                        recordingSession.trim(SettingsManager.getInstance()
                                                .autoScreenRecordingMaxDuration());
                                        break;
                                    default:
                                        break;
                                }
                            }

                            @Override
                            public void onStart() {

                            }

                            @Override
                            public void onRecording(long presentationTimeUs) {

                            }
                        });
                    }
                }
            }));
        }
    }

    @Override
    public void onTrimMemory(int level) {
        super.onTrimMemory(level);
        if (level == ComponentCallbacks2.TRIM_MEMORY_UI_HIDDEN && isManualRecording) {
            cancelManualScreenRecording();
        }
    }


    private void cancelManualScreenRecording() {
        if (SettingsManager.getInstance().isScreenCurrentlyRecorded()) {
            SettingsManager.getInstance().setScreenCurrentlyRecorded(false);
            InternalScreenRecordHelper.getInstance().cancel();

            if (recordingSession != null) {
                recordingSession.destroy(new ScreenRecorder.Callback() {
                    @Override
                    public void onStop(Throwable error) {
                        if (recordingSession != null) {
                            recordingSession.deleteVideoFile();
                        }
                        stopForeground(true);
                        stopSelf();
                    }

                    @Override
                    public void onStart() {

                    }

                    @Override
                    public void onRecording(long presentationTimeUs) {

                    }
                });
            }
        }
    }

    @Override
    public void onDestroy() {
        if (SettingsManager.getInstance().isScreenCurrentlyRecorded()) {
            SettingsManager.getInstance().setScreenCurrentlyRecorded(false);
        }

        super.onDestroy();

        if (manualScreenRecordingDisposable != null
                && !manualScreenRecordingDisposable.isDisposed()) {
            manualScreenRecordingDisposable.dispose();
        }

        if (autoScreenRecordingDisposable != null
                && !autoScreenRecordingDisposable.isDisposed()) {
            autoScreenRecordingDisposable.dispose();
        }

        unSubscribeToSessionEvents();
        unSubscribeToSdkStateEvents();

        cancelNotification(this, SCREEN_RECORDING_NOTIFICATION_ID);
    }

    private void unSubscribeToSdkStateEvents() {
        if (sdkStateDisposable != null && !sdkStateDisposable.isDisposed()) {
            sdkStateDisposable.dispose();
        }
    }

    private void unSubscribeToSessionEvents() {
        if (!sessionDisposable.isDisposed()) {
            sessionDisposable.dispose();
        }
    }

    @Override
    public IBinder onBind(Intent intent) {
        throw new UnsupportedOperationException("Not yet implemented");
    }

}


