package com.instabug.chat.synchronization;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.chat.Constants;
import com.instabug.chat.cache.ChatsCacheManager;
import com.instabug.chat.model.Attachment;
import com.instabug.chat.model.Chat;
import com.instabug.chat.model.Message;
import com.instabug.chat.model.MessageAction;
import com.instabug.chat.settings.ChatSettings;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.storage.cache.InMemoryCache;
import com.instabug.library.user.UserManagerWrapper;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;


/**
 * Receiver for new messages received from message pulling
 *
 * @author mSobhy
 */
public class NewMessagesHandler {

    /**
     * Instance of {@code MessageReceiver}
     */
    private static NewMessagesHandler INSTANCE;

    /**
     * List of all interested parties to be notified of new messages being received
     */
    private List<OnNewMessagesReceivedListener> onNewMessagesReceivedListeners = new ArrayList<>();

    /**
     * @return an instance of {@code MessageReceiver}
     */
    public static NewMessagesHandler getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new NewMessagesHandler();
        }
        return INSTANCE;
    }

    /**
     * Handles distributing {@code JSONArray} new messages to
     * {@link #onNewMessagesReceivedListeners}
     *
     * @param newMessagesJsonArray new messages received from pulling
     */
    public void handleNewMessagesReceived(Context context, boolean shouldInvalidateCache,
                                          JSONObject... newMessagesJsonArray) {
        List<Message> newMessages = parseNewMessages(newMessagesJsonArray);
        List<Message> shouldBeNotifiedMessages = getShouldBeNotifiedMessages(newMessages);

        if (shouldInvalidateCache) {
            invalidateThenUpdateChatsCache(context, newMessages);
        } else {
            updateChatsCache(context, newMessages);
        }

        if (shouldBeNotifiedMessages.size() > 0)
            fireNewMessagesRunnable();

        if (onNewMessagesReceivedListeners.size() > 0) {
            notifyAllListeners(shouldBeNotifiedMessages);
        } else {
            //todo why throwing exception?
            throw new IllegalStateException("No one is listening for unread messages");
        }
    }

    /**
     * Updates Cache with the newly received messages
     *
     * @param messages to be added to Cache
     * @see com.instabug.library.internal.storage.cache.CacheManager
     */
    private void updateChatsCache(Context context, List<Message> messages) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "updating chats cache new messages count: " + messages.size());

        for (Message message : messages) {
            if (isRemoteMessage(message)) {
                addRemoteMessage(message);
            } else if (isLocalMessageAndReadyToBeSynced(message)) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Message with id:" + message.getId() + " is ready to be synced");
                try {
                    ChatsCacheManager.updateLocalMessageWithSyncedMessage(context, message);
                } catch (IOException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to update local message with synced message, " + e.getMessage(), e);
                }
            }
        }
    }

    /**
     * Invalidate Cache then add the newly received messages
     *
     * @param messages to be added to Cache
     * @see com.instabug.library.internal.storage.cache.CacheManager
     */
    private void invalidateThenUpdateChatsCache(Context context, List<Message> messages) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "START Invalidate Cache");
        final List<Message> notSentMessages = ChatsCacheManager.getNotSentMessages();

        InMemoryCache<String, Chat> chatsCache = ChatsCacheManager.getCache();
        if (chatsCache != null) {
            chatsCache.invalidate();
        }

        InstabugSDKLogger.v(Constants.LOG_TAG, "finish Invalidate Cache");
        List<Message> combinedMessages = combineMessages(messages, notSentMessages);
        updateChatsCache(context, combinedMessages);
    }

    private List<Message> combineMessages(final List<Message> remoteMessages, List<Message>
            localMessages) {
        List<Message> combinedMessages = new ArrayList<>(remoteMessages);
        for (Message localMessage : localMessages) {
            if (isLocalMessageChatStillExist(localMessage, remoteMessages)) {
                combinedMessages.add(localMessage);
            }
            if (localMessage.getMessageState() == Message.MessageState.SENT
                    && getRemoteMessageThatEqualLocalMessage(localMessage, remoteMessages) !=
                    null) {
                combinedMessages.remove(getRemoteMessageThatEqualLocalMessage(localMessage,
                        remoteMessages));
            }
        }
        return combinedMessages;
    }

    private boolean isLocalMessageChatStillExist(Message localMessage, List<Message>
            remoteMessages) {
        for (Message remoteMessage : remoteMessages) {
            if (localMessage.getChatId() != null && localMessage.getChatId().equals(remoteMessage.getChatId())) {
                return true;
            }
        }
        return false;
    }

    @Nullable
    private Message getRemoteMessageThatEqualLocalMessage(Message localMessage, List<Message>
            remoteMessages) {
        for (Message remoteMessage : remoteMessages) {
            if (localMessage.getId().equals(remoteMessage.getId())) {
                return remoteMessage;
            }
        }
        return null;
    }

    private List<Message> getShouldBeNotifiedMessages(List<Message> messages) {
        List<Message> shouldBeNotifiedMessages = new ArrayList<>(messages);
        for (Message message : messages) {
            if (isLocalMessageAndSynced(message)
                    || isLocalMessageAndReadyToBeSynced(message)
                    || isLocalMessageAndNotFullySent(message)
                    || message.isInbound()
                    || message.isRead()) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Message removed from list to be notified");
                shouldBeNotifiedMessages.remove(message);
            }
        }
        return shouldBeNotifiedMessages;
    }

    @Nullable
    private Chat getMessageChat(Message message) {
        if (message.getChatId() == null) { return null; }
        InMemoryCache<String, Chat> chatsCache = ChatsCacheManager.getCache();
        if (chatsCache != null) {
            Chat chat = chatsCache.get(message.getChatId());
            if (chat != null) {
                return chat;
            }
        }
        InstabugSDKLogger.e(Constants.LOG_TAG, "No local chats match messages's chat");
        return null;
    }

    @Nullable
    private List<Message> getMessageChatMessages(Message message) {
        Chat chat = getMessageChat(message);
        if (chat == null) {
            return null;
        }
        return chat.getMessages();
    }

    @Nullable
    private Message isLocalMessage(Message message) {
        List<Message> localeMessages = getMessageChatMessages(message);
        if (localeMessages != null) {
            for (final Message localMessage : localeMessages) {
                if (localMessage.getId().equals(message.getId())) {
                    return localMessage;
                }
            }
        }
        return null;
    }

    private boolean isRemoteMessage(Message message) {
        return isLocalMessage(message) == null;
    }

    @SuppressLint({"ERADICATE_PARAMETER_NOT_NULLABLE"})
    private void addRemoteMessage(Message message) {
        // get message chat
        Chat chat = getMessageChat(message);
        if (chat == null && message.getChatId() != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Chat with id " + message.getChatId() + " doesn't " +
                    "exist, creating new one");
            chat = new Chat(message.getChatId());
            chat.setChatState(Chat.ChatState.SENT);
        }
        // add remote message to chat
        if(chat != null) {
            chat.getMessages().add(message);
        InstabugSDKLogger.v(Constants.LOG_TAG, "Message added to cached chat: " + chat);
        }
        // add chat to cache
        InMemoryCache<String, Chat> chatsCache = ChatsCacheManager.getCache();
        if (chatsCache != null && chat != null) {
            chatsCache.put(chat.getId(), chat);
        }
    }

    private boolean isLocalMessageAndSynced(Message message) {
        Message localMessage = isLocalMessage(message);
        return localMessage != null
                && localMessage.getId().equals(message.getId())
                && localMessage.getMessageState().equals(Message.MessageState.SYNCED)
                && localMessage.getAttachments().size() == message.getAttachments().size();
    }

    private boolean isLocalMessageAndReadyToBeSynced(Message message) {
        Message localMessage = isLocalMessage(message);
        return localMessage != null
                && localMessage.getId().equals(message.getId())
                && localMessage.getMessageState().equals(Message.MessageState.READY_TO_BE_SYNCED)
                && localMessage.getAttachments().size() == message.getAttachments().size();
    }

    private boolean isLocalMessageAndNotFullySent(Message message) {
        Message localMessage = isLocalMessage(message);
        return localMessage != null
                && localMessage.getId().equals(message.getId())
                && localMessage.getMessageState().equals(Message.MessageState.SENT)
                && localMessage.getAttachments().size() != message.getAttachments().size();
    }

    /**
     * Notifies <b>some</b> {@code onNewMessagesReceivedListeners} of new messages being received
     * with
     * the following logic:<br/>
     * These listeners will be called in reverse order, and will not be called when all messages
     * have been processed prior to it.
     *
     * @param messages messages to be sent to listeners
     * @see #addOnNewMessagesReceivedListener(OnNewMessagesReceivedListener)
     */
    private void notifyAllListeners(List<Message> messages) {
        if (ChatSettings.isNotificationEnable()) {
            for (int i = onNewMessagesReceivedListeners.size() - 1; i >= 0; i--) {
                OnNewMessagesReceivedListener listener = onNewMessagesReceivedListeners.get(i);
                InstabugSDKLogger.d(Constants.LOG_TAG, "Notifying listener " + listener);
                if (messages != null && messages.size() > 0) {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Notifying listener with " + messages.size() + " " +
                            "message(s)");
                    messages = listener.onNewMessagesReceived(messages);
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Notified listener remained " + (messages != null ?
                            messages.size() : null) + " message(s) to be sent to next listener");
                } else {
                    break;
                }
            }
        } else {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Chat notification disabled");
        }
    }

    /**
     * Add a new {@code OnNewMessagesReceivedListener} that will get notified with new messages<br/>
     * <b>NOTE:</b> these listeners will be called in reverse order, and will not be called
     * when all messages have been processed prior to it.
     *
     * @param listener to be added
     */
    public void addOnNewMessagesReceivedListener(OnNewMessagesReceivedListener listener) {
        if (!onNewMessagesReceivedListeners.contains(listener)) {
            onNewMessagesReceivedListeners.add(listener);
        }
    }

    /**
     * Removes {@code listener} from list of {@link #onNewMessagesReceivedListeners}
     *
     * @param listener to be removed
     */
    public void removeOnNewMessagesReceivedListener(OnNewMessagesReceivedListener listener) {
        onNewMessagesReceivedListeners.remove(listener);
    }

    /**
     * Parses all un-notified Message JSON to {@code List<BaseMessage>}
     *
     * @param newMessagesJsonArray JSON to parse
     * @return a list of {@link Message}'s to be sent to listeners
     */
    @VisibleForTesting
    public List<Message> parseNewMessages(JSONObject[] newMessagesJsonArray) {
        List<Message> parsedMessage = new ArrayList<>();
        for (int i = 0; i < newMessagesJsonArray.length; i++) {
            try {
                JSONObject messageObject = newMessagesJsonArray[i];
                JSONArray attachmentsArray = messageObject.getJSONArray("attachments");
                JSONArray actionsArray = messageObject.getJSONArray("actions");
                Message message = new Message(messageObject.getString("id"), UserManagerWrapper.getUserName(), UserManagerWrapper.getUserEmail(), InstabugCore.getPushNotificationToken())
                        .setChatId(messageObject.getString("chat_number"))
                        .setBody(messageObject.getString("body"))
                        .setSenderName(messageObject.getJSONObject("from").getString("name"))
                        .setSenderAvatarUrl(messageObject.getString("avatar"))
                        .setMessageState(Message.MessageState.SYNCED);
                if (messageObject.getString("messaged_at") != null
                        && !messageObject.getString("messaged_at").equals("")
                        && !messageObject.getString("messaged_at").equals("null")
                        && InstabugDateFormatter.getDate(messageObject.getString("messaged_at"))
                        != null) {
                    message.setMessagedAt(InstabugDateFormatter.getDate(messageObject.getString
                            ("messaged_at")).getTime() / 1000);
                }
                if (messageObject.getString("read_at") != null
                        && !messageObject.getString("read_at").equals("")
                        && !messageObject.getString("read_at").equals("null")
                        && InstabugDateFormatter.getDate(messageObject.getString("read_at")) !=
                        null) {
                    message.setReadAt(InstabugDateFormatter.getDate(messageObject.getString
                            ("read_at")).getTime() / 1000);
                }

                String messageDirection = messageObject.getString("direction");
                if (messageDirection != null) {
                    switch (messageDirection) {
                        case "inbound":
                            message.setDirection(Message.Direction.INBOUND);
                            break;
                        case "outbound":
                            message.setDirection(Message.Direction.OUTBOUND);
                            break;
                        default:
                            break;
                    }
                }

                for (int j = attachmentsArray.length() - 1; j >= 0; j--) {
                    JSONObject attachmentObject = attachmentsArray.getJSONObject(j);
                    JSONObject attachmentMetadataJsonObject = attachmentObject.getJSONObject
                            ("metadata");
                    Attachment attachment = new Attachment();
                    attachment.setUrl(attachmentObject.getString("url"));
                    attachment.setState(Attachment.AttachmentState.STATE_SYNCED);
                    String fileType = attachmentMetadataJsonObject.getString("file_type");
                    attachment.setType(fileType);
                    message.getAttachments().add(attachment);
                }

                for (int j = actionsArray.length() - 1; j >= 0; j--) {
                    JSONObject actionObject = actionsArray.getJSONObject(j);
                    MessageAction action = new MessageAction();
                    action.fromJson(actionObject.toString());
                    message.addAction(action);
                }
                parsedMessage.add(message);
            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to parse chat message");
            }
        }
        return parsedMessage;
    }

    private void fireNewMessagesRunnable() {
        if (ChatSettings.getNewMessageRunnable() != null) {
            try {
                Handler handler = new Handler(Looper.getMainLooper());
                handler.post(ChatSettings.getNewMessageRunnable());
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "new message runnable failed to run.", e);
            }
        }
    }
}