package com.instabug.bug.invocation.invocationdialog;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.view.View;
import android.widget.FrameLayout;

import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.view.ViewCompat;
import androidx.fragment.app.FragmentTransaction;

import com.instabug.library.Instabug;
import com.instabug.library.OnSdkDismissCallback;
import com.instabug.bug.R;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.BaseFragmentActivity;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.ui.InstabugDialogActivityBase;
import com.instabug.library.util.ScreenUtility;
import com.instabug.library.util.StatusBarUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

/**
 * Created by mzelzoghbi on 4/18/18.
 */
public class InstabugDialogActivity extends InstabugDialogActivityBase<InstabugDialogActivityPresenter>
        implements InstabugDialogFragment.Callbacks, InstabugDialogActivityContract.View, View.OnClickListener,
        InstabugDialogActivityContract.AnimationProvider {

    private final static String TAG = "InstabugDialogActivity";
    public final static String KEY_DIALOG_ITEMS = "dialog_items";
    public final static String KEY_DIALOG_TITLE = "dialog_title";
    public final static String KEY_SCREENSHOT_URI = "screenshot_uri";
    /**
     * By default, accessibility TalkBack tool takes the TextView contentDescription from the text attribute.
     * According to product, the Prompt options Screen title is modified when we launch the main options but kept as is in the sub-categories options.
     * So, this key was made to see whether or not we should change the content description of the title TextView.
     */
    public static final String KEY_DIALOG_SHOULD_OVERRIDE_TITLE_DESC = "dialog_should_override_title_desc";
    /**
     * this key is for killing the activity after any dialog item clicked
     */
    private static final String KEY_SHOULD_BE_KILLED = "should_be_killed";
    // shared view to animate dialogs
    @Nullable
    private View[] sharedViews;

    private boolean shouldBeKilled = false;
    @Nullable
    private Uri screenshotUri;

    @Nullable
    private ArrayList<InstabugDialogItem> dialogItems;
    /**
     * To avoid multiple view inflation from the onReume method.
     */
    private boolean isInflated = false;
    private static Locale currentDialogLocale;

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static Intent getIntent(Context context, @Nullable String title, @Nullable Uri screenshot,
                                   @Nullable ArrayList<InstabugDialogItem> items, boolean shouldBeKilled) {
        Intent intent = new Intent(context, InstabugDialogActivity.class);
        intent.putExtra(KEY_DIALOG_TITLE, title);
        intent.putExtra(KEY_SCREENSHOT_URI, screenshot);
        intent.putExtra(KEY_DIALOG_ITEMS, items);
        intent.putExtra(KEY_SHOULD_BE_KILLED, shouldBeKilled);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        return intent;
    }

    @Override
    public AppCompatActivity getViewContext() {
        return null;
    }

    @Override
    public void finishActivity() {

    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        overridePendingTransition(R.anim.ib_core_anim_fade_in, R.anim.ib_core_anim_fade_out);
        super.onCreate(savedInstanceState);
        StatusBarUtils.setStatusBarColorForPrompt(this);

        if (ScreenUtility.hasNavBar(this) && !ScreenUtility.isLandscape(this)) {
            FrameLayout layout = findViewById(R.id.ib_fragment_container);
            if (layout != null) {
                layout.setPadding(layout.getPaddingLeft(), layout.getPaddingTop(),
                        layout.getPaddingRight(), layout.getPaddingBottom() + ScreenUtility.getNavigationBarHeight(getResources()));
            }
        }
        screenshotUri = getIntent().getParcelableExtra(KEY_SCREENSHOT_URI);
        runOnSdkInvokedRunnable();

        /* By default, Talkback explicitly announces the screen title when launched.
         In the prompt screen, we have no screen title which makes the system falls back to the App name.
         So, in a scenario where the app name is XYZ, the Talkback will announce XYZ when it launches the Prompt options screen.
         So, as a workaround, title is set to an empty string.*/
        setTitle(" ");
    }

    private InstabugDialogActivityPresenter createPresenter() {
        return new InstabugDialogActivityPresenter(this);
    }

    @SuppressLint("NULL_DEREFERENCE")
    private void runOnSdkInvokedRunnable() {
        if (SettingsManager.getInstance().getOnSdkInvokedCallback() != null) {
            SettingsManager.getInstance().getOnSdkInvokedCallback().onSdkInvoked();
        }
    }

    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);
        String dialogTitle = getIntent().getStringExtra(KEY_DIALOG_TITLE);

        @SuppressWarnings("unchecked")
        ArrayList<InstabugDialogItem> dialogItems = (ArrayList<InstabugDialogItem>) intent.getSerializableExtra(KEY_DIALOG_ITEMS);
        if (areDialogItemsTheSame(dialogItems)) {
            return;
        }
        setIntent(intent);
        if (dialogItems == null) {
            finish();
            return;
        }
        dialogTitle = dialogTitle == null ? "" : dialogTitle;
        setContent(dialogTitle, true, dialogItems);

        if (intent.getBooleanExtra(KEY_SHOULD_BE_KILLED, false)) {
            shouldBeKilled = true;
        }
    }

    private boolean areDialogItemsTheSame(@Nullable ArrayList<InstabugDialogItem> newDialogItems) {
        if (this.dialogItems != null) {
            return this.dialogItems.equals(newDialogItems);
        }
        return newDialogItems == null;
    }

    @Override
    protected int getLayout() {
        return R.layout.ib_lyt_activity_dialog;
    }

    @Override
    @SuppressWarnings("unchecked")
    protected void initViews() {
        if (presenter == null) {
            presenter = createPresenter();
        }
        if (getIntent().getSerializableExtra(KEY_DIALOG_ITEMS) == null) {
            finish();
            return;
        }
        findViewById(R.id.ib_fragment_container).setOnClickListener(this);
        findViewById(R.id.ib_dialog_container).setOnClickListener(this);

        if (getIntent().getBooleanExtra(KEY_SHOULD_BE_KILLED, false)) {
            shouldBeKilled = true;
        }

        if (dialogItems == null) {
            dialogItems = (ArrayList<InstabugDialogItem>) getIntent().getSerializableExtra(KEY_DIALOG_ITEMS);
        }

    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        outState.putSerializable(KEY_DIALOG_ITEMS, dialogItems);
        super.onSaveInstanceState(outState);
    }

    @Override
    protected void onRestoreInstanceState(Bundle savedInstanceState) {
        super.onRestoreInstanceState(savedInstanceState);
        dialogItems = (ArrayList<InstabugDialogItem>) savedInstanceState.getSerializable(KEY_DIALOG_ITEMS);
    }

    @Override
    public void onDialogItemClicked(@Nullable InstabugDialogItem item, View... sharedViews) {
        this.sharedViews = sharedViews;
        if (presenter != null) presenter.onItemClicked(item, screenshotUri);
        if (shouldBeKilled) {
            finish();
        }
    }

    @Override
    public void removeScreenshotIfNeeded(InstabugDialogItem chatsListAction) {
        if (presenter != null) presenter.removeScreenShotIfNeeded(chatsListAction);
    }


    @Override
    public void onClick(View v) {
        if (v.getId() == R.id.ib_fragment_container || v.getId() == R.id.ib_dialog_container) {
            if (presenter != null) {
                presenter.resetPromptOptionsSelection();
            }
            finish();
        }
    }

    @Override
    public void onBackPressed() {
        overridePendingTransition(R.anim.ib_core_anim_fade_in, R.anim.ib_core_anim_fade_out);
        if (presenter != null) presenter.onBackPressed();
        super.onBackPressed();
    }

    @Override
    public void finish() {
        if (presenter != null && !presenter.isPromptOptionSelected()) {
            OnSdkDismissCallback onSdkDismissCallback = SettingsManager.getInstance().getOnSdkDismissCallback();
            if (onSdkDismissCallback != null) {
                onSdkDismissCallback.call(OnSdkDismissCallback.DismissType.CANCEL, OnSdkDismissCallback.ReportType.OTHER);
            }
        }
        super.finish();
        overridePendingTransition(R.anim.ib_core_anim_fade_in, R.anim.ib_core_anim_fade_out);
    }


    @Override
    public void onStart() {
        super.onStart();
        if (presenter != null) presenter.onViewStarted();
    }

    @Override
    public void onStop() {
        super.onStop();
        if (presenter != null) presenter.onViewStopped();
    }

    /**
     * Before dumbing the attachments, fist check if the activity is really finishing or
     * it's just an orientation change.
     */
    @Override
    protected void onDestroy() {
        if (isFinishing() && presenter != null && !presenter.isPromptOptionSelected()) {
            presenter.dumpAttachments(screenshotUri);
        }
        super.onDestroy();
    }

    @Override
    protected void onResume() {
        super.onResume();

        // recreate activity if locale changed
        if (currentDialogLocale != null &&
                !InstabugCore.getLocale(this).equals(currentDialogLocale)) {
            finish();
            Instabug.show();
        }

        if (!isInflated) {
            getSupportFragmentManager().beginTransaction()
                    .replace(R.id.ib_fragment_container,
                            InstabugDialogFragment.newInstance(getIntent().getStringExtra(KEY_DIALOG_TITLE),
                                    true,
                                    dialogItems))
                    .commit();
            isInflated = true;
        }

        SettingsManager.getInstance().setPromptOptionsScreenShown(true);
    }

    @Override
    protected void onPause() {
        super.onPause();
        SettingsManager.getInstance().setPromptOptionsScreenShown(false);
        currentDialogLocale = InstabugCore.getLocale(this);
    }

    @Override
    @SuppressWarnings("unchecked")
    @Nullable
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_RETURN_ANNOTATION")
    public List<InstabugDialogItem> getDialogItems() {
        if (getIntent() != null) {
            return (ArrayList<InstabugDialogItem>)
                    getIntent().getSerializableExtra(KEY_DIALOG_ITEMS);
        }
        return null;
    }

    @Override
    public void onInitialScreenShotNotRequired() {
        if (presenter != null) presenter.dumpAttachments(screenshotUri);

    }

    @Override
    public void setContent(@Nullable String title, boolean shouldOverrideTitleContentDescription, ArrayList<InstabugDialogItem> items) {
        FragmentTransaction fragmentTransaction = getSupportFragmentManager().beginTransaction();
        try {
            if (sharedViews != null) {
                for (View sharedView : sharedViews) {
                    if (sharedView != null) {
                        String transitionName = ViewCompat.getTransitionName(sharedView);
                        if (transitionName != null)
                            fragmentTransaction.addSharedElement(sharedView, transitionName);
                    }
                }
            }
        } catch (IllegalArgumentException exception) {
            // for some reason only Google developers and Allah knows, IllegalArgumentException is
            // thrown when you change orientation and click on any dialog fragment item.
        }
        dialogItems = items;
        fragmentTransaction
                .setCustomAnimations(0, R.anim.ib_core_anim_invocation_dialog_exit)
                .addToBackStack(null)
                .replace(R.id.ib_fragment_container, InstabugDialogFragment.newInstance(title, shouldOverrideTitleContentDescription, items))
                .commit();
    }

    @Override
    public int getSlidInLeftAnimation() {
        return R.anim.ib_core_anim_slide_in_left;
    }

    @Override
    public int getSlidInRightAnimation() {
        return R.anim.ib_core_anim_slide_in_right;
    }

    @Override
    public int getSlidOutLeftAnimation() {
        return R.anim.ib_core_anim_slide_out_left;
    }

    @Override
    public int getSlidOutRightAnimation() {
        return R.anim.ib_core_anim_slide_out_right;
    }

    @Override
    public int getFadeInAnimation() {
        return R.anim.ib_core_anim_fade_in;
    }

    @Override
    public int getEnterAnimation() {
        if (presenter != null) return presenter.getContentEnterAnimation();
        else return 0;
    }

    @Override
    public int getExitAnimation() {
        if (presenter != null) return presenter.getContentExitAnimation();
        else return 0;
    }
}
