package com.instabug.bug.internal.video;

import static android.app.Activity.RESULT_OK;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Handler;
import android.os.Looper;

import androidx.annotation.Nullable;

import com.instabug.bug.invocation.invoker.ScreenRecordingFab;
import com.instabug.library.Instabug;
import com.instabug.library.core.eventbus.ScreenRecordingEventBus;
import com.instabug.library.internal.InstabugMediaProjectionIntent;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.video.ScreenRecordingContract;
import com.instabug.library.internal.video.ScreenRecordingEvent;
import com.instabug.library.internal.video.ScreenRecordingFileHolder;
import com.instabug.library.invocation.InvocationManagerContract;
import com.instabug.library.util.MicUtils;

import java.io.File;

import io.reactivexport.Observable;
import io.reactivexport.subjects.BehaviorSubject;
import io.reactivexport.subjects.Subject;


public class InternalScreenRecordHelper implements ScreenRecordingContract,
        ScreenRecordingFab.ScreenRecordingFloatingBtnEventListener {


    private static InternalScreenRecordHelper INSTANCE;
    private final Subject<Boolean> stopSubject;
    @Nullable
    private ScreenRecordingFab fab;
    private boolean canStopRecording;
    @Nullable
    private ScreenRecordingFileHolder fileHolder;
    private boolean isCurrentlyRecording = false;

    private InternalScreenRecordHelper() {
        stopSubject = BehaviorSubject.createDefault(false);
    }

    public synchronized static InternalScreenRecordHelper getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new InternalScreenRecordHelper();
        }
        return INSTANCE;
    }

    @SuppressLint("NULL_DEREFERENCE")
    public void init() {
        InvocationManagerContract contract = CoreServiceLocator.getInvocationManagerContract();
        if (contract != null) {
            contract.switchOffInvocation();
        }
        fileHolder = ScreenRecordingFileHolder.getInstance();

        if (fab == null) {
            fab = new ScreenRecordingFab(this);
        }
        fab.init();
    }

    public void start() {
        isCurrentlyRecording = true;
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            Intent screenRecordingIntent = ScreenRecordingService.newIntent(context, RESULT_OK, InstabugMediaProjectionIntent.getMediaProjectionIntent(), true);
            ScreenRecordingService.startScreenRecordingService(context, screenRecordingIntent);
        }

        new Handler(Looper.getMainLooper()).postDelayed(() -> {
            if (isRecording()) {
                stopSubject.onNext(true);
            }
        }, 1000);
    }

    @Override
    public void pause() {

    }

    @Override
    public void stop(int duration) {
        if (canStopRecording) {
            ScreenRecordingEventBus.getInstance().post(new ScreenRecordingEvent(ScreenRecordingEvent.RECORDING_FINISHED, getAutoScreenRecordingFileUri(), duration));
            release();
        }
    }

    @SuppressLint("NULL_DEREFERENCE")
    public void release() {
        if (fab != null) {
            fab.release();
        }
        InvocationManagerContract contract = CoreServiceLocator.getInvocationManagerContract();
        if (contract != null) {
            contract.switchOnInvocation();
        }

        //unmute mic if muted
        if (Instabug.getApplicationContext() != null) {
            MicUtils.unmuteMic(Instabug.getApplicationContext());
        }
        stopSubject.onNext(false);

        isCurrentlyRecording = false;

    }

    @Nullable
    @Override
    public Uri getAutoScreenRecordingFileUri() {
        return fileHolder == null ? null : fileHolder.getAutoScreenRecordingFileUri();
    }

    @Override
    public void delete() {
        if (fileHolder != null) {
            fileHolder.delete();
        }
    }

    @Override
    public void clear() {
        isCurrentlyRecording = false;
        if (fileHolder != null) {
            fileHolder.clear();
        }
    }

    @Override
    public boolean isEnabled() {
        return true;
    }

    public boolean isRecording() {
        return isCurrentlyRecording;
    }

    public void cancel() {
        if (isRecording()) {
            resetFab();
        }
    }

    public Observable<Boolean> getIsStoppableObservable() {
        return stopSubject.hide()
                .doOnNext(aBoolean -> canStopRecording = aBoolean);
    }

    void onRecordingFinished() {
        ScreenRecordingEventBus.getInstance().post(new ScreenRecordingEvent(ScreenRecordingEvent.RECORDING_FILE_READY, getAutoScreenRecordingFileUri()));
    }

    void onRecordingError() {
        ScreenRecordingEventBus.getInstance().post(new ScreenRecordingEvent(ScreenRecordingEvent.RECORDING_ERROR, null));
    }

    void setAutoScreenRecordingFile(File autoScreenRecordingFile) {
        if (this.fileHolder != null) {
            this.fileHolder.setAutoScreenRecordingFile(autoScreenRecordingFile);
        }
    }

    void startTimerOnRecordingFAB() {
        if (fab != null) {
            fab.startTimerOnRecordingButton();
        }
    }

    private void resetFab() {
        if (fab != null) {
            fab.release();
            fab.init();
        }
    }
}

