package com.instabug.chat.synchronization;

import static com.instabug.chat.RepliesWrapper.isMessagingServiceAvailable;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Handler;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.chat.Constants;
import com.instabug.chat.cache.ChatsCacheManager;
import com.instabug.chat.cache.ReadQueueCacheManager;
import com.instabug.chat.eventbus.ChatTimeUpdatedEventBus;
import com.instabug.chat.model.ReadMessage;
import com.instabug.chat.network.service.MessagingService;
import com.instabug.chat.settings.ChatSettings;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.eventpublisher.Subscriber;
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.net.HttpURLConnection;
import java.util.List;

import io.reactivexport.functions.Consumer;

/**
 * @author mesbah.
 */
@SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
public class SynchronizationManager {

    private static final String TTL = "ttl";
    private static final String MISSING_MESSAGES = "missing_messages";
    private static final long STOP_PULLING = -1L;
    private static volatile SynchronizationManager INSTANCE;
    @Nullable
    private Handler handler;
    @Nullable
    private SyncRunnable syncRunnable;
    @Nullable
    private IBGDisposable chatTimeSubscriber;

    private boolean shouldSync = false;
    private boolean isSyncing = false;

    private Consumer<Long> syncAction = new Consumer<Long>() {
        @Override
        public void accept(Long TTL) {
            if (shouldSync() && handler != null && syncRunnable != null) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "Waiting " + TTL + " seconds until the next chats sync");
                handler.postDelayed(syncRunnable, (TTL * 1000));
            }
        }

        private boolean shouldSync() {
            return shouldSync && handler != null && syncRunnable != null;
        }
    };

    private final Subscriber<Long> chattingTimeUpdateAction = event -> sync(false);

    private SynchronizationManager(final Context context) {
        PoolProvider.postMainThreadTask(new Runnable() {
            @Override
            public void run() {
                handler = new Handler();
                syncRunnable = new SyncRunnable(context);
                subscribeToChatTimeUpdatedEvents();
            }
        });
    }

    public static void init(Context context) {
        if (INSTANCE == null) {
            INSTANCE = new SynchronizationManager(context);
        }
    }

    public synchronized static SynchronizationManager getInstance() {
        if (INSTANCE == null && Instabug.getApplicationContext() != null) {
            init(Instabug.getApplicationContext());
        }
        return INSTANCE;
    }

    @VisibleForTesting
    static SynchronizationManager getInstanceUnModified() {
        return INSTANCE;
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
    static void tearDown() {
        INSTANCE = null;
    }

    public void sync(boolean force) {
        if(force)
            resetTTL();

        Handler handler = this.handler;
        if (handler == null || syncRunnable == null) return;

        if (isMessagingServiceAvailable() && !isSyncing()) {
            stop();
            shouldSync = true;
            handler.post(syncRunnable);
        }
        this.handler = handler;
    }

    public void stop() {
        shouldSync = false;
        if (handler != null && syncRunnable != null) {
            handler.removeCallbacks(syncRunnable);
        }
    }

    @SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
    public void release() {
        stop();
        unSubscribeToChatTimeUpdatedEvents();
        handler = null;
        syncRunnable = null;
        SynchronizationManager.tearDown();
    }

    private boolean isSyncing() {
        return isSyncing;
    }

    private void syncMessages(final Context context, final Consumer<Long> action) {
        if (isMessagingServiceAvailable()) {
            try {
                isSyncing = true;
                final List<ReadMessage> currentReadMessage = ReadQueueCacheManager.getInstance()
                        .getAll();
                MessagingService.getInstance().syncMessages(ChatsCacheManager
                                .getLastMessageMessagedAt(), ChatsCacheManager
                                .getTotalMessagesCount(),
                        ReadQueueCacheManager.getInstance().getReadMessagesArray(), new Request.Callbacks<RequestResponse, Throwable>() {
                            @Override
                            public void onSucceeded(@Nullable RequestResponse response) {
                                if (response != null) {
                                    handleSuccessResponse(response, context, action);
                                }
                                clearReadMessages(currentReadMessage);
                            }

                            @Override
                            public void onFailed(Throwable throwable) {
                                handleFailureResponse(action);
                            }
                        });
            } catch (JSONException e) {

                handleFailureResponse(action);
            }
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Can't sync chats because device is offline");
            //schedule next pull with last TTL value.
            try {
                action.accept(ChatSettings.getTTL());
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Syncing chats got error: " + e.getMessage());
            }
        }
    }

    private void handleSuccessResponse(RequestResponse response, Context context, Consumer<Long>
            action) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Chats synced successfully");
        isSyncing = false;
        try {
            Object responseBody = response.getResponseBody();
            if (responseBody instanceof String) {
                handleReceivedMessages(context, parseReceivedMessages((String) responseBody), response.getResponseCode() == HttpURLConnection
                        .HTTP_NOT_AUTHORITATIVE);
                handleTTL(parseTTL((String) responseBody), action);
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Exception was occurred," + e.getMessage() + " while handling chats sync response", e);
            try {
                action.accept(ChatSettings.getTTL());
            } catch (Exception e1) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Exception was occurred," + e1.getMessage());
            }
        }
    }

    private void handleFailureResponse(Consumer<Long> action) {
        InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while sync messages");
        isSyncing = false;
        try {
            action.accept(ChatSettings.getTTL());
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Exception was occurred while sync messages," + e.getMessage());
        }
    }

    private long parseTTL(String syncRequestResponse) throws JSONException {
        JSONObject responseJsonObject = new JSONObject(syncRequestResponse);
        return responseJsonObject.getLong(TTL);
    }

    private void handleTTL(long TTL, Consumer<Long> action) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Next TTL: " + TTL);
        if (TTL != STOP_PULLING) {
            ChatSettings.setTTL(TTL);
            try {
                action.accept(TTL);
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Exception was occurred while handling TTL," + e.getMessage());
            }
        }
    }

    private JSONArray parseReceivedMessages(String syncRequestResponse) throws JSONException {
        JSONObject responseJsonObject = new JSONObject(syncRequestResponse);
        return responseJsonObject.getJSONArray(MISSING_MESSAGES);
    }

    private void handleReceivedMessages(Context context, JSONArray receivedMessages, boolean
            shouldInvalidateCache) throws JSONException {
        JSONObject[] messages = new JSONObject[0];
        if (receivedMessages.length() != 0) {
            InstabugSDKLogger.v(Constants.LOG_TAG, receivedMessages.length() + "new messages received");
            messages = new JSONObject[receivedMessages.length()];
            for (int i = 0; i < receivedMessages.length(); i++) {
                messages[i] = receivedMessages.getJSONObject(i);
            }
        }
        NewMessagesHandler.getInstance().handleNewMessagesReceived(context,
                shouldInvalidateCache, messages);
    }

    private void clearReadMessages(List<ReadMessage> readMessages) {
        ReadQueueCacheManager.getInstance().notify(readMessages);
    }

    private boolean isFeaturesFetchedBefore() {
        return InstabugCore.isFeaturesFetchedBefore();
    }

    private class SyncRunnable implements Runnable {

        WeakReference<Context> contextWeakReference;

        SyncRunnable(Context context) {
            contextWeakReference = new WeakReference<>(context);
        }

        @Override
        public void run() {
            if (isMessagingServiceAvailable()) {
                PoolProvider.postIOTaskWithCheck(new Runnable() {
                    @Override
                    public void run() {
                        if (contextWeakReference != null && contextWeakReference.get() != null)
                            syncMessages(contextWeakReference.get(), syncAction);
                        else {
                            try {
                                syncAction.accept(ChatSettings.getTTL());
                            } catch (Exception e) {
                                InstabugSDKLogger.e(Constants.LOG_TAG, "Exception was occurred," + e.getMessage());
                            }
                        }
                    }
                });
            }
        }
    }

    private void subscribeToChatTimeUpdatedEvents() {
        chatTimeSubscriber = ChatTimeUpdatedEventBus.getInstance()
                .subscribe(chattingTimeUpdateAction);
    }

    private void unSubscribeToChatTimeUpdatedEvents() {
        if (chatTimeSubscriber != null)
            chatTimeSubscriber.dispose();
    }

    private void resetTTL() {
        ChatSettings.setTTL(0);
    }
}
