package com.instabug.chat.model;

import android.annotation.SuppressLint;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.tokenmapping.TokenMappingServiceLocator;
import com.instabug.library.user.UserManagerWrapper;

import org.jetbrains.annotations.NotNull;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;

/**
 * @author vezikon
 */
public class Message implements Cacheable, Serializable {

    private static final long serialVersionUID = 6106269076155338045L;

    // keys used by toJson() & fromJson() methods
    @VisibleForTesting
    static final String KEY_ID = "id";
    @VisibleForTesting
    static final String KEY_CHAT_ID = "chat_id";
    @VisibleForTesting
    static final String KEY_BODY = "body";
    @VisibleForTesting
    static final String KEY_SENDER_NAME = "sender_name";
    @VisibleForTesting
    static final String KEY_SENDER_AVATAR_URL = "sender_avatar_url";
    @VisibleForTesting
    static final String KEY_MESSAGED_AT = "messaged_at";
    @VisibleForTesting
    static final String KEY_READ = "read";
    @VisibleForTesting
    static final String KEY_READ_AT = "read_at";
    @VisibleForTesting
    static final String KEY_ATTACHMENTS = "attachments";
    @VisibleForTesting
    static final String KEY_ACTIONS = "actions";
    @VisibleForTesting
    static final String KEY_DIRECTION = "direction";
    @VisibleForTesting
    static final String KEY_MESSAGES_STATE = "messages_state";
    static final String KEY_APP_TOKEN = "application_token";

    private String id;
    @Nullable
    private String chatId;
    @Nullable
    private String body;
    @Nullable
    private String senderName;
    @Nullable
    private String senderAvatarUrl;
    private long messagedAt;
    private boolean isRead;
    private long readAt;
    private ArrayList<Attachment> attachments;
    private ArrayList<MessageAction> actions;
    private Direction direction;
    private MessageState messageState;
    @Nullable
    private String requesterName;
    private String requesterEmail;
    private String deviceToken;
    @Nullable
    private String appToken;

    public Message(@Nullable String requesterName, String requesterEmail, String deviceToken) {
        this(String.valueOf(System.currentTimeMillis()), requesterName, requesterEmail, deviceToken);
    }

    public Message(String id, @Nullable String requesterName, String requesterEmail, String deviceToken) {
        this.id = id;
        attachments = new ArrayList<>();
        actions = new ArrayList<>();
        this.direction = Direction.NOT_AVAILABLE;
        this.messageState = MessageState.NOT_AVAILABLE;
        this.requesterName = requesterName;
        this.requesterEmail = requesterEmail;
        this.deviceToken = deviceToken;
        this.appToken = TokenMappingServiceLocator.getTokenMappingConfigs().getAvailableAppToken();
    }

    public static JSONArray toJson(ArrayList<Message> messages) throws JSONException {
        JSONArray messagesJsonArray = new JSONArray();
        for (int i = 0; i < messages.size(); i++) {
            messagesJsonArray.put(new JSONObject(messages.get(i).toJson()));
        }
        return messagesJsonArray;
    }

    public static ArrayList<Message> fromJson(JSONArray messagesJsonArray) throws JSONException {
        ArrayList<Message> messages = new ArrayList<>();
        for (int i = 0; i < messagesJsonArray.length(); i++) {
            Message message = new Message(UserManagerWrapper.getUserName(), UserManagerWrapper.getUserEmail(), InstabugCore.getPushNotificationToken());
            message.fromJson(messagesJsonArray.getJSONObject(i).toString());
            messages.add(message);
        }
        return messages;
    }

    public String getId() {
        return id;
    }

    public Message setId(String id) {
        this.id = id;
        return this;
    }

    @Nullable
    public String getChatId() {
        return chatId;
    }

    public Message setChatId(String chatId) {
        this.chatId = chatId;
        return this;
    }

    @Nullable
    public String getBody() {
        return body;
    }

    public Message setBody(String body) {
        this.body = body;
        return this;
    }

    public boolean isRead() {
        return isRead;
    }

    public Message setRead(boolean read) {
        isRead = read;
        return this;
    }

    public long getReadAt() {
        return readAt;
    }

    public Message setReadAt(long readAt) {
        this.readAt = readAt;
        if (readAt != 0L) {
            isRead = true;
        }
        return this;
    }

    public long getMessagedAt() {
        return messagedAt;
    }

    public Message setMessagedAt(long messagedAt) {
        this.messagedAt = messagedAt;
        return this;
    }

    @Nullable
    public String getSenderName() {
        return senderName;
    }

    public Message setSenderName(@Nullable String senderName) {
        this.senderName = senderName;
        return this;
    }

    @Nullable
    public String getSenderAvatarUrl() {
        return senderAvatarUrl;
    }

    public Message setSenderAvatarUrl(String senderAvatarUrl) {
        this.senderAvatarUrl = senderAvatarUrl;
        return this;
    }

    public MessageState getMessageState() {
        return messageState;
    }

    public Message setMessageState(MessageState messageState) {
        this.messageState = messageState;
        return this;
    }

    public ArrayList<Attachment> getAttachments() {
        return attachments;
    }

    public Message setAttachments(@NonNull ArrayList<Attachment> attachments) {
        this.attachments = attachments;
        return this;
    }

    public Message addAttachment(Attachment attachment) {
        attachments.add(attachment);
        return this;
    }

    public ArrayList<MessageAction> getActions() {
        return actions;
    }

    public Message setActions(ArrayList<MessageAction> actions) {
        this.actions = actions;
        return this;
    }

    public Message addAction(MessageAction action) {
        actions.add(action);
        return this;
    }

    public Direction getMessageDirection() {
        return direction;
    }

    public Message setDirection(Direction direction) {
        this.direction = direction;
        if (direction == Direction.INBOUND) {
            isRead = true;
        }
        return this;
    }

    @Nullable
    public String getAppToken() {
        return appToken;
    }

    public void setAppToken(String appToken) {
        this.appToken = appToken;
    }

    @Nullable
    public String getRequesterName() {
        return requesterName;
    }

    public String getRequesterEmail() {
        return requesterEmail;
    }

    public String getDeviceToken() {
        return deviceToken;
    }

    public boolean isInbound() {
        return direction != null && direction == Direction.INBOUND;
    }

    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String toJson() throws JSONException {
        JSONObject messageJsonObject = new JSONObject();
        messageJsonObject.put(KEY_ID, getId())
                .put(KEY_CHAT_ID, getChatId())
                .put(KEY_BODY, getBody())
                .put(KEY_SENDER_NAME, getSenderName())
                .put(KEY_SENDER_AVATAR_URL, getSenderAvatarUrl())
                .put(KEY_MESSAGED_AT, getMessagedAt())
                .put(KEY_READ, isRead())
                .put(KEY_READ_AT, getReadAt())
                .put(KEY_MESSAGES_STATE, getMessageState().toString())
                .put(KEY_DIRECTION, getMessageDirection().toString())
                .put(KEY_ATTACHMENTS, Attachment.toJson(getAttachments()))
                .put(KEY_ACTIONS, MessageAction.toJson(getActions()))
                .put(KEY_APP_TOKEN, getAppToken());
        return messageJsonObject.toString();
    }

    public void fromJson(String messageAsJson) throws JSONException {
        JSONObject messageJsonObject = new JSONObject(messageAsJson);
        if (messageJsonObject.has(KEY_ID))
            setId(messageJsonObject.getString(KEY_ID));
        if (messageJsonObject.has(KEY_CHAT_ID))
            setChatId(messageJsonObject.getString(KEY_CHAT_ID));
        if (messageJsonObject.has(KEY_BODY))
            setBody(messageJsonObject.getString(KEY_BODY));
        if (messageJsonObject.has(KEY_SENDER_NAME))
            setSenderName(messageJsonObject.getString(KEY_SENDER_NAME));
        if (messageJsonObject.has(KEY_SENDER_AVATAR_URL))
            setSenderAvatarUrl(messageJsonObject.getString(KEY_SENDER_AVATAR_URL));
        if (messageJsonObject.has(KEY_MESSAGED_AT))
            setMessagedAt(messageJsonObject.getLong(KEY_MESSAGED_AT));
        if (messageJsonObject.has(KEY_READ))
            setRead(messageJsonObject.getBoolean(KEY_READ));
        if (messageJsonObject.has(KEY_READ_AT))
            setReadAt(messageJsonObject.getLong(KEY_READ_AT));
        if (messageJsonObject.has(KEY_ATTACHMENTS))
            setAttachments(Attachment.fromJson(messageJsonObject.getJSONArray(KEY_ATTACHMENTS)));
        if (messageJsonObject.has(KEY_ACTIONS))
            setActions(MessageAction.fromJson(messageJsonObject.getJSONArray(KEY_ACTIONS)));
        if (messageJsonObject.has(KEY_DIRECTION)) {
            Direction direction;
            switch (messageJsonObject.getString(KEY_DIRECTION)) {
                case "inbound":
                    direction = Direction.INBOUND;
                    break;
                case "outbound":
                    direction = Direction.OUTBOUND;
                    break;
                default:
                    direction = Direction.NOT_AVAILABLE;
                    break;
            }
            setDirection(direction);
        }
        if (messageJsonObject.has(KEY_MESSAGES_STATE))
            setMessageState(MessageState.valueOf(messageJsonObject.getString(KEY_MESSAGES_STATE)));
        if (messageJsonObject.has(KEY_APP_TOKEN))
            setAppToken(messageJsonObject.getString(KEY_APP_TOKEN));
    }

    @Override
    @NonNull
    public String toString() {
        return "Message:[" + id + ", " + chatId + ", " + body + ", " + messagedAt + ", "
                + readAt + ", " + senderName + ", " + senderAvatarUrl + ", "
                + messageState + ", " + direction + ", " + isRead + ", " + attachments + "]";
    }

    @Override
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public boolean equals(Object message) {
        if (message != null && message instanceof Message) {
            Message comparedMessage = (Message) message;
            if (String.valueOf(comparedMessage.getId()).equals(String.valueOf(getId()))
                    && String.valueOf(comparedMessage.getChatId()).equals(String.valueOf
                    (getChatId()))
                    && String.valueOf(comparedMessage.getSenderName()).equals(String.valueOf
                    (getSenderName()))
                    && String.valueOf(comparedMessage.getSenderAvatarUrl()).equals(String.valueOf
                    (getSenderAvatarUrl()))
                    && String.valueOf(comparedMessage.getBody()).equals(String.valueOf(getBody()))
                    && comparedMessage.getMessagedAt() == getMessagedAt()
                    && comparedMessage.getMessageState() == getMessageState()
                    && comparedMessage.getMessageDirection() == getMessageDirection()
                    && comparedMessage.isInbound() == isInbound()
                    && comparedMessage.isRead() == isRead()
                    && comparedMessage.getReadAt() == getReadAt()
                    && comparedMessage.getAttachments() != null
                    && comparedMessage.getAttachments().size() == getAttachments().size()
                    && comparedMessage.getActions() != null
                    && comparedMessage.getActions().size() == getActions().size()) {
                for (int i = 0; i < comparedMessage.getAttachments().size(); i++) {
                    if (!(comparedMessage.getAttachments().get(i).equals(getAttachments().get(i))))
                        return false;
                }
                for (int i = 0; i < comparedMessage.getActions().size(); i++) {
                    if (!(comparedMessage.getActions().get(i).equals(getActions().get(i))))
                        return false;
                }
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    @Override
    public int hashCode() {
        if (getId() != null)
            return getId().hashCode();
        else
            return -1;
    }

    public enum MessageState {
        STAY_OFFLINE, READY_TO_BE_SENT, SENT, READY_TO_BE_SYNCED, SYNCED, NOT_AVAILABLE
    }

    public enum Direction {
        INBOUND("inbound"), OUTBOUND("outbound"), NOT_AVAILABLE("not-available");
        private final String direction;

        Direction(String direction) {
            this.direction = direction;
        }

        @NotNull
        @Override
        public String toString() {
            return direction;
        }
    }

    public static class Comparator implements java.util.Comparator<Message>, Serializable {
        public static final int COMPARE_BY_CONVERSATION_ID = 1;
        public static final int COMPARE_BY_DATE = 2;

        private int compareBy = COMPARE_BY_DATE;

        public Comparator() {

        }

        public Comparator(int compareBy) {
            this.compareBy = compareBy;
        }

        @Override
        public int compare(Message lhs, Message rhs) {
            if (lhs == null || rhs == null) return 0;
            switch (compareBy) {
                case COMPARE_BY_CONVERSATION_ID:
                    if (lhs.getChatId() != null && rhs.getChatId() != null) {
                        return lhs.getChatId().compareTo(rhs.getChatId());
                    }
                case COMPARE_BY_DATE:
                    Date lhsDate = new Date(lhs.getMessagedAt());
                    Date rhsDate = new Date(rhs.getMessagedAt());
                    return lhsDate.compareTo(rhsDate);
            }
            throw new IllegalStateException("Message comparator wasn't provided comparison " +
                    "messageIssueType");
        }
    }

    private void writeObject(java.io.ObjectOutputStream stream) throws java.io.IOException {
        throw new java.io.NotSerializableException(getClass().getName());
    }

    private void readObject(java.io.ObjectInputStream stream) throws java.io.IOException,
            ClassNotFoundException {
        throw new java.io.NotSerializableException(getClass().getName());
    }
}