package com.instabug.chat.model;

import android.annotation.SuppressLint;
import android.text.TextUtils;
import android.webkit.MimeTypeMap;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringDef;

import com.instabug.library.internal.storage.cache.Cacheable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.ArrayList;

import static com.instabug.chat.model.Attachment.AttachmentState.STATE_NOT_AVAILABLE;
import static com.instabug.chat.model.Attachment.AttachmentState.STATE_OFFLINE;
import static com.instabug.chat.model.Attachment.AttachmentState.STATE_SYNCED;
import static com.instabug.chat.model.Attachment.AttachmentType.TYPE_AUDIO;
import static com.instabug.chat.model.Attachment.AttachmentType.TYPE_IMAGE;
import static com.instabug.chat.model.Attachment.AttachmentType.TYPE_IMAGE_GALLERY;
import static com.instabug.chat.model.Attachment.AttachmentType.TYPE_IMAGE_SCREENSHOT;
import static com.instabug.chat.model.Attachment.AttachmentType.TYPE_NOT_AVAILABLE;
import static com.instabug.chat.model.Attachment.AttachmentType.TYPE_VIDEO;
import static com.instabug.chat.model.Attachment.AttachmentType.TYPE_VIDEO_GALLERY;
import static com.instabug.chat.model.Attachment.AttachmentType.TYPE_VIDEO_RECORD;

public class Attachment implements Cacheable, Serializable {

    // keys used by toJson() & fromJson() methods
    static final String KEY_NAME = "name";
    static final String KEY_LOCALE_PATH = "local_path";
    static final String KEY_URL = "url";
    static final String KEY_TYPE = "type";
    static final String KEY_ATTACHMENT_STATE = "attachment_state";
    static final String KEY_VIDEO_ENCODED = "video_encoded";
    static final String KEY_DURATION = "duration";

    @Nullable
    private String name;
    @Nullable
    private String localPath;
    @Nullable
    private String url;
    @Nullable
    @AttachmentType
    private String type;
    @AttachmentState
    @Nullable
    private String state;
    private boolean isVideoEncoded = false;
    @Nullable
    private String duration;

    public Attachment() {
        setType(TYPE_NOT_AVAILABLE);
        setState(STATE_NOT_AVAILABLE);
    }

    public static JSONArray toJson(ArrayList<Attachment> attachments) throws JSONException {
        JSONArray attachmentsJsonArray = new JSONArray();
        for (int i = 0; i < attachments.size(); i++) {
            attachmentsJsonArray.put(new JSONObject(attachments.get(i).toJson()));
        }
        return attachmentsJsonArray;
    }

    public static ArrayList<Attachment> fromJson(JSONArray attachmentsJsonArray) throws
            JSONException {
        ArrayList<Attachment> attachments = new ArrayList<>();
        for (int i = 0; i < attachmentsJsonArray.length(); i++) {
            Attachment attachment = new Attachment();
            attachment.fromJson(attachmentsJsonArray.getJSONObject(i).toString());
            attachments.add(attachment);
        }
        return attachments;
    }

    @Nullable
    public String getName() {
        return name;
    }

    public Attachment setName(@Nullable String name) {
        this.name = name;
        return this;
    }

    @Nullable
    public String getLocalPath() {
        return localPath;
    }

    public Attachment setLocalPath(@Nullable String localPath) {
        this.localPath = localPath;
        return this;
    }

    @Nullable
    public String getUrl() {
        return url;
    }

    public Attachment setUrl(String url) {
        this.url = url;
        return this;
    }

    @AttachmentType
    @Nullable
    public String getType() {
        return type;
    }

    public Attachment setType(@AttachmentType String type) {
        this.type = type;
        return this;
    }

    public boolean isVideoEncoded() {
        return isVideoEncoded;
    }

    public Attachment setVideoEncoded(boolean videoEncoded) {
        isVideoEncoded = videoEncoded;
        return this;
    }

    @Nullable
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String getFileType() {
        String fileType;
        String extension = MimeTypeMap.getFileExtensionFromUrl(getName());
        if (extension != null && !TextUtils.isEmpty(extension)) {
            fileType = MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension);
            return (fileType != null && !fileType.equals("")) ? fileType : getType();
        } else {
            return getType();
        }
    }

    @Nullable
    @AttachmentState
    public String getState() {
        return state;
    }

    public Attachment setState(@AttachmentState String state) {
        this.state = state;
        return this;
    }

    @Nullable
    public String getDuration() {
        return duration;
    }

    public void setDuration(@Nullable String duration) {
        this.duration = duration;
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String toJson() throws JSONException {
        JSONObject attachment = new JSONObject();
        attachment.put(KEY_NAME, getName())
                .put(KEY_LOCALE_PATH, getLocalPath())
                .put(KEY_URL, getUrl())
                .put(KEY_TYPE, getType())
                .put(KEY_VIDEO_ENCODED, isVideoEncoded())
                .put(KEY_DURATION, getDuration());
        if (getState() != null) {
            attachment.put(KEY_ATTACHMENT_STATE, getState().toString());
        }
        return attachment.toString();
    }

    @Override
    @SuppressLint("NULL_DEREFERENCE")
    public void fromJson(String attachmentAsJson) throws JSONException {
        JSONObject attachmentJsonObject = new JSONObject(attachmentAsJson);
        if (attachmentJsonObject.has(KEY_NAME))
            setName(attachmentJsonObject.getString(KEY_NAME));
        if (attachmentJsonObject.has(KEY_LOCALE_PATH))
            setLocalPath(attachmentJsonObject.getString(KEY_LOCALE_PATH));
        if (attachmentJsonObject.has(KEY_URL))
            setUrl(attachmentJsonObject.getString(KEY_URL));
        if (attachmentJsonObject.has(KEY_TYPE)) {
            switch (attachmentJsonObject.getString(KEY_TYPE)) {
                case TYPE_IMAGE_GALLERY:
                    setType(TYPE_IMAGE_GALLERY);
                    break;
                case TYPE_IMAGE_SCREENSHOT:
                    setType(TYPE_IMAGE_SCREENSHOT);
                    break;
                case TYPE_AUDIO:
                    setType(TYPE_AUDIO);
                    break;
                case TYPE_VIDEO_RECORD:
                    setType(TYPE_VIDEO_RECORD);
                    break;
                case TYPE_VIDEO_GALLERY:
                    setType(TYPE_VIDEO_GALLERY);
                    break;
                case TYPE_VIDEO:
                    setType(TYPE_VIDEO);
                    break;
                case TYPE_IMAGE:
                    setType(TYPE_IMAGE);
                    break;
                default:
                    setType(TYPE_NOT_AVAILABLE);
                    break;
            }
        }
        if (attachmentJsonObject.has(KEY_ATTACHMENT_STATE)) {

            switch (attachmentJsonObject.getString(KEY_ATTACHMENT_STATE)) {
                case STATE_OFFLINE:
                    setState(STATE_OFFLINE);
                    break;
                case STATE_SYNCED:
                    setState(STATE_SYNCED);
                    break;
                default:
                    setState(STATE_NOT_AVAILABLE);
                    break;
            }
        }

        if (attachmentJsonObject.has(KEY_VIDEO_ENCODED))
            setVideoEncoded(attachmentJsonObject.getBoolean(KEY_VIDEO_ENCODED));
        if (attachmentJsonObject.has(KEY_DURATION))
            setDuration(attachmentJsonObject.getString(KEY_DURATION));
    }

    @Override
    @NonNull
    public String toString() {
        return "Name: " + getName() + ", Local Path: " + getLocalPath() + ", Type: " + getType()
                + ", Url: " + getUrl() + ", Attachment State: " + getState();
    }

    @Override
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public boolean equals(Object attachment) {
        if (attachment != null && attachment instanceof Attachment) {
            Attachment comparedAttachment = (Attachment) attachment;
            return String.valueOf(comparedAttachment.getName()).equals(String.valueOf(getName()))
                    && String.valueOf(comparedAttachment.getLocalPath()).equals(String.valueOf
                    (getLocalPath()))
                    && String.valueOf(comparedAttachment.getUrl()).equals(String.valueOf(getUrl()))
                    && (comparedAttachment.getType() != null && getType() != null)
                    && comparedAttachment.getType().equals(getType())
                    && (comparedAttachment.getState() != null && getState() != null)
                    && comparedAttachment.getState().equals(getState())
                    && comparedAttachment.isVideoEncoded() == isVideoEncoded()
                    && String.valueOf(comparedAttachment.getDuration()).equals(String.valueOf
                    (getDuration()));
        } else {
            return false;
        }
    }

    @Override
    public int hashCode() {
        if (getName() != null)
            return getName().hashCode();
        else
            return -1;
    }

    @Retention(RetentionPolicy.SOURCE)
    @StringDef({
            TYPE_IMAGE, TYPE_IMAGE_GALLERY, TYPE_IMAGE_SCREENSHOT,
            TYPE_VIDEO_RECORD, TYPE_VIDEO_GALLERY, TYPE_VIDEO,
            TYPE_AUDIO,
            TYPE_NOT_AVAILABLE})
    public @interface AttachmentType {
        // keys used to restrict attachment type
        String TYPE_IMAGE = "image";
        String TYPE_IMAGE_GALLERY = "image_gallery";
        String TYPE_IMAGE_SCREENSHOT = "extra_image";
        String TYPE_VIDEO_GALLERY = "video_gallery";
        String TYPE_VIDEO_RECORD = "extra_video";
        String TYPE_VIDEO = "video";
        String TYPE_AUDIO = "audio";
        String TYPE_NOT_AVAILABLE = "not_available";
    }

    @Retention(RetentionPolicy.SOURCE)
    @StringDef({
            STATE_OFFLINE,
            STATE_SYNCED,
            STATE_NOT_AVAILABLE})
    public @interface AttachmentState {
        // keys used to restrict attachment state
        String STATE_OFFLINE = "offline";
        String STATE_SYNCED = "synced";
        String STATE_NOT_AVAILABLE = "not_available";
    }
}
