package com.instabug.chat.annotation;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.drawable.ShapeDrawable;
import android.graphics.drawable.shapes.RectShape;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Parcelable;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;
import androidx.core.view.AccessibilityDelegateCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;

import com.instabug.bug.R;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.AttrResolver;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.DrawableUtils;
import com.instabug.library.view.IconView;
import com.instabug.library.view.ViewUtils;

/**
 * Created by tarek on 5/15/17.
 */

public class AnnotationLayout extends LinearLayout implements View.OnClickListener {

    private static final String ANNOTATION_LAYOUT = "instabug_annotation_layout";
    private static final String DRAWING_MODE = "drawingMode";
    @Nullable
    private AnnotationView mAnnotationView;
    @Nullable
    private ColorPickerPopUpView colorPicker;
    private int tintingColor;
    @Nullable
    private LinearLayout annotationActionsContainer;
    @Nullable
    private RelativeLayout iconBrushLayout;
    @Nullable
    private ImageView iconBrush, iconMagnify, iconBlur, iconUndo;
    @Nullable
    private View brushIndicator;
    @Nullable
    private View border;
    @Nullable
    private ShapeSuggestionsLayout shapeSuggestionsLayout;

    public AnnotationLayout(Context context) {
        this(context, null);
    }

    public AnnotationLayout(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public AnnotationLayout(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initViews();
        configureViews();
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public AnnotationLayout(Context context, @Nullable AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
        initViews();
        configureViews();
    }

    /**
     * Initialize view
     */
    private void initViews() {
        //Inflate xml resource, pass "this" as the parent, we use <merge> tag in xml to avoid
        //redundant parent, otherwise a LinearLayout will be added to this LinearLayout ending up
        //with two view groups
        inflate(getContext(), R.layout.instabug_annotation_view, this);

        annotationActionsContainer = findViewById(R.id.instabug_annotation_actions_container);
        shapeSuggestionsLayout = findViewById(R.id.shapeSuggestionsLayout);

        iconBrushLayout = findViewById(R.id.icon_brush_layout);
        iconBrush = findViewById(R.id.icon_brush);
        iconMagnify = findViewById(R.id.icon_magnify);
        iconBlur = findViewById(R.id.icon_blur);
        iconUndo = findViewById(R.id.icon_undo);

        border = findViewById(R.id.instabug_annotation_image_border);
        mAnnotationView = findViewById(R.id.instabug_annotation_image);
        colorPicker = findViewById(R.id.instabug_color_picker);
        brushIndicator = findViewById(R.id.brush_indicator);
    }

    private void configureViews() {
        if (shapeSuggestionsLayout != null) {
            shapeSuggestionsLayout.setOnShapeSelectedListener(index -> {
                if (index == 1 && mAnnotationView != null) {
                    mAnnotationView.setRecognizedShape();
                }
            });
        }

        setAnnotationButtonsEnabled(false);
        configureBrushIconAccessibility();
        configureAnnotationView(mAnnotationView);

        if (colorPicker != null) {
            colorPicker.setPopUpBackgroundColor(
                    AttrResolver.resolveAttributeColor(getContext(), R.attr.ib_annotation_color_picker_bg_color)
            );
        }

        registerAnnotationIconsClickListeners();

        setViewSelector(iconMagnify);
        setViewSelector(iconUndo);
        tintingColor = ContextCompat.getColor(getContext(), R.color.ib_core_annotation_tinting_color);
    }

    private void configureAnnotationView(@Nullable AnnotationView currentAnnotationView) {
        if (currentAnnotationView == null)
            return;

        currentAnnotationView.setDrawingMode(AnnotationView.DrawingMode.DRAW_PATH);

        if (iconBrush != null) {
            DrawableUtils.setDrawableTintColor(iconBrush, InstabugCore.getPrimaryColor());
        }
        if (colorPicker != null) {
            currentAnnotationView.setDrawingColor(colorPicker.getSelectedColor());
        }

        currentAnnotationView.setOnActionDownListener(() -> {
            if (colorPicker != null && colorPicker.getVisibility() == View.VISIBLE) {
                colorPicker.setVisibility(View.GONE);
            }
            if (shapeSuggestionsLayout != null) {
                shapeSuggestionsLayout.hide();
            }
        });

        currentAnnotationView.setOnPathRecognizedListener(this::showShapeSuggestions);
        currentAnnotationView.setOnNewMagnifierAddingAbilityChangedListener(ability -> {
            if (iconMagnify != null) {
                iconMagnify.setEnabled(ability);
            }
        });

        if (colorPicker == null)
            return;

        colorPicker.setOnColorSelectionListener((color, index) -> {
            currentAnnotationView.setDrawingColor(color);
            if (colorPicker != null) {
                colorPicker.setVisibility(View.GONE);
            }
            if (brushIndicator != null) {
                brushIndicator.setBackgroundColor(color);
            }
        });

    }

    // Shapes are added in the following order (original "hand drawn" then recognized shape)
    private void showShapeSuggestions(final Path... paths) {
        if (shapeSuggestionsLayout != null) {
            shapeSuggestionsLayout.removeAllViews();
            for (int index = 0; index < paths.length; index++) {
                shapeSuggestionsLayout.addShapeSuggestion(index == 0 ?
                                R.string.ibg_bug_annotation_original_shape_content_description
                                : R.string.ibg_bug_annotation_recognized_shape_content_description,
                        paths[index]);
            }
            shapeSuggestionsLayout.show();
        }
    }

    private void setBorder() {
        final int borderThickness = ViewUtils.convertDpToPx(getContext(), 4);
        final int padding = ViewUtils.convertDpToPx(getContext(), 2);
        ShapeDrawable borderBackground = new ShapeDrawable();
        borderBackground.setShape(new RectShape());
        borderBackground.getPaint().setColor(InstabugCore.getPrimaryColor());
        borderBackground.getPaint().setStyle(Paint.Style.STROKE);
        borderBackground.getPaint().setStrokeWidth(borderThickness);
        if (border != null) {
            border.setPadding(padding, padding, padding, padding);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                border.setBackground(borderBackground);
            } else {
                border.setBackgroundDrawable(borderBackground);
            }
        }
    }

    private void setViewSelector(@Nullable final ImageView viewSelector) {
        if (viewSelector != null) {
            viewSelector.setOnTouchListener((view, event) -> {
                if (event.getAction() == MotionEvent.ACTION_DOWN) {
                    DrawableUtils.setDrawableTintColor(viewSelector, InstabugCore.getPrimaryColor());
                } else if (event.getAction() == MotionEvent.ACTION_UP) {
                    DrawableUtils.setDrawableTintColor(viewSelector, tintingColor);
                }
                return false;
            });
        }
    }

    @Override
    public void onClick(View v) {
        //Hide shapeSuggestionsLayout with clicking any bottom icons.
        if (shapeSuggestionsLayout != null) {
            shapeSuggestionsLayout.hide();
        }

        int viewId = v.getId();

        if (viewId == R.id.icon_brush_layout) {

            switchColorPickerVisibility();
            if (mAnnotationView != null) {
                mAnnotationView.setDrawingMode(AnnotationView.DrawingMode.DRAW_PATH);
            }
            resetColorSelection();
            DrawableUtils.setDrawableTintColor(iconBrush, InstabugCore.getPrimaryColor());
        } else if (viewId == R.id.icon_magnify) {
            if (mAnnotationView != null) {
                mAnnotationView.addMagnifier();
            }
            hideColorPicker();
        } else if (viewId == R.id.icon_blur) {
            if (mAnnotationView != null) {
                mAnnotationView.setDrawingMode(AnnotationView.DrawingMode.DRAW_BLUR);
            }
            resetColorSelection();
            DrawableUtils.setDrawableTintColor(iconBlur, InstabugCore.getPrimaryColor());
            hideColorPicker();
        } else if (viewId == R.id.icon_undo) {
            if (mAnnotationView != null) {
                mAnnotationView.undo();
            }
            hideColorPicker();
        }
    }

    private void switchColorPickerVisibility() {
        if (colorPicker != null) {
            colorPicker.setVisibility(colorPicker.getVisibility() == View.VISIBLE ? View.GONE : View.VISIBLE);
        }
    }

    private void hideColorPicker() {
        if (colorPicker != null && colorPicker.getVisibility() == View.VISIBLE) {
            colorPicker.setVisibility(View.GONE);
        }
    }

    private void resetColorSelection() {
        if (annotationActionsContainer != null) {
            int annotationActionsCount = annotationActionsContainer.getChildCount();
            for (int i = 0; i < annotationActionsCount; i++) {
                if (annotationActionsContainer.getChildAt(i) instanceof IconView) {
                    ((TextView) annotationActionsContainer.getChildAt(i)).setTextColor(tintingColor);
                }
            }
        }
        DrawableUtils.setDrawableTintColor(iconBrush, tintingColor);
        DrawableUtils.setDrawableTintColor(iconBlur, tintingColor);
    }

    /**
     * Set the uri of the image which you want to annotate on it.
     *
     * @param imageUri The image uri.
     */
    public void setBaseImage(@NonNull Uri imageUri, @Nullable final Runnable runnable) {
        if (imageUri.getPath() != null && mAnnotationView != null) {
            BitmapUtils.loadBitmap(imageUri.getPath(), mAnnotationView, () -> {
                setAnnotationButtonsEnabled(true);
                setBorder();
                if (runnable != null) {
                    runnable.run();
                }
            });
        }
    }

    public void setBitmap(Bitmap bitmap) {
        if (mAnnotationView != null) {
            mAnnotationView.setImageBitmap(bitmap);
        }
        setAnnotationButtonsEnabled(true);
        setBorder();
    }

    private void registerAnnotationIconsClickListeners() {
        if (iconBrushLayout != null) {
            iconBrushLayout.setOnClickListener(this);
        }
        if (iconMagnify != null) {
            iconMagnify.setOnClickListener(this);
        }
        if (iconBlur != null) {
            iconBlur.setOnClickListener(this);
        }
        if (iconUndo != null) {
            iconUndo.setOnClickListener(this);
        }
    }

    private void setAnnotationButtonsEnabled(boolean isEnabled) {
        if (iconBrush != null) {
            iconBrush.setEnabled(isEnabled);
        }
        if (iconMagnify != null) {
            iconMagnify.setEnabled(isEnabled);
        }
        if (iconBlur != null) {
            iconBlur.setEnabled(isEnabled);
        }
        if (iconUndo != null) {
            iconUndo.setEnabled(isEnabled);
        }
    }

    private void configureBrushIconAccessibility() {
        if (iconBrush != null && AccessibilityUtils.isTalkbackEnabled()) {
            ViewCompat.setAccessibilityDelegate(iconBrush, new AccessibilityDelegateCompat() {
                @Override
                public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                    super.onInitializeAccessibilityNodeInfo(host, info);
                    info.setRoleDescription("Button");
                }
            });
        }
    }

    /**
     * @return a bitmap from the annotated image.
     */
    @Nullable
    public Bitmap getAnnotatedBitmap() {
        if (mAnnotationView != null) {
            return mAnnotationView.toBitmap();
        }
        return null;
    }

    @Nullable
    @Override
    protected Parcelable onSaveInstanceState() {
        Bundle bundle = new Bundle();
        bundle.putParcelable(ANNOTATION_LAYOUT, super.onSaveInstanceState());
        if (mAnnotationView != null) {
            bundle.putSerializable(DRAWING_MODE, mAnnotationView.getDrawingMode());
        }
        return bundle;
    }

    @Override
    protected void onRestoreInstanceState(Parcelable state) {
        if (mAnnotationView != null && colorPicker != null) {
            mAnnotationView.setDrawingColor(colorPicker.getSelectedColor());

            if (state instanceof Bundle) {
                Bundle bundle = (Bundle) state;
                AnnotationView.DrawingMode drawingMode = (AnnotationView.DrawingMode)
                        bundle.getSerializable(DRAWING_MODE);
                resetColorSelection();
                if (drawingMode == AnnotationView.DrawingMode.DRAW_BLUR) {
                    DrawableUtils.setDrawableTintColor(iconBlur, InstabugCore.getPrimaryColor());
                } else {
                    DrawableUtils.setDrawableTintColor(iconBrush, InstabugCore.getPrimaryColor());
                }
                state = bundle.getParcelable(ANNOTATION_LAYOUT);
            }
            if (state != null) {
                super.onRestoreInstanceState(state);
            }
        }
    }
}