package com.instabug.bug.onboardingbugreporting.utils

import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Intent
import android.content.pm.PackageManager
import android.os.Build
import android.os.Handler
import android.os.Looper
import com.instabug.bug.Constants
import com.instabug.bug.invocation.InvocationManager
import com.instabug.bug.onboardingbugreporting.OnboardingActivity
import com.instabug.bug.onboardingbugreporting.State
import com.instabug.bug.settings.BugSettings
import com.instabug.library.Instabug
import com.instabug.library.PresentationManager
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventSubscriber
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.eventbus.coreeventbus.TYPE_SESSION
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.invocation.InstabugInvocationEvent
import com.instabug.library.settings.SettingsManager
import com.instabug.library.tracking.InstabugInternalTrackingDelegate
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.util.threading.runDefensive


class WelcomeMessageHelperApiImp : WelcomeMessageHelperApiContract {
    private var welcomeMessageIBGDisposable: IBGDisposable? = null

    override fun showIntroMessage() {
        val shouldAutoShowOnBoarding = SettingsManager.getInstance().shouldAutoShowOnboarding()
        InstabugSDKLogger.v(
            Constants.LOG_TAG,
            "Checking if should show welcome message, \n" +
                "Should show $shouldAutoShowOnBoarding, \n" +
                "Welcome message state ${BugSettings.getInstance().welcomeMessageState}"
        )
        if (shouldAutoShowOnBoarding) {
            val currentLooper = Looper.myLooper() ?: Looper.getMainLooper()
            Handler(currentLooper).postDelayed({
                val currentInstabugInvocationEvents =
                    InvocationManager.getInstance().currentInstabugInvocationEvents
                if (BugSettings.getInstance().welcomeMessageState != State.DISABLED &&
                    !currentInstabugInvocationEvents.isNullOrEmpty() &&
                    isCurrentInvocationEventNotNone()
                ) {
                    InstabugSDKLogger.v(
                        Constants.LOG_TAG,
                        "get welcome message " + BugSettings.getInstance().welcomeMessageState
                    )
                    showWelcomeMessage(BugSettings.getInstance().welcomeMessageState)
                }
            }, 10000)
        }
    }

    override fun isCurrentInvocationEventNotNone(): Boolean {
        val instabugInvocationEvents =
            InvocationManager.getInstance().currentInstabugInvocationEvents
        return instabugInvocationEvents != null && (instabugInvocationEvents.size != 1 || instabugInvocationEvents[0] != InstabugInvocationEvent.NONE)
    }


    override fun openOnBoardingActivity(@State welcomeMessageState: Int) {
        PresentationManager.getInstance()
            ?.show(openOnBoardingRunnable(welcomeMessageState))
    }

    private fun openOnBoardingRunnable(welcomeMessageState: Int) = runDefensive {
        val target: Activity = InstabugInternalTrackingDelegate.getInstance()
            ?.targetActivity ?: return@runDefensive

        if (!target.isFinishing) {
            val onBoardingIntent = OnboardingActivity.getIntent(target, welcomeMessageState)
            val targetResolveActivity = resolveActivity(target, onBoardingIntent)
            if (targetResolveActivity) {
                target.startActivity(onBoardingIntent)
            } else {
                reportOnBoardingError(welcomeMessageState)
            }
        }
    }

    private fun resolveActivity(
        target: Activity,
        onBoardingIntent: Intent
    ) = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
        target.packageManager?.resolveActivity(
            onBoardingIntent,
            PackageManager.ResolveInfoFlags.of(0L)
        ) != null
    } else {
        target.packageManager?.resolveActivity(onBoardingIntent, 0) != null
    }

    private fun reportOnBoardingError(@State welcomeMessageState: Int) {
        val customizedErrorMessage =
            "Failed to show welcome message with state: $welcomeMessageState\ndue to error at: "
        val throwable = ActivityNotFoundException("OnboardingActivity")
        IBGDiagnostics.reportNonFatal(
            throwable,
            customizedErrorMessage + "\n" + (throwable.message.orEmpty())
        )
    }

    override fun showWelcomeMessage(@State welcomeMessageState: Int) {
        if (!Instabug.isEnabled()) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG,
                "Cannot show intro message while SDK is Disabled"
            )
            return
        }

        if (welcomeMessageState == State.DISABLED) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG,
                "Cannot show onboarding message while WelcomeMessageState is DISABLED"
            )
            return
        }

        val currentInstabugInvocationEvents =
            InvocationManager.getInstance().currentInstabugInvocationEvents
        if ((currentInstabugInvocationEvents.isNullOrEmpty())
            || !isCurrentInvocationEventNotNone()
        ) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG,
                "Cannot show onboarding message while invocation event is NONE"
            )
            return
        }

        when {
            !InstabugCore.isAppOnForeground() && welcomeMessageIBGDisposable == null -> {
                // subscribe to core events session only
                welcomeMessageIBGDisposable = IBGCoreEventSubscriber.subscribe { ibgSdkCoreEvent ->
                    if (ibgSdkCoreEvent.type == TYPE_SESSION) {
                        if (ibgSdkCoreEvent is IBGSdkCoreEvent.Session.SessionStarted && !InstabugCore.isForegroundBusy()) {
                            openOnBoardingActivity(welcomeMessageState)
                        }
                        unsubscribeFromIBGWelcomeMessageEvents()
                    }
                }
            }

            InstabugCore.isForegroundNotBusy() -> {
                openOnBoardingActivity(welcomeMessageState)
            }

            welcomeMessageIBGDisposable == null -> {
                // subscribe to core events invocation and foreground status change
                welcomeMessageIBGDisposable = IBGCoreEventSubscriber.subscribe { coreEvent ->
                    handleCoreEvents(welcomeMessageState, coreEvent)
                }
            }
        }
    }

    override fun handleCoreEvents(@State welcomeMessageState: Int, coreEvent: IBGSdkCoreEvent?) {
        if (coreEvent is IBGSdkCoreEvent.ForegroundAvailable) {
            PoolProvider.postDelayedTask({ handleForegroundAvailable(welcomeMessageState) }, 1000)
        }
    }

    override fun handleForegroundAvailable(@State welcomeMessageState: Int) {
        if (InstabugCore.isForegroundBusy()) return
        PoolProvider.postMainThreadTask { openOnBoardingActivity(welcomeMessageState) }
        unsubscribeFromIBGWelcomeMessageEvents()
    }

    override fun unsubscribeFromIBGWelcomeMessageEvents() {
        welcomeMessageIBGDisposable?.dispose()
        welcomeMessageIBGDisposable = null
    }

}

