package com.instabug.bug.invocation.invocationdialog;

import static com.instabug.library.core.plugin.PluginPromptOption.PromptOptionIdentifier.ASK_QUESTION;
import static com.instabug.library.core.plugin.PluginPromptOption.PromptOptionIdentifier.BUG_REPORT;
import static com.instabug.library.core.plugin.PluginPromptOption.PromptOptionIdentifier.CHAT_LIST;
import static com.instabug.library.core.plugin.PluginPromptOption.PromptOptionIdentifier.FEEDBACK;
import static com.instabug.library.util.ThemeApplier.applyPrimaryTextStyle;
import static com.instabug.library.util.ThemeApplier.applySecondaryTextStyle;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.PorterDuff;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AbsListView;
import android.widget.BaseAdapter;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.Nullable;
import androidx.core.view.AccessibilityDelegateCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;

import com.instabug.bug.R;
import com.instabug.library.Instabug;
import com.instabug.library.model.IBGTheme;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.InstabugAppData;

import java.util.ArrayList;

public class InstabugDialogAdapter extends BaseAdapter {

    private ArrayList<InstabugDialogItem> dialogListItems;

    @Nullable
    private IBGTheme theme;

    @Override
    public boolean hasStableIds() {
        return true;
    }

    public InstabugDialogAdapter(@Nullable IBGTheme theme) {
        dialogListItems = new ArrayList<>();
        this.theme = theme;
    }

    public static void setMargins(View v, int l, int t, int r, int b) {
        if (v != null && v.getLayoutParams() instanceof ViewGroup.MarginLayoutParams) {
            ViewGroup.MarginLayoutParams p = (ViewGroup.MarginLayoutParams) v.getLayoutParams();
            p.setMargins(l, t, r, b);
            v.requestLayout();
        }
    }

    @Override
    public int getCount() {
        return dialogListItems.size();
    }

    @Override
    public InstabugDialogItem getItem(int position) {
        return dialogListItems.get(position);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_RETURN_ANNOTATION")
    @Nullable
    @Override
    public View getView(final int position, View root, final ViewGroup parent) {
        View convertView = root;
        if (parent != null) {
            ViewHolder viewHolder;
            if (convertView == null) {
                convertView = LayoutInflater.from(parent.getContext())
                        .inflate(R.layout.ib_dialog_list_item, parent, false);
                viewHolder = new ViewHolder(convertView);
                convertView.setTag(viewHolder);
            } else {
                viewHolder = (ViewHolder) convertView.getTag();
            }

            if (parent instanceof AbsListView && AccessibilityUtils.isTalkbackEnabled()) {
                final View finalConvertView = convertView;
                View.OnClickListener onClickListener = v -> ((AbsListView) parent)
                        .performItemClick(finalConvertView, position, getItemId(position));
                if (viewHolder.root != null) {
                    viewHolder.root.setOnClickListener(onClickListener);
                    ViewCompat.setImportantForAccessibility(viewHolder.root, ViewCompat.IMPORTANT_FOR_ACCESSIBILITY_YES);
                }

            }
            bind(viewHolder, getItem(position));
            return convertView;
        }

        return null;
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void bind(ViewHolder viewHolder, final InstabugDialogItem dialogListItems) {
        if (viewHolder.title != null) {
            viewHolder.title.setText(dialogListItems.getTitle());
        }
        if (viewHolder.description != null) {
            if (TextUtils.isEmpty(dialogListItems.getDescription())) {
                viewHolder.description.setVisibility(View.GONE);
            } else {
                viewHolder.description.setVisibility(View.VISIBLE);
                viewHolder.description.setText(dialogListItems.getDescription());
                // Handle its accessibility
                ViewCompat.setAccessibilityDelegate(viewHolder.description, new AccessibilityDelegateCompat() {

                    @Override
                    public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                        super.onInitializeAccessibilityNodeInfo(host, info);
                        info.setContentDescription(getItemDescription(dialogListItems));
                    }
                });
            }
        }
        if (viewHolder.icon != null) {
            if (dialogListItems.getResDrawable() != 0) {
                viewHolder.icon.setImageResource(dialogListItems.getResDrawable());
                viewHolder.icon.setVisibility(View.VISIBLE);
                viewHolder.icon.getDrawable().setColorFilter(SettingsManager.getInstance().getPrimaryColor(), PorterDuff.Mode.SRC_IN);
            } else {
                viewHolder.icon.setVisibility(View.GONE);
                setMargins(viewHolder.title, 0, 0, 0, 0);
                setMargins(viewHolder.description, 0, 4, 0, 0);
            }
        }
        applyTheme(viewHolder);
    }

    private void applyTheme(ViewHolder viewHolder) {
        applyPrimaryTextStyle(viewHolder.title, theme);
        applySecondaryTextStyle(viewHolder.description, theme);
    }


    private String getItemDescription(InstabugDialogItem dialogListItem) {
        Context context = Instabug.getApplicationContext();
        switch (dialogListItem.getIdentifier()) {
            case BUG_REPORT:
                if (context != null) {
                    return context.getString(R.string.ib_bug_report_bug_description);
                }
            case FEEDBACK:
                if (context != null) {
                    return context.getString(R.string.ib_bug_report_feedback_description);
                }
            case ASK_QUESTION:
                if (context != null) {
                    return context.getString(R.string.ib_bug_report_question_description);
                }
            case CHAT_LIST:
            default:
                return "";
        }
    }

    private String getAppName() {
        if (Instabug.getApplicationContext() != null) {
            return new InstabugAppData(Instabug.getApplicationContext()).getAppName();
        }
        return "";
    }

    public void setPluginPromptOptions(ArrayList<InstabugDialogItem> dialogListItems) {
        this.dialogListItems = dialogListItems;
    }

    private static class ViewHolder {
        @Nullable
        public View root;
        @Nullable
        public ImageView icon;
        @Nullable
        public TextView title;
        @Nullable
        public TextView description;

        ViewHolder(View view) {
            root = view.findViewById(R.id.instabug_prompt_option_container);
            icon = view.findViewById(R.id.instabug_prompt_option_icon);
            title = view.findViewById(R.id.instabug_prompt_option_title);
            description = view.findViewById(R.id.instabug_prompt_option_description);
        }
    }
}
