/*
 * Copyright (c) 2017 Yrom Wang <http://www.yrom.net>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.instabug.bug.internal.video.customencoding;

import android.annotation.TargetApi;
import android.media.MediaCodecInfo;
import android.media.MediaCodecList;
import android.media.MediaFormat;
import android.os.Build;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.Size;

import com.instabug.bug.internal.video.BugsVideoUtils;
import com.instabug.bug.settings.BugSettings;
import com.instabug.library.Constants;
import com.instabug.library.internal.video.customencoding.VideoEncoderConfig;
import com.instabug.library.util.InstabugSDKLogger;

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
public class VideoEncodeConfig {

    private static final int VIDEO_ENCODING_BITRATE = 8 * 1000 * 1000;
    private static final int VIDEO_FRAMES_INTERVAL = 5;
    private static final int VIDEO_FRAME_RATE = 30;
    private static final String VIDEO_MIMETYPE = MediaFormat.MIMETYPE_VIDEO_AVC;

    private final int width;
    private final int height;
    private final int density;
    private final String codecName;
    @Nullable
    private MediaCodecInfo mVideoCodecInfo;

    public VideoEncodeConfig(int width, int height, int density) {
        int calculatedWidth = 0;
        int calculatedHeight = 0;
        double[] dimens = getDimensInBounded(width, height);
        if (dimens != null && dimens.length >= 2) {
            calculatedWidth = (int) dimens[0];
            calculatedHeight = (int) dimens[1];
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG,"Invalid dimensions retrieved");
        }
        this.width = calculatedWidth;
        this.height = calculatedHeight;
        this.density = density;
        MediaCodecInfo videoCodecInfo = getVideoCodecInfo(VIDEO_MIMETYPE);
        this.codecName = videoCodecInfo != null ? videoCodecInfo.getName() : "";
    }

    /**
     * Create the format specifications for the selected video encoder configurations,
     * Those specs are required by the decoder to able to play the result video right.
     */
    MediaFormat toFormat() {
        VideoEncoderConfig externalConfig = BugSettings.getVideoEncoderConfig();
        InstabugSDKLogger.d(Constants.LOG_TAG, "Custom Video Encoder Config: " + externalConfig);

        // Some hardware encoders accept only heights divisible by 16 ¯\_(ツ)_/¯
        MediaFormat format = externalConfig != null ? MediaFormat.createVideoFormat(VIDEO_MIMETYPE, externalConfig.getWidth(), (externalConfig.getHeight() / 16) * 16) : MediaFormat.createVideoFormat(VIDEO_MIMETYPE, width, (height / 16) * 16);
        format.setInteger(MediaFormat.KEY_COLOR_FORMAT, externalConfig != null ? externalConfig.getColorFormat() : MediaCodecInfo.CodecCapabilities.COLOR_FormatSurface);
        format.setInteger(MediaFormat.KEY_BIT_RATE, externalConfig != null ? externalConfig.getBitrate() : VIDEO_ENCODING_BITRATE);
        format.setInteger(MediaFormat.KEY_FRAME_RATE, externalConfig != null ? externalConfig.getFrameRate() : VIDEO_FRAME_RATE);
        format.setInteger(MediaFormat.KEY_I_FRAME_INTERVAL, externalConfig != null ? externalConfig.getIFrameInterval() : VIDEO_FRAMES_INTERVAL);
        return format;
    }

    @NonNull
    @Override
    public String toString() {
        return "VideoEncodeConfig{" +
                "width=" + width +
                ", height=" + height +
                ", bitrate=" + VIDEO_ENCODING_BITRATE +
                ", framerate=" + VIDEO_FRAME_RATE +
                ", iframeInterval=" + VIDEO_FRAMES_INTERVAL +
                ", codecName='" + getCodecName() + '\'' +
                ", mimeType='" + VIDEO_MIMETYPE + '\'' +
                '}';
    }

    @Nullable
    private MediaCodecInfo getVideoCodecInfo(String codecType) {
        if (codecType == null)
            return null;

        if (mVideoCodecInfo == null) {
            mVideoCodecInfo = findEncodersByType(VIDEO_MIMETYPE);
        }

        return mVideoCodecInfo;
    }



    public int getWidth() {
        return width;
    }

    public int getHeight() {
        return height;
    }

    public int getDensity() {
        return density / 4;
    }

    public String getCodecName() {
        VideoEncoderConfig externalConfig = BugSettings.getVideoEncoderConfig();
        return externalConfig != null ? externalConfig.getCodec() : codecName;
    }

    @Nullable
    private MediaCodecInfo findEncodersByType(String mimeType) {
        MediaCodecList codecList = new MediaCodecList(MediaCodecList.ALL_CODECS);
        for (MediaCodecInfo info : codecList.getCodecInfos()) {
            if (!info.isEncoder()) {
                continue;
            }
            try {
                MediaCodecInfo.CodecCapabilities cap = info.getCapabilitiesForType(mimeType);
                if (cap == null) continue;
            } catch (IllegalArgumentException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG,"IllegalArgumentException" + e.getMessage());
                continue;
            }
            return info;
        }

        return null;
    }

    @Size(2)
    @Nullable
    private double[] getDimensInBounded(double width, double height) {
        MediaCodecInfo mediaCodecInfo = getVideoCodecInfo(VIDEO_MIMETYPE);
        if (mediaCodecInfo != null) {
            MediaCodecInfo.VideoCapabilities videoCapabilities = mediaCodecInfo
                    .getCapabilitiesForType(VIDEO_MIMETYPE)
                    .getVideoCapabilities();
            if (videoCapabilities != null) {
                Integer maxSupportedWidth = videoCapabilities.getSupportedWidths().getUpper();
                Integer maxSupportedHeight = videoCapabilities.getSupportedHeights().getUpper();
                if (maxSupportedWidth != null && maxSupportedHeight != null) {
                    return BugsVideoUtils.getDimensInBounded(
                            width,
                            height,
                            maxSupportedWidth,
                            maxSupportedHeight);
                }
            }
        }
        return new double[]{0, 0};
    }
}
