/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hazelcast.shaded.org.apache.calcite.adapter.enumerable;

import com.hazelcast.shaded.org.apache.calcite.plan.Convention;
import com.hazelcast.shaded.org.apache.calcite.plan.RelTraitSet;
import com.hazelcast.shaded.org.apache.calcite.rel.RelNode;
import com.hazelcast.shaded.org.apache.calcite.rel.convert.ConverterRule;
import com.hazelcast.shaded.org.apache.calcite.rel.core.TableModify;
import com.hazelcast.shaded.org.apache.calcite.rel.logical.LogicalTableModify;
import com.hazelcast.shaded.org.apache.calcite.schema.ModifiableTable;

import com.hazelcast.shaded.org.checkerframework.checker.nullness.qual.Nullable;

/** Planner rule that converts a {@link LogicalTableModify} to an {@link EnumerableTableModify}.
 * You may provide a custom config to convert other nodes that extend {@link TableModify}.
 *
 * @see EnumerableRules#ENUMERABLE_TABLE_MODIFICATION_RULE */
public class EnumerableTableModifyRule extends ConverterRule {
  /** Default configuration. */
  public static final Config DEFAULT_CONFIG = Config.INSTANCE
      .withConversion(LogicalTableModify.class, Convention.NONE,
          EnumerableConvention.INSTANCE, "EnumerableTableModificationRule")
      .withRuleFactory(EnumerableTableModifyRule::new);

  /** Creates an EnumerableTableModifyRule. */
  protected EnumerableTableModifyRule(Config config) {
    super(config);
  }

  @Override public @Nullable RelNode convert(RelNode rel) {
    final TableModify modify = (TableModify) rel;
    final ModifiableTable modifiableTable =
        modify.getTable().unwrap(ModifiableTable.class);
    if (modifiableTable == null) {
      return null;
    }
    final RelTraitSet traitSet =
        modify.getTraitSet().replace(EnumerableConvention.INSTANCE);
    return new EnumerableTableModify(
        modify.getCluster(), traitSet,
        modify.getTable(),
        modify.getCatalogReader(),
        convert(modify.getInput(), traitSet),
        modify.getOperation(),
        modify.getUpdateColumnList(),
        modify.getSourceExpressionList(),
        modify.isFlattened());
  }
}
