/*
 * ProGuard -- shrinking, optimization, obfuscation, and preverification
 *             of Java bytecode.
 *
 * Copyright (c) 2002-2021 Guardsquare NV
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package proguard.optimize.evaluation;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import proguard.FullyQualifiedClassNameChecker;
import proguard.classfile.*;
import proguard.classfile.visitor.*;
import proguard.optimize.OptimizationInfoClassFilter;
import proguard.optimize.info.SimpleEnumMarker;

/**
 * This ClassVisitor marks all program classes that it visits as simple enums,
 * if their methods qualify.
 *
 * @author Eric Lafortune
 */
public class SimpleEnumClassChecker
implements   ClassVisitor
{
    private static final Logger logger = LogManager.getLogger(SimpleEnumClassChecker.class);


    private final ClassVisitor  simpleEnumMarker      = new OptimizationInfoClassFilter(
                                                        new SimpleEnumMarker(true));
    private final MemberVisitor instanceMemberChecker = new MemberAccessFilter(0,
                                                                               AccessConstants.STATIC,
                                                        new MemberToClassVisitor(
                                                        new SimpleEnumMarker(false)));


    // Implementations for ClassVisitor.

    @Override
    public void visitAnyClass(Clazz clazz) { }


    @Override
    public void visitProgramClass(ProgramClass programClass)
    {
        // Does the class have the simple enum constructor?
        if (programClass.findMethod(ClassConstants.METHOD_NAME_INIT,
                                    ClassConstants.METHOD_TYPE_INIT_ENUM) != null)
        {
            logger.debug("SimpleEnumClassChecker: [{}] is a candidate simple enum, without extra fields", programClass.getName());

            // Mark it.
            simpleEnumMarker.visitProgramClass(programClass);

            // Unmark it again if it has any instance fields
            // or methods.
            programClass.fieldsAccept(instanceMemberChecker);
            programClass.methodsAccept(instanceMemberChecker);
        }
    }
}
