/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.auth.internal;

import com.google.api.client.auth.openidconnect.IdToken;
import com.google.api.client.auth.openidconnect.IdTokenVerifier;
import com.google.api.client.googleapis.auth.oauth2.GooglePublicKeysManager;
import com.google.api.client.json.webtoken.JsonWebSignature;
import com.google.api.client.util.ArrayMap;
import com.google.api.client.util.Clock;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.firebase.auth.FirebaseAuthException;
import com.google.firebase.auth.internal.KeyManagers;
import com.google.firebase.internal.NonNull;
import java.io.IOException;
import java.math.BigDecimal;
import java.security.GeneralSecurityException;
import java.security.PublicKey;
import java.util.Collection;
import java.util.Collections;

public final class FirebaseTokenVerifier
extends IdTokenVerifier {
    static final String ID_TOKEN_CERT_URL = "https://www.googleapis.com/robot/v1/metadata/x509/securetoken@system.gserviceaccount.com";
    static final String ID_TOKEN_ISSUER_PREFIX = "https://securetoken.google.com/";
    static final String SESSION_COOKIE_CERT_URL = "https://www.googleapis.com/identitytoolkit/v3/relyingparty/publicKeys";
    static final String SESSION_COOKIE_ISSUER_PREFIX = "https://session.firebase.google.com/";
    private static final String FIREBASE_AUDIENCE = "https://identitytoolkit.googleapis.com/google.identity.identitytoolkit.v1.IdentityToolkit";
    private static final String ERROR_INVALID_CREDENTIAL = "ERROR_INVALID_CREDENTIAL";
    private static final String ERROR_RUNTIME_EXCEPTION = "ERROR_RUNTIME_EXCEPTION";
    private static final String PROJECT_ID_MATCH_MESSAGE = "Make sure the %s comes from the same Firebase project as the service account used to authenticate this SDK.";
    private static final String VERIFY_TOKEN_DOCS_MESSAGE = "See %s for details on how to retrieve %s.";
    private static final String ALGORITHM = "RS256";
    private final String projectId;
    private final GooglePublicKeysManager publicKeysManager;
    private final String method;
    private final String shortName;
    private final String articledShortName;
    private final String projectIdMatchMessage;
    private final String verifyTokenMessage;

    private FirebaseTokenVerifier(Builder builder) {
        super((IdTokenVerifier.Builder)builder);
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)builder.projectId) ? 1 : 0) != 0, (Object)"projectId must be set");
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)builder.shortName) ? 1 : 0) != 0, (Object)"shortName must be set");
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)builder.method) ? 1 : 0) != 0, (Object)"method must be set");
        this.projectId = builder.projectId;
        this.shortName = builder.shortName;
        this.articledShortName = this.prefixWithIndefiniteArticle(this.shortName);
        this.method = builder.method;
        this.publicKeysManager = (GooglePublicKeysManager)Preconditions.checkNotNull((Object)builder.publicKeysManager, (Object)"publicKeysManager must be set");
        this.projectIdMatchMessage = String.format(PROJECT_ID_MATCH_MESSAGE, this.shortName);
        this.verifyTokenMessage = String.format(VERIFY_TOKEN_DOCS_MESSAGE, builder.docUrl, this.articledShortName);
    }

    public void verifyTokenAndSignature(IdToken token) throws FirebaseAuthException {
        boolean isLegacyCustomToken;
        IdToken.Payload payload = token.getPayload();
        JsonWebSignature.Header header = token.getHeader();
        String errorMessage = null;
        boolean isCustomToken = payload.getAudience() != null && payload.getAudience().equals(FIREBASE_AUDIENCE);
        boolean bl = isLegacyCustomToken = header.getAlgorithm() != null && header.getAlgorithm().equals("HS256") && payload.get((Object)"v") != null && payload.get((Object)"v").equals(new BigDecimal(0)) && payload.get((Object)"d") != null && payload.get((Object)"d") instanceof ArrayMap && ((ArrayMap)payload.get((Object)"d")).get((Object)"uid") != null;
        if (header.getKeyId() == null) {
            errorMessage = isCustomToken ? String.format("%s expects %s, but was given a custom token.", this.method, this.articledShortName) : (isLegacyCustomToken ? String.format("%s expects %s, but was given a legacy custom token.", this.method, this.articledShortName) : String.format("Firebase %s has no \"kid\" claim.", this.shortName));
        } else if (header.getAlgorithm() == null || !header.getAlgorithm().equals(ALGORITHM)) {
            errorMessage = String.format("Firebase %s has incorrect algorithm. Expected \"%s\" but got \"%s\".", this.shortName, ALGORITHM, header.getAlgorithm());
        } else if (!token.verifyAudience(this.getAudience())) {
            errorMessage = String.format("Firebase %s has incorrect \"aud\" (audience) claim. Expected \"%s\" but got \"%s\". %s", this.shortName, this.concat(this.getAudience()), this.concat(token.getPayload().getAudienceAsList()), this.projectIdMatchMessage);
        } else if (!token.verifyIssuer(this.getIssuers())) {
            errorMessage = String.format("Firebase %s has incorrect \"iss\" (issuer) claim. Expected \"%s\" but got \"%s\". %s", this.shortName, this.concat(this.getIssuers()), token.getPayload().getIssuer(), this.projectIdMatchMessage);
        } else if (payload.getSubject() == null) {
            errorMessage = String.format("Firebase %s has no \"sub\" (subject) claim.", this.shortName);
        } else if (payload.getSubject().isEmpty()) {
            errorMessage = String.format("Firebase %s has an empty string \"sub\" (subject) claim.", this.shortName);
        } else if (payload.getSubject().length() > 128) {
            errorMessage = String.format("Firebase %s has \"sub\" (subject) claim longer than 128 characters.", this.shortName);
        } else if (!token.verifyTime(this.getClock().currentTimeMillis(), this.getAcceptableTimeSkewSeconds())) {
            errorMessage = String.format("Firebase %s has expired or is not yet valid. Get a fresh %s and try again.", this.shortName, this.shortName);
        }
        if (errorMessage != null) {
            throw new FirebaseAuthException(ERROR_INVALID_CREDENTIAL, String.format("%s %s", errorMessage, this.verifyTokenMessage));
        }
        try {
            if (!this.verifySignature(token)) {
                throw new FirebaseAuthException(ERROR_INVALID_CREDENTIAL, String.format("Firebase %s isn't signed by a valid public key. %s", this.shortName, this.verifyTokenMessage));
            }
        }
        catch (IOException | GeneralSecurityException e) {
            throw new FirebaseAuthException(ERROR_RUNTIME_EXCEPTION, "Error while verifying signature.", e);
        }
    }

    private String prefixWithIndefiniteArticle(String word) {
        if ("aeiouAEIOU".indexOf(word.charAt(0)) < 0) {
            return "a " + word;
        }
        return "an " + word;
    }

    private String concat(Collection<String> collection) {
        StringBuilder stringBuilder = new StringBuilder();
        for (String inputLine : collection) {
            stringBuilder.append(inputLine.trim()).append(", ");
        }
        return stringBuilder.substring(0, stringBuilder.length() - 2);
    }

    private boolean verifySignature(IdToken token) throws GeneralSecurityException, IOException {
        for (PublicKey key : this.publicKeysManager.getPublicKeys()) {
            if (!token.verifySignature(key)) continue;
            return true;
        }
        return false;
    }

    public String getProjectId() {
        return this.projectId;
    }

    @NonNull
    public static FirebaseTokenVerifier createIdTokenVerifier(@NonNull String projectId, @NonNull KeyManagers keyManagers, @NonNull Clock clock) {
        return new Builder().setProjectId(ID_TOKEN_ISSUER_PREFIX, projectId).setPublicKeysManager(keyManagers.getIdTokenKeysManager()).setShortName("ID token").setMethod("verifyIdToken()").setDocUrl("https://firebase.google.com/docs/auth/admin/verify-id-tokens").setClock(clock).build();
    }

    @NonNull
    public static FirebaseTokenVerifier createSessionCookieVerifier(@NonNull String projectId, @NonNull KeyManagers keyManagers, @NonNull Clock clock) {
        return new Builder().setProjectId(SESSION_COOKIE_ISSUER_PREFIX, projectId).setPublicKeysManager(keyManagers.getSessionCookieKeysManager()).setShortName("session cookie").setMethod("verifySessionCookie()").setDocUrl("https://firebase.google.com/docs/auth/admin/manage-cookies").setClock(clock).build();
    }

    public static class Builder
    extends IdTokenVerifier.Builder {
        private String projectId;
        private String shortName;
        private String method;
        private String docUrl;
        private GooglePublicKeysManager publicKeysManager;

        public String getProjectId() {
            return this.projectId;
        }

        public Builder setProjectId(String issuerPrefix, String projectId) {
            this.projectId = projectId;
            this.setAudience(Collections.singleton(projectId));
            this.setIssuer(issuerPrefix + projectId);
            return this;
        }

        public Builder setShortName(String shortName) {
            this.shortName = shortName;
            return this;
        }

        public Builder setMethod(String method) {
            this.method = method;
            return this;
        }

        public Builder setDocUrl(String docUrl) {
            this.docUrl = docUrl;
            return this;
        }

        public Builder setClock(Clock clock) {
            return (Builder)super.setClock(clock);
        }

        public Builder setPublicKeysManager(GooglePublicKeysManager publicKeysManager) {
            this.publicKeysManager = publicKeysManager;
            return this;
        }

        public FirebaseTokenVerifier build() {
            return new FirebaseTokenVerifier(this);
        }
    }
}

