/*
 * Copyright 2010-2018 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.serialization.deserialization.builtins

import ksp.org.jetbrains.kotlin.builtins.BuiltInsLoader
import ksp.org.jetbrains.kotlin.builtins.StandardNames
import ksp.org.jetbrains.kotlin.descriptors.ModuleDescriptor
import ksp.org.jetbrains.kotlin.descriptors.NotFoundClasses
import ksp.org.jetbrains.kotlin.descriptors.PackageFragmentProvider
import ksp.org.jetbrains.kotlin.descriptors.PackageFragmentProviderImpl
import ksp.org.jetbrains.kotlin.descriptors.deserialization.AdditionalClassPartsProvider
import ksp.org.jetbrains.kotlin.descriptors.deserialization.ClassDescriptorFactory
import ksp.org.jetbrains.kotlin.descriptors.deserialization.PlatformDependentDeclarationFilter
import ksp.org.jetbrains.kotlin.incremental.components.LookupTracker
import ksp.org.jetbrains.kotlin.name.FqName
import ksp.org.jetbrains.kotlin.resolve.sam.SamConversionResolverImpl
import ksp.org.jetbrains.kotlin.serialization.deserialization.*
import ksp.org.jetbrains.kotlin.storage.StorageManager
import java.io.InputStream

class BuiltInsLoaderImpl : BuiltInsLoader {
    private val resourceLoader = BuiltInsResourceLoader()

    override fun createPackageFragmentProvider(
        storageManager: StorageManager,
        builtInsModule: ModuleDescriptor,
        classDescriptorFactories: Iterable<ClassDescriptorFactory>,
        platformDependentDeclarationFilter: PlatformDependentDeclarationFilter,
        additionalClassPartsProvider: AdditionalClassPartsProvider,
        isFallback: Boolean
    ): PackageFragmentProvider {
        return createBuiltInPackageFragmentProvider(
            storageManager,
            builtInsModule,
            StandardNames.BUILT_INS_PACKAGE_FQ_NAMES,
            classDescriptorFactories,
            platformDependentDeclarationFilter,
            additionalClassPartsProvider,
            isFallback,
            resourceLoader::loadResource
        )
    }

    fun createBuiltInPackageFragmentProvider(
        storageManager: StorageManager,
        module: ModuleDescriptor,
        packageFqNames: Set<FqName>,
        classDescriptorFactories: Iterable<ClassDescriptorFactory>,
        platformDependentDeclarationFilter: PlatformDependentDeclarationFilter,
        additionalClassPartsProvider: AdditionalClassPartsProvider = AdditionalClassPartsProvider.None,
        isFallback: Boolean,
        loadResource: (String) -> InputStream?
    ): PackageFragmentProvider {
        val packageFragments = packageFqNames.mapNotNull { fqName ->
            val resourcePath = BuiltInSerializerProtocol.getBuiltInsFilePath(fqName)
            val inputStream = loadResource(resourcePath)
            inputStream?.let { BuiltInsPackageFragmentImpl.create(fqName, storageManager, module, inputStream, isFallback) }
        }
        val provider = PackageFragmentProviderImpl(packageFragments)

        val notFoundClasses = NotFoundClasses(storageManager, module)

        val components = DeserializationComponents(
            storageManager,
            module,
            DeserializationConfiguration.Default,
            DeserializedClassDataFinder(provider),
            AnnotationAndConstantLoaderImpl(module, notFoundClasses, BuiltInSerializerProtocol),
            provider,
            LocalClassifierTypeSettings.Default,
            ErrorReporter.DO_NOTHING,
            LookupTracker.DO_NOTHING,
            FlexibleTypeDeserializer.ThrowException,
            classDescriptorFactories,
            notFoundClasses,
            ContractDeserializer.DEFAULT,
            additionalClassPartsProvider,
            platformDependentDeclarationFilter,
            BuiltInSerializerProtocol.extensionRegistry,
            samConversionResolver = SamConversionResolverImpl(storageManager, samWithReceiverResolvers = emptyList())
        )

        for (packageFragment in packageFragments) {
            packageFragment.initialize(components)
        }

        return provider
    }
}
