/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.resolve.checkers

import ksp.org.jetbrains.kotlin.config.AnalysisFlags
import ksp.org.jetbrains.kotlin.config.LanguageVersionSettings
import ksp.org.jetbrains.kotlin.config.ReturnValueCheckerMode
import ksp.org.jetbrains.kotlin.descriptors.annotations.KotlinTarget
import ksp.org.jetbrains.kotlin.diagnostics.Errors
import ksp.org.jetbrains.kotlin.name.FqName
import ksp.org.jetbrains.kotlin.psi.KtAnnotated
import ksp.org.jetbrains.kotlin.psi.KtAnnotationEntry
import ksp.org.jetbrains.kotlin.resolve.AdditionalAnnotationChecker
import ksp.org.jetbrains.kotlin.resolve.BindingContext
import ksp.org.jetbrains.kotlin.resolve.BindingTrace

object ReturnValueAnnotationChecker : AdditionalAnnotationChecker {
    val mustUseReturnValueFq = FqName("kotlin.MustUseReturnValue")
    val ignorableFq = FqName("kotlin.IgnorableReturnValue")

    override fun checkEntries(
        entries: List<KtAnnotationEntry>,
        actualTargets: List<KotlinTarget>,
        trace: BindingTrace,
        annotated: KtAnnotated?,
        languageVersionSettings: LanguageVersionSettings,
    ) {
        if (languageVersionSettings.getFlag(AnalysisFlags.returnValueCheckerMode) != ReturnValueCheckerMode.DISABLED) return

        for (entry in entries) {
            val descriptor = trace.get(BindingContext.ANNOTATION, entry) ?: continue
            val name = descriptor.fqName ?: continue
            if (name == mustUseReturnValueFq || name == ignorableFq) {
                trace.report(Errors.IGNORABILITY_ANNOTATIONS_WITH_CHECKER_DISABLED.on(entry))
            }
        }
    }
}
