/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.psi;

import ksp.com.intellij.lang.ASTNode;
import ksp.com.intellij.psi.PsiElement;
import ksp.com.intellij.psi.search.LocalSearchScope;
import ksp.com.intellij.psi.search.SearchScope;
import ksp.com.intellij.psi.util.PsiTreeUtil;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;
import ksp.org.jetbrains.kotlin.lexer.KtTokens;
import ksp.org.jetbrains.kotlin.psi.stubs.KotlinTypeParameterStub;
import ksp.org.jetbrains.kotlin.psi.stubs.elements.KtStubElementTypes;
import ksp.org.jetbrains.kotlin.types.Variance;

public class KtTypeParameter extends KtNamedDeclarationStub<KotlinTypeParameterStub> {

    public KtTypeParameter(@NotNull ASTNode node) {
        super(node);
    }

    public KtTypeParameter(@NotNull KotlinTypeParameterStub stub) {
        super(stub, KtStubElementTypes.TYPE_PARAMETER);
    }

    @Override
    public <R, D> R accept(@NotNull KtVisitor<R, D> visitor, D data) {
        return visitor.visitTypeParameter(this, data);
    }

    @NotNull
    public Variance getVariance() {
        KotlinTypeParameterStub stub = getGreenStub();
        if (stub != null) {
            if (stub.isOutVariance()) return Variance.OUT_VARIANCE;
            if (stub.isInVariance()) return Variance.IN_VARIANCE;
            return Variance.INVARIANT;
        }

        KtModifierList modifierList = getModifierList();
        if (modifierList == null) return Variance.INVARIANT;

        if (modifierList.hasModifier(KtTokens.OUT_KEYWORD)) return Variance.OUT_VARIANCE;
        if (modifierList.hasModifier(KtTokens.IN_KEYWORD)) return Variance.IN_VARIANCE;
        return Variance.INVARIANT;
    }

    @Nullable
    public KtTypeReference setExtendsBound(@Nullable KtTypeReference typeReference) {
        KtTypeReference currentExtendsBound = getExtendsBound();
        if (currentExtendsBound != null) {
            if (typeReference == null) {
                PsiElement colon = findChildByType(KtTokens.COLON);
                if (colon != null) colon.delete();
                currentExtendsBound.delete();
                return null;
            }
            return (KtTypeReference) currentExtendsBound.replace(typeReference);
        }

        if (typeReference != null) {
            PsiElement colon = addAfter(new KtPsiFactory(getProject()).createColon(), getNameIdentifier());
            return (KtTypeReference) addAfter(typeReference, colon);
        }

        return null;
    }

    @Nullable
    public KtTypeReference getExtendsBound() {
        return getStubOrPsiChild(KtStubElementTypes.TYPE_REFERENCE);
    }

    @NotNull
    @Override
    public SearchScope getUseScope() {
        KtTypeParameterListOwner owner = PsiTreeUtil.getParentOfType(this, KtTypeParameterListOwner.class);
        return new LocalSearchScope(owner != null ? owner : this);
    }
}
