/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.light.classes.symbol.parameters

import ksp.com.intellij.lang.Language
import ksp.com.intellij.openapi.util.Pair
import ksp.com.intellij.openapi.util.TextRange
import ksp.com.intellij.psi.*
import ksp.com.intellij.psi.impl.InheritanceImplUtil
import ksp.com.intellij.psi.impl.PsiClassImplUtil
import ksp.com.intellij.psi.impl.light.LightElement
import ksp.com.intellij.psi.javadoc.PsiDocComment
import ksp.com.intellij.psi.search.SearchScope
import ksp.org.jetbrains.kotlin.analysis.api.KaSession
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaModule
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaTypeParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.symbols.sourcePsiSafe
import ksp.org.jetbrains.kotlin.analysis.api.types.KaClassType
import ksp.org.jetbrains.kotlin.analysis.api.types.KaErrorType
import ksp.org.jetbrains.kotlin.analysis.api.types.KaTypeMappingMode
import ksp.org.jetbrains.kotlin.asJava.classes.KotlinSuperTypeListBuilder
import ksp.org.jetbrains.kotlin.asJava.classes.cannotModify
import ksp.org.jetbrains.kotlin.asJava.classes.lazyPub
import ksp.org.jetbrains.kotlin.asJava.elements.KtLightAbstractAnnotation
import ksp.org.jetbrains.kotlin.asJava.elements.KtLightDeclaration
import ksp.org.jetbrains.kotlin.idea.KotlinLanguage
import ksp.org.jetbrains.kotlin.light.classes.symbol.*
import ksp.org.jetbrains.kotlin.light.classes.symbol.annotations.AnnotationsBox
import ksp.org.jetbrains.kotlin.light.classes.symbol.annotations.GranularAnnotationsBox
import ksp.org.jetbrains.kotlin.light.classes.symbol.annotations.SymbolAnnotationsProvider
import ksp.org.jetbrains.kotlin.name.StandardClassIds
import ksp.org.jetbrains.kotlin.psi.KtTypeParameter
import ksp.org.jetbrains.kotlin.psi.psiUtil.startOffset

internal class SymbolLightTypeParameter private constructor(
    private val parent: SymbolLightTypeParameterList,
    private val index: Int,
    private val typeParameterSymbolPointer: KaSymbolPointer<KaTypeParameterSymbol>,
    override val kotlinOrigin: KtTypeParameter?,
) : LightElement(parent.manager, KotlinLanguage.INSTANCE), PsiTypeParameter,
    KtLightDeclaration<KtTypeParameter, PsiTypeParameter> {

    constructor(
        ktAnalysisSession: KaSession,
        parent: SymbolLightTypeParameterList,
        index: Int,
        typeParameterSymbol: KaTypeParameterSymbol,
    ) : this(
        parent = parent,
        index = index,
        typeParameterSymbolPointer = with(ktAnalysisSession) { typeParameterSymbol.createPointer() },
        kotlinOrigin = typeParameterSymbol.sourcePsiSafe(),
    )

    private val ktModule: KaModule get() = parent.ktModule

    private inline fun <T> withTypeParameterSymbol(crossinline action: KaSession.(KaTypeParameterSymbol) -> T): T =
        typeParameterSymbolPointer.withSymbol(ktModule, action)

    override val givenAnnotations: List<KtLightAbstractAnnotation> get() = invalidAccess()

    override fun copy(): PsiElement = copyTo(parent)

    internal fun copyTo(parent: SymbolLightTypeParameterList): SymbolLightTypeParameter = SymbolLightTypeParameter(
        parent,
        index,
        typeParameterSymbolPointer,
        kotlinOrigin,
    )

    override fun accept(visitor: PsiElementVisitor) {
        if (visitor is JavaElementVisitor) {
            visitor.visitTypeParameter(this)
        } else {
            super<LightElement>.accept(visitor)
        }
    }

    private val _extendsList: PsiReferenceList by lazyPub {
        val listBuilder = KotlinSuperTypeListBuilder(
            this,
            kotlinOrigin = null,
            manager = manager,
            language = language,
            role = PsiReferenceList.Role.EXTENDS_LIST
        )

        withTypeParameterSymbol { typeParameterSymbol ->
            typeParameterSymbol.upperBounds
                .filter { type ->
                    when (type) {
                        is KaClassType -> type.classId != StandardClassIds.Any
                        is KaErrorType -> false
                        else -> true
                    }
                }
                .mapNotNull {
                    mapType(it, this@SymbolLightTypeParameter, KaTypeMappingMode.GENERIC_ARGUMENT)
                }
                .forEach { listBuilder.addReference(it) }
        }

        listBuilder
    }

    override fun getExtendsList(): PsiReferenceList = _extendsList
    override fun getExtendsListTypes(): Array<PsiClassType> = PsiClassImplUtil.getExtendsListTypes(this)

    //PsiClass simple implementation
    override fun getImplementsList(): PsiReferenceList? = null
    override fun getImplementsListTypes(): Array<PsiClassType> = PsiClassImplUtil.getImplementsListTypes(this)
    override fun getSuperClass(): PsiClass? = PsiClassImplUtil.getSuperClass(this)
    override fun getInterfaces(): Array<PsiClass> = PsiClassImplUtil.getInterfaces(this)
    override fun getSupers(): Array<PsiClass> = PsiClassImplUtil.getSupers(this)
    override fun getSuperTypes(): Array<PsiClassType> = PsiClassImplUtil.getSuperTypes(this)
    override fun getConstructors(): Array<PsiMethod> = PsiMethod.EMPTY_ARRAY
    override fun getInitializers(): Array<PsiClassInitializer> = PsiClassInitializer.EMPTY_ARRAY
    override fun getAllFields(): Array<PsiField> = PsiField.EMPTY_ARRAY
    override fun getAllMethods(): Array<PsiMethod> = PsiMethod.EMPTY_ARRAY
    override fun getAllInnerClasses(): Array<PsiClass> = PsiClass.EMPTY_ARRAY
    override fun findFieldByName(name: String?, checkBases: Boolean): PsiField? = null
    override fun findMethodBySignature(patternMethod: PsiMethod, checkBases: Boolean): PsiMethod? = null
    override fun findMethodsBySignature(patternMethod: PsiMethod, checkBases: Boolean): Array<PsiMethod> = PsiMethod.EMPTY_ARRAY
    override fun findMethodsAndTheirSubstitutorsByName(name: String, checkBases: Boolean): List<Pair<PsiMethod, PsiSubstitutor>> =
        emptyList()

    override fun getAllMethodsAndTheirSubstitutors(): List<Pair<PsiMethod, PsiSubstitutor>> = emptyList()
    override fun findInnerClassByName(name: String?, checkBases: Boolean): PsiClass? = null
    override fun getLBrace(): PsiElement? = null
    override fun getRBrace(): PsiElement? = null
    override fun getScope(): PsiElement = parent

    override fun isInheritor(baseClass: PsiClass, checkDeep: Boolean): Boolean {
        return InheritanceImplUtil.isInheritor(this, baseClass, checkDeep)
    }

    override fun isInheritorDeep(baseClass: PsiClass, classToByPass: PsiClass?): Boolean {
        return InheritanceImplUtil.isInheritorDeep(this, baseClass, classToByPass)
    }

    override fun getVisibleSignatures(): MutableCollection<HierarchicalMethodSignature> = mutableListOf()
    override fun setName(name: String): PsiElement = cannotModify()
    override fun getNameIdentifier(): PsiIdentifier? = null
    override fun getModifierList(): PsiModifierList? = null
    override fun hasModifierProperty(name: String): Boolean = false
    override fun getOwner(): PsiTypeParameterListOwner = parent.owner
    override fun getParent(): PsiElement = parent
    override fun getContainingClass(): PsiClass? = null
    override fun getDocComment(): PsiDocComment? = null
    override fun isDeprecated(): Boolean = false
    override fun getTypeParameters(): Array<PsiTypeParameter> = PsiTypeParameter.EMPTY_ARRAY
    override fun hasTypeParameters(): Boolean = false
    override fun getTypeParameterList(): PsiTypeParameterList? = null
    override fun getQualifiedName(): String? = null
    override fun getMethods(): Array<PsiMethod> = PsiMethod.EMPTY_ARRAY
    override fun findMethodsByName(name: String?, checkBases: Boolean): Array<PsiMethod> = PsiMethod.EMPTY_ARRAY
    override fun getFields(): Array<PsiField> = PsiField.EMPTY_ARRAY
    override fun getInnerClasses(): Array<PsiClass> = PsiClass.EMPTY_ARRAY
    override fun isInterface(): Boolean = false
    override fun isAnnotationType(): Boolean = false
    override fun isEnum(): Boolean = false
    override fun addAnnotation(qualifiedName: String): PsiAnnotation = cannotModify()
    //End of PsiClass simple implementation

    private val _name: String by lazyPub {
        withTypeParameterSymbol { it.name.asString() }
    }

    override fun getName(): String = _name

    override fun getIndex(): Int = index

    private val annotationsBox: AnnotationsBox = GranularAnnotationsBox(
        annotationsProvider = SymbolAnnotationsProvider(ktModule, typeParameterSymbolPointer)
    )

    override fun getAnnotations(): Array<PsiAnnotation> = annotationsBox.annotationsArray(this)
    override fun findAnnotation(qualifiedName: String): PsiAnnotation? = annotationsBox.findAnnotation(this, qualifiedName)
    override fun getAnnotation(fqn: String): PsiAnnotation? = findAnnotation(fqn)
    override fun hasAnnotation(fqn: String): Boolean = annotationsBox.hasAnnotation(this, fqn)
    override fun getApplicableAnnotations(): Array<PsiAnnotation> = annotations

    override fun toString(): String = "SymbolLightTypeParameter:$name"

    override fun getNavigationElement(): PsiElement = kotlinOrigin ?: parent.navigationElement
    override fun getLanguage(): Language = KotlinLanguage.INSTANCE

    override fun getUseScope(): SearchScope = kotlinOrigin?.useScope ?: parent.useScope

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is SymbolLightTypeParameter || other.ktModule != ktModule || other.index != index) return false
        if (kotlinOrigin != null || other.kotlinOrigin != null) {
            return other.kotlinOrigin == kotlinOrigin
        }

        return compareSymbolPointers(typeParameterSymbolPointer, other.typeParameterSymbolPointer) &&
                other.parent == parent
    }

    override fun hashCode(): Int = kotlinOrigin?.hashCode() ?: name.hashCode()
    override fun isEquivalentTo(another: PsiElement): Boolean {
        return basicIsEquivalentTo(this, another) || isOriginEquivalentTo(another)
    }

    override fun getText(): String? = kotlinOrigin?.text
    override fun getTextRange(): TextRange? = kotlinOrigin?.textRange
    override fun getContainingFile(): PsiFile = parent.containingFile
    override fun getTextOffset(): Int = kotlinOrigin?.startOffset ?: -1
    override fun getStartOffsetInParent(): Int = kotlinOrigin?.startOffsetInParent ?: -1

    override fun isValid(): Boolean = super.isValid() && kotlinOrigin?.isValid ?: typeParameterSymbolPointer.isValid(ktModule)
}
