/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.light.classes.symbol.parameters

import ksp.com.intellij.psi.PsiIdentifier
import ksp.com.intellij.psi.PsiModifierList
import ksp.com.intellij.psi.PsiType
import ksp.org.jetbrains.kotlin.analysis.api.KaSession
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaNamedFunctionSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaSymbolVisibility
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.types.KaTypeNullability
import ksp.org.jetbrains.kotlin.asJava.classes.lazyPub
import ksp.org.jetbrains.kotlin.codegen.coroutines.SUSPEND_FUNCTION_COMPLETION_PARAMETER_NAME
import ksp.org.jetbrains.kotlin.light.classes.symbol.annotations.EmptyAnnotationsProvider
import ksp.org.jetbrains.kotlin.light.classes.symbol.annotations.GranularAnnotationsBox
import ksp.org.jetbrains.kotlin.light.classes.symbol.annotations.NullabilityAnnotationsProvider
import ksp.org.jetbrains.kotlin.light.classes.symbol.isValid
import ksp.org.jetbrains.kotlin.light.classes.symbol.methods.SymbolLightMethodBase
import ksp.org.jetbrains.kotlin.light.classes.symbol.modifierLists.SymbolLightClassModifierList
import ksp.org.jetbrains.kotlin.light.classes.symbol.nonExistentType
import ksp.org.jetbrains.kotlin.light.classes.symbol.withSymbol
import ksp.org.jetbrains.kotlin.name.StandardClassIds
import ksp.org.jetbrains.kotlin.psi.KtParameter

internal class SymbolLightSuspendContinuationParameter(
    private val functionSymbolPointer: KaSymbolPointer<KaNamedFunctionSymbol>,
    private val containingMethod: SymbolLightMethodBase,
) : SymbolLightParameterBase(containingMethod) {
    private inline fun <T> withFunctionSymbol(crossinline action: KaSession.(KaNamedFunctionSymbol) -> T): T {
        return functionSymbolPointer.withSymbol(ktModule, action)
    }

    override fun getName(): String = SUSPEND_FUNCTION_COMPLETION_PARAMETER_NAME

    override fun getNameIdentifier(): PsiIdentifier? = null

    override fun getType(): PsiType = _type

    private val _type by lazyPub {
        withFunctionSymbol { functionSymbol ->
            val ktType = buildClassType(StandardClassIds.Continuation) { argument(functionSymbol.returnType) }
            ktType.asPsiType(
                this@SymbolLightSuspendContinuationParameter,
                allowErrorTypes = true,
                getTypeMappingMode(ktType),
                allowNonJvmPlatforms = true,
            ) ?: nonExistentType()
        }
    }

    override fun isVarArgs(): Boolean = false

    override fun getModifierList(): PsiModifierList = _modifierList

    private val _modifierList: PsiModifierList by lazyPub {
        SymbolLightClassModifierList(
            containingDeclaration = this,
            annotationsBox = GranularAnnotationsBox(
                annotationsProvider = EmptyAnnotationsProvider,
                additionalAnnotationsProvider = NullabilityAnnotationsProvider {
                    if (withFunctionSymbol { it.visibility == KaSymbolVisibility.PRIVATE })
                        KaTypeNullability.UNKNOWN
                    else
                        KaTypeNullability.NON_NULLABLE
                },
            ),
        )
    }

    override fun hasModifierProperty(p0: String): Boolean = false

    override val kotlinOrigin: KtParameter? = null

    override fun equals(other: Any?): Boolean = this === other ||
            other is SymbolLightSuspendContinuationParameter &&
            containingMethod == other.containingMethod

    override fun hashCode(): Int = name.hashCode() * 31 + containingMethod.hashCode()

    override fun isValid(): Boolean = super.isValid() && functionSymbolPointer.isValid(ktModule)
}
