/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.ir.backend.js.lower

import ksp.org.jetbrains.kotlin.backend.common.BodyLoweringPass
import ksp.org.jetbrains.kotlin.backend.common.CommonBackendContext
import ksp.org.jetbrains.kotlin.backend.common.ScopeWithIr
import ksp.org.jetbrains.kotlin.backend.common.compilationException
import ksp.org.jetbrains.kotlin.backend.common.lower.SingleAbstractMethodLowering
import ksp.org.jetbrains.kotlin.descriptors.DescriptorVisibilities
import ksp.org.jetbrains.kotlin.descriptors.DescriptorVisibility
import ksp.org.jetbrains.kotlin.ir.declarations.*
import ksp.org.jetbrains.kotlin.ir.expressions.IrBody
import ksp.org.jetbrains.kotlin.ir.expressions.IrTypeOperatorCall
import ksp.org.jetbrains.kotlin.ir.symbols.IrSymbol
import ksp.org.jetbrains.kotlin.ir.types.IrType
import ksp.org.jetbrains.kotlin.ir.types.classOrNull
import ksp.org.jetbrains.kotlin.ir.types.defaultType
import ksp.org.jetbrains.kotlin.ir.util.file
import ksp.org.jetbrains.kotlin.ir.util.parentClassOrNull

class JsSingleAbstractMethodLowering(context: CommonBackendContext) : SingleAbstractMethodLowering(context), BodyLoweringPass {

    override fun getWrapperVisibility(expression: IrTypeOperatorCall, scopes: List<ScopeWithIr>): DescriptorVisibility =
        DescriptorVisibilities.LOCAL

    override val IrType.needEqualsHashCodeMethods get() = true

    private var enclosingBodyContainer: IrDeclaration? = null

    override fun lower(irFile: IrFile) {
        super<BodyLoweringPass>.lower(irFile)
    }

    override fun lower(irBody: IrBody, container: IrDeclaration) {
        cachedImplementations.clear()
        inlineCachedImplementations.clear()
        enclosingContainer = container.parentClassOrNull ?: container.file
        enclosingBodyContainer = container

        irBody.transformChildrenVoid()

        for (wrapper in cachedImplementations.values + inlineCachedImplementations.values) {
            val parentClass = wrapper.parent as IrDeclarationContainer
            parentClass.declarations += wrapper
        }
    }

    override fun currentScopeSymbol(): IrSymbol? {
        return super.currentScopeSymbol() ?: (enclosingBodyContainer as? IrSymbolOwner)?.symbol
    }

    override fun getSuperTypeForWrapper(typeOperand: IrType): IrType {
        // FE doesn't allow type parameters for now.
        // And since there is a to-do in common SingleAbstractMethodLowering (at function visitTypeOperator),
        // we don't have to be more saint than a pope here.
        return typeOperand.classOrNull?.defaultType
            ?: compilationException("Unsupported SAM conversion", typeOperand)
    }
}
