/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.ir.backend.js.lower

import ksp.org.jetbrains.kotlin.backend.common.DeclarationTransformer
import ksp.org.jetbrains.kotlin.backend.common.bridges.FunctionHandle
import ksp.org.jetbrains.kotlin.backend.common.bridges.generateBridges
import ksp.org.jetbrains.kotlin.backend.common.compilationException
import ksp.org.jetbrains.kotlin.backend.common.lower.*
import ksp.org.jetbrains.kotlin.descriptors.Modality
import ksp.org.jetbrains.kotlin.ir.UNDEFINED_OFFSET
import ksp.org.jetbrains.kotlin.ir.backend.js.JsCommonBackendContext
import ksp.org.jetbrains.kotlin.ir.backend.js.utils.realOverrideTarget
import ksp.org.jetbrains.kotlin.ir.builders.*
import ksp.org.jetbrains.kotlin.ir.builders.declarations.buildFun
import ksp.org.jetbrains.kotlin.ir.declarations.*
import ksp.org.jetbrains.kotlin.ir.expressions.IrExpression
import ksp.org.jetbrains.kotlin.ir.symbols.IrTypeParameterSymbol
import ksp.org.jetbrains.kotlin.ir.types.IrType
import ksp.org.jetbrains.kotlin.ir.types.classifierOrNull
import ksp.org.jetbrains.kotlin.ir.util.*
import ksp.org.jetbrains.kotlin.utils.memoryOptimizedPlus
import ksp.org.jetbrains.kotlin.utils.toSmartList

/**
 * Constructs bridges for inherited generic functions
 *
 *  Example: for given class hierarchy
 *
 *          class C<T>  {
 *            fun foo(t: T) = ...
 *          }
 *
 *          class D : C<Int> {
 *            override fun foo(t: Int) = impl
 *          }
 *
 *  it adds method D that delegates generic calls to implementation:
 *
 *          class D : C<Int> {
 *            override fun foo(t: Int) = impl
 *            fun foo(t: Any?) = foo(t as Int)  // Constructed bridge
 *          }
 */
abstract class BridgesConstruction<T : JsCommonBackendContext>(val context: T) : DeclarationTransformer {

    private val specialBridgeMethods = SpecialBridgeMethods(context)

    abstract fun getFunctionSignature(function: IrSimpleFunction): Any

    /**
     * Usually just returns [irFunction]'s parameters, but special transformations may be required if,
     * for example, we're dealing with an external function, and that function contains a vararg,
     * which we must extract and convert to an array.
     */
    protected open fun extractParameters(
        blockBodyBuilder: IrBlockBodyBuilder,
        irFunction: IrSimpleFunction,
        bridge: IrSimpleFunction
    ): List<IrValueDeclaration> = irFunction.parameters

    // Should dispatch receiver type be casted inside a bridge.
    open val shouldCastDispatchReceiver: Boolean = false

    override fun transformFlat(declaration: IrDeclaration): List<IrDeclaration>? {
        if (declaration !is IrSimpleFunction || declaration.isStaticMethodOfClass || declaration.parent !is IrClass) return null

        return generateBridges(declaration)?.let { listOf(declaration) + it }
    }

    private fun generateBridges(function: IrSimpleFunction): List<IrDeclaration>? {
        val (specialOverride: IrSimpleFunction?, specialOverrideInfo) =
            specialBridgeMethods.findSpecialWithOverride(function) ?: Pair(null, null)

        val specialOverrideSignature = specialOverride?.let { FunctionAndSignature(it) }

        val bridgesToGenerate = generateBridges(
            function = IrBasedFunctionHandle(function),
            signature = { FunctionAndSignature(it.function) }
        )

        if (bridgesToGenerate.isEmpty()) return null

        val result = mutableListOf<IrDeclaration>()

        for ((from, to) in bridgesToGenerate) {
            if (!from.function.parentAsClass.isInterface &&
                from.function.isReal &&
                from.function.modality != Modality.ABSTRACT &&
                !to.function.isReal
            ) {
                continue
            }

            // Don't build bridges for functions with the same signature.
            // TODO: This should be caught earlier in bridgesToGenerate
            if (FunctionAndSignature(to.function.realOverrideTarget) == FunctionAndSignature(from.function.realOverrideTarget))
                continue

            if (from.function.correspondingPropertySymbol != null && from.function.isEffectivelyExternal()) {
                // TODO: Revisit bridges from external properties
                continue
            }

            val bridge: IrDeclaration = when {
                specialOverrideSignature == from ->
                    createBridge(function, from.function, to.function, specialOverrideInfo)

                else ->
                    createBridge(function, from.function, to.function, null)
            }


            result += bridge
        }

        return result
    }

    // Ported from from jvm.lower.BridgeLowering
    private fun createBridge(
        function: IrSimpleFunction,
        bridge: IrSimpleFunction,
        delegateTo: IrSimpleFunction,
        specialMethodInfo: SpecialMethodWithDefaultInfo?
    ): IrFunction {

        val origin = getBridgeOrigin(bridge)

        // TODO: Support offsets for debug info
        val irFunction = context.irFactory.buildFun {
            updateFrom(bridge)
            this.startOffset = UNDEFINED_OFFSET
            this.endOffset = UNDEFINED_OFFSET
            this.name = bridge.name
            this.returnType = bridge.returnType
            this.modality = Modality.FINAL
            this.isFakeOverride = false
            this.origin = origin
        }.apply {
            parent = function.parent
            copyTypeParametersFrom(bridge)
            val substitutionMap = makeTypeParameterSubstitutionMap(bridge, this)
            copyValueParametersFrom(bridge, substitutionMap)
            annotations = annotations memoryOptimizedPlus bridge.annotations
            // the js function signature building process (jsFunctionSignature()) uses dfs throught overriddenSymbols for getting js name,
            // therefore it is very important to put bridge symbol at the beginning, it allows to get correct js function name
            overriddenSymbols = mutableSetOf(bridge.symbol).also {
                it.addAll(overriddenSymbols)
                it.addAll(delegateTo.overriddenSymbols)
            }.toSmartList()
        }

        irFunction.body = context.irFactory.createBlockBody(UNDEFINED_OFFSET, UNDEFINED_OFFSET) {
            statements += context.createIrBuilder(irFunction.symbol).irBlockBody(irFunction) {
                val extractedParameters = extractParameters(this, irFunction, bridge)
                if (specialMethodInfo != null) {
                    extractedParameters
                        .map { it as? IrValueParameter ?: compilationException("Expected a value parameter", it) }
                        .filter { it.kind != IrParameterKind.DispatchReceiver }
                        .take(specialMethodInfo.argumentsToCheck)
                        .forEach { parameter ->
                            +irIfThen(
                                context.irBuiltIns.unitType,
                                irNot(irIs(irGet(parameter), delegateTo.parameters[parameter.indexInParameters].type)),
                                irReturn(specialMethodInfo.defaultValueGenerator(irFunction))
                            )
                        }
                }

                val call = irCall(delegateTo.symbol)
                for (delegateParam in delegateTo.parameters) {
                    if ((delegateTo.isSuspend xor irFunction.isSuspend) && delegateParam.indexInParameters == extractedParameters.lastIndex) {
                        break
                    }

                    var argument: IrExpression = irGet(extractedParameters[delegateParam.indexInParameters])
                    if (delegateParam.kind != IrParameterKind.DispatchReceiver || shouldCastDispatchReceiver) {
                        argument = irCastIfNeeded(argument, delegateParam.type)
                    }
                    call.arguments[delegateParam] = argument
                }

                +irReturn(call)
            }.statements
        }

        return irFunction
    }

    /**
     * Copies the value parameters from [bridge] to [this]. If [bridge] is external and contains a vararg parameter,
     * only copies the parameters before the vararg.
     * The rest parameters are expected to be obtained later using the `arguments` object in JS.
     */
    private fun IrSimpleFunction.copyValueParametersFrom(bridge: IrSimpleFunction, substitutionMap: Map<IrTypeParameterSymbol, IrType>) {
        var valueParametersToCopy = bridge.parameters
        if (bridge.isEffectivelyExternal()) {
            valueParametersToCopy = valueParametersToCopy.takeWhile { it.varargElementType == null }
        }
        parameters = parameters memoryOptimizedPlus valueParametersToCopy.map { p -> p.copyTo(this, type = p.type.substitute(substitutionMap)) }
    }

    abstract fun getBridgeOrigin(bridge: IrSimpleFunction): IrDeclarationOrigin

    // TODO: get rid of Unit check
    private fun IrBlockBodyBuilder.irCastIfNeeded(argument: IrExpression, type: IrType): IrExpression =
        if (argument.type.classifierOrNull == type.classifierOrNull) argument else irAs(argument, type)

    // Wrapper around function that compares and hashCodes it based on signature
    // Designed to be used as a Signature type parameter in backend.common.bridges
    inner class FunctionAndSignature(val function: IrSimpleFunction) {

        // TODO: Use type-upper-bound-based signature instead of Strings
        // Currently strings are used for compatibility with a hack-based name generator

        private val signature = getFunctionSignature(function)

        override fun equals(other: Any?): Boolean {
            if (this === other) return true
            if (other !is BridgesConstruction<*>.FunctionAndSignature) return false

            return signature == other.signature
        }

        override fun hashCode(): Int = signature.hashCode()
    }
}

// Handle for common.bridges
data class IrBasedFunctionHandle(val function: IrSimpleFunction) : FunctionHandle {
    override val isDeclaration = function.run { isReal || findInterfaceImplementation() != null }

    override val isAbstract: Boolean =
        function.modality == Modality.ABSTRACT

    override val mayBeUsedAsSuperImplementation =
        !function.parentAsClass.isInterface

    override fun getOverridden() =
        function.overriddenSymbols.map { IrBasedFunctionHandle(it.owner) }
}

private fun IrSimpleFunction.findInterfaceImplementation(): IrSimpleFunction? {
    if (isReal) return null

    return resolveFakeOverride()?.run { if (parentAsClass.isInterface) this else null }
}
