/*
 * Copyright 2010-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ksp.org.jetbrains.kotlin.incremental

import ksp.org.jetbrains.kotlin.descriptors.*
import ksp.org.jetbrains.kotlin.load.java.JavaDescriptorVisibilities
import ksp.org.jetbrains.kotlin.metadata.ProtoBuf
import ksp.org.jetbrains.kotlin.metadata.deserialization.BinaryVersion
import ksp.org.jetbrains.kotlin.metadata.java.JavaClassProtoBuf
import ksp.org.jetbrains.kotlin.metadata.deserialization.MetadataVersion
import ksp.org.jetbrains.kotlin.metadata.serialization.MutableVersionRequirementTable
import ksp.org.jetbrains.kotlin.resolve.DescriptorUtils
import ksp.org.jetbrains.kotlin.resolve.scopes.MemberScope
import ksp.org.jetbrains.kotlin.serialization.DescriptorSerializer
import ksp.org.jetbrains.kotlin.serialization.KotlinSerializerExtensionBase
import ksp.org.jetbrains.kotlin.serialization.deserialization.builtins.BuiltInSerializerProtocol

// It uses BuiltInSerializerProtocol for annotations serialization
class JavaClassesSerializerExtension : KotlinSerializerExtensionBase(BuiltInSerializerProtocol) {
    override val metadataVersion: BinaryVersion
        get() = MetadataVersion.INVALID_VERSION

    override fun serializeClass(
            descriptor: ClassDescriptor,
            proto: ProtoBuf.Class.Builder,
            versionRequirementTable: MutableVersionRequirementTable,
            childSerializer: DescriptorSerializer
    ) {
        super.serializeClass(descriptor, proto, versionRequirementTable, childSerializer)
        if (descriptor.visibility == JavaDescriptorVisibilities.PACKAGE_VISIBILITY) {
            proto.setExtension(JavaClassProtoBuf.isPackagePrivateClass, true)
        }
    }

    override fun serializeConstructor(descriptor: ConstructorDescriptor,
                                      proto: ProtoBuf.Constructor.Builder,
                                      childSerializer: DescriptorSerializer) {
        super.serializeConstructor(descriptor, proto, childSerializer)
        if (descriptor.visibility == JavaDescriptorVisibilities.PACKAGE_VISIBILITY) {
            proto.setExtension(JavaClassProtoBuf.isPackagePrivateConstructor, true)
        }
    }

    override fun serializeFunction(
        descriptor: FunctionDescriptor,
        proto: ProtoBuf.Function.Builder,
        versionRequirementTable: MutableVersionRequirementTable?,
        childSerializer: DescriptorSerializer
    ) {
        super.serializeFunction(descriptor, proto, versionRequirementTable, childSerializer)
        if (descriptor.visibility == JavaDescriptorVisibilities.PACKAGE_VISIBILITY) {
            proto.setExtension(JavaClassProtoBuf.isPackagePrivateMethod, true)
        }

        if (descriptor.dispatchReceiverParameter == null) {
            proto.setExtension(JavaClassProtoBuf.isStaticMethod, true)
        }
    }

    override fun serializeProperty(
            descriptor: PropertyDescriptor,
            proto: ProtoBuf.Property.Builder,
            versionRequirementTable: MutableVersionRequirementTable?,
            childSerializer: DescriptorSerializer
    ) {
        super.serializeProperty(descriptor, proto, versionRequirementTable, childSerializer)
        if (descriptor.visibility == JavaDescriptorVisibilities.PACKAGE_VISIBILITY) {
            proto.setExtension(JavaClassProtoBuf.isPackagePrivateField, true)
        }

        if (descriptor.dispatchReceiverParameter == null) {
            proto.setExtension(JavaClassProtoBuf.isStaticField, true)
        }
    }

    override fun shouldUseNormalizedVisibility() = true

    override val customClassMembersProducer =
            object : ClassMembersProducer {
                override fun getCallableMembers(classDescriptor: ClassDescriptor) =
                        arrayListOf<CallableMemberDescriptor>().apply {
                            addAll(classDescriptor.unsubstitutedMemberScope.getSortedCallableDescriptors())
                            addAll(classDescriptor.staticScope.getSortedCallableDescriptors())
                        }
            }

    private fun MemberScope.getSortedCallableDescriptors(): Collection<CallableMemberDescriptor> =
        DescriptorUtils.getAllDescriptors(this)
            .filterIsInstance<CallableMemberDescriptor>()
            .filter { it.kind != CallableMemberDescriptor.Kind.FAKE_OVERRIDE }
            .let { DescriptorSerializer.sort(it) }
}
