/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.syntax

import ksp.org.jetbrains.kotlin.*
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.diagnostics.typeParametersList
import ksp.org.jetbrains.kotlin.fir.declarations.FirProperty
import ksp.org.jetbrains.kotlin.psi.KtExpression
import ksp.org.jetbrains.kotlin.psi.KtProperty

// KtExpression is the appropriate PsiElement type; local variables are used in increments/decrements of dot-qualified expressions.
object FirLocalVariableTypeParametersSyntaxChecker : FirDeclarationSyntaxChecker<FirProperty, KtExpression>() {
    override fun isApplicable(element: FirProperty, source: KtSourceElement): Boolean =
        source.kind !is KtFakeSourceElementKind && element.isLocal

    override fun checkPsi(
        element: FirProperty,
        source: KtPsiSourceElement,
        psi: KtExpression,
        context: CheckerContext,
        reporter: DiagnosticReporter
    ) {
        if (psi is KtProperty && psi.typeParameterList != null) {
            reporter.reportOn(source, FirErrors.LOCAL_VARIABLE_WITH_TYPE_PARAMETERS, context)
        }
    }

    override fun checkLightTree(
        element: FirProperty,
        source: KtLightSourceElement,
        context: CheckerContext,
        reporter: DiagnosticReporter
    ) {
        val node = source.lighterASTNode
        if (node.tokenType != KtNodeTypes.PROPERTY) return
        source.treeStructure.typeParametersList(source.lighterASTNode)?.let { _ ->
            reporter.reportOn(source, FirErrors.LOCAL_VARIABLE_WITH_TYPE_PARAMETERS, context)
        }
    }
}
