/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.getModifierList
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.expressions.FirQualifiedAccessExpression
import ksp.org.jetbrains.kotlin.fir.types.FirPlaceholderProjection
import ksp.org.jetbrains.kotlin.fir.types.FirStarProjection
import ksp.org.jetbrains.kotlin.fir.types.FirTypeProjectionWithVariance
import ksp.org.jetbrains.kotlin.fir.utils.exceptions.withFirEntry
import ksp.org.jetbrains.kotlin.types.Variance
import ksp.org.jetbrains.kotlin.utils.exceptions.errorWithAttachment

object FirProjectionsOnNonClassTypeArgumentChecker : FirQualifiedAccessExpressionChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirQualifiedAccessExpression) {
        for (it in expression.typeArguments) {
            when (it) {
                is FirStarProjection -> reporter.reportOn(it.source, FirErrors.PROJECTION_ON_NON_CLASS_TYPE_ARGUMENT)
                is FirTypeProjectionWithVariance -> {
                    if (it.variance != Variance.INVARIANT) {
                        val modifierSource = it.source.getModifierList()?.modifiers?.firstOrNull()?.source
                        reporter.reportOn(modifierSource ?: it.source, FirErrors.PROJECTION_ON_NON_CLASS_TYPE_ARGUMENT)
                    }
                }
                is FirPlaceholderProjection -> errorWithAttachment("Placeholder projection shouldn't exist during checker phase.") {
                    withFirEntry("expression", expression)
                }
            }
        }
    }
}
